with Ada.Text_IO;
with Ada.Exceptions;
with Class_Test_Result_File;
with Interpolation_Table.Singly_Indexed;

use type Interpolation_Table.Singly_Indexed.Instance;

------------------------------------------------------------------------------------------
-- Test driver for the singly-indexed interpolation table.
------------------------------------------------------------------------------------------
procedure Interpolation_Table.Singly_Indexed_CTD_3 is

   Table_File : Ada.Text_IO.File_Type;

   Table      : aliased Interpolation_Table.Singly_Indexed.Instance;
   Same_Table : aliased Interpolation_Table.Singly_Indexed.Instance;

   Table_Name      : constant String := "Temp_Table";

   type Test_Table is array (1..10) of Float;

   Table_Data : constant Test_Table :=
     (0.0, 1.0, 2.0, 4.0, 8.0, 16.0, 32.0, 64.0, 128.0, 256.0);


begin

   -----------------
   -- Common setup
   -----------------

   -- Create a text file matching Table_Data
   Ada.Text_IO.Put_Line ("Creating ASCII table file " & Table_Name & ".txt");
   Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

   Ada.Text_IO.Put_Line (File => Table_File, Item =>
                           "# Temporary single-valued interpolation table created by");
   Ada.Text_IO.Put_Line (File => Table_File, Item =>
                           "# interpolation_table.singly_indexed_ctd_6.adb");

   for Value in Table_Data'range loop

      Ada.Text_IO.Put_Line
        (File => Table_File,
         Item => Float'Image(Float(Value)) & ", " & Float'Image(Table_Data(Value)));
   end loop;

   Ada.Text_IO.Close(Table_File);


   -- Call Read_ASCII to read the file in
   Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & ".txt");

   Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
                                                  Table => Table);


   ----------------------------------------------------------------------------
   -- Test Case 9
   --
   -- Inputs           : A table with varring slopes, and a point between two of
   --                    the independent points.
   -- Expected Results : A value linerarly between the two dependent points.
   --
   -- Purpose          : Verify that Interpolate returns the correct dependent
   --                    point when given values not matching an independent.
   Test_Case_9 : declare
      Good : Boolean;
   begin
       -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Interpolating inside table ..."); Ada.Text_IO.Flush;

      for Value in Table_Data'First .. Test_Table'Last - 1 loop

         Good := Interpolation_Table.Singly_Indexed.Interpolate
           (Input => Float(Value) + 0.5,
            Table => Table'access) =
           Table_Data(Value) + ((Table_Data(Value + 1) - Table_Data(Value)) / 2.0);

         exit when not Good;
      end loop;

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Success => Good, Number => 9);

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (Success => False, Number => 9);
         if Ada.Text_IO.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_9;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 10
   --
   -- Inputs           : A table with varring slopes, and a point lower than
   --                    the independent points.
   -- Expected Results : A value linerarly lower than the smallest dependent
   --                    based on the slope between the smallest two.
   --
   -- Purpose          : Verify that Interpolate returns the correct dependent
   --                    point when given a values lower than any dependent.
   Test_Case_10 : declare
      Good : Boolean;
   begin
       -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Interpolating under table ..."); Ada.Text_IO.Flush;

      Good := Interpolation_Table.Singly_Indexed.Interpolate
        (Input => 0.5,
         Table => Table'access) = Table_Data(1);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Good);

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_IO.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_10;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 11
   --
   -- Inputs           : A table with varring slopes, and a point between two of
   --                    the independent points (twice).
   -- Expected Results : The same value both times.
   --
   -- Purpose          : Verify the output of interpolate when the internal
   --                    "cacheing" is used.
   Test_Case_11 : declare
      Good : Boolean;
   begin
       -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Comparing duplicate interpolations ..."); Ada.Text_IO.Flush;

      for Value in Table_Data'First .. Test_Table'Last - 1 loop

         Good := Interpolation_Table.Singly_Indexed.Interpolate
           (Input => Float(Value) + 0.5,
            Table => Table'access) = Interpolation_Table.Singly_Indexed.Interpolate
           (Input => Float(Value) + 0.5,
            Table => Table'access);

         exit when not Good;
      end loop;

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Success => Good);

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (Success => False);
         if Ada.Text_IO.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_11;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 12
   --
   -- Inputs           : A table with more than one value for its independent
   -- Expected Results : The minimum value for the independent should be
   --                    returned.
   --
   -- Purpose          : Verify the output of First_Independent.
   Test_Case_12 : declare
      Good : Boolean;
   begin
       -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifying First_Independent ..."); Ada.Text_IO.Flush;

      Good := Interpolation_Table.Singly_Indexed.First_Independent(Table)
        = Float(Table_Data'First);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Success => Good);

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (Success => False);
         if Ada.Text_IO.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_12;
   ----------------------------------------------------------------------------
   ----------------------------------------------------------------------------
   -- Test Case 13
   --
   -- Inputs           : A table with more than one value for its independent
   -- Expected Results : The maximum value for the independent should be
   --                    returned.
   --
   -- Purpose          : Verify the output of First_Independent.
   Test_Case_13 : declare
      Good : Boolean;
   begin
       -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifying Last_Independent ..."); Ada.Text_IO.Flush;

      Good := Interpolation_Table.Singly_Indexed.Last_Independent(Table)
        = Float(Table_Data'Last);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Success => Good);

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (Success => False);
         if Ada.Text_IO.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_13;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 14
   --
   -- Inputs           : A valid object file
   -- Expected Results : An "interpolation" of each independent value should
   --                    match the original depenent values.
   -- Purpose          : Verify that a table object file is handled properly
   Test_Case_14 : declare

      Good  : Boolean := True;
   begin
      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifing table object file support..."); Ada.Text_IO.Flush;

      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
           Float'Image(Float(Value)) & "," & Float'Image(Table_Data(Value)));

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      Interpolation_Table.Singly_Indexed.Write (File_Name => Table_Name & ".ito",
        Table => (1 => Table));

      Interpolation_Table.Read (File_Name => Table_Name & ".ito",
        Table => Table);

      for Value in Test_Table'Range loop

         Good := Interpolation_Table.Singly_Indexed.Interpolate
           (Input => Float(Value),
            Table => Table'access) = Table_Data(Value);

         exit when not Good;
      end loop;

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_14;
   ----------------------------------------------------------------------------

exception
when Event : others =>
   Ada.Text_IO.Put_Line ("unhandled exception:");
   Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
end Interpolation_Table.Singly_Indexed_CTD_3;
