with Ada.Text_IO;
with Ada.Exceptions;
with Class_Test_Result_File;
with Interpolation_Table.Singly_Indexed;

use type Interpolation_Table.Singly_Indexed.Instance;

------------------------------------------------------------------------------------------
-- Test driver for the singly-indexed interpolation table.
------------------------------------------------------------------------------------------
procedure Interpolation_Table.Singly_Indexed_CTD_2 is

   Table_File : Ada.Text_IO.File_Type;

   Table      : aliased Interpolation_Table.Singly_Indexed.Instance;
   Same_Table : aliased Interpolation_Table.Singly_Indexed.Instance;

   Table_Name      : constant String := "Temp_Table";

   type Test_Table is array (1..10) of Float;

   Table_Data : constant Test_Table :=
     (0.0, 10.0, 20.0, 30.0, 40.0, 50.0, 60.0, 70.0, 80.0, 90.01010101011111);


begin

   ----------------------------------------------------------------------------
   -- Test Case 8
   --
   -- Inputs           : A table
   -- Expected Results : An ascii table file should be produced which, when read
   --                    back in, matches the input table.
   -- Purpose          : Verify that Write_ASCII writes a table in a form that
   --                    Read_ASCII can correctly read back in.
   Test_Case_8 : declare
      Good : Boolean;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_6.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
           Float'Image(Float(Value)) & ", " & Float'Image(Table_Data(Value)));

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & ".txt");

      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Use Write_ASCII to write it back out
      Ada.Text_IO.Put_Line ("Writing table to " & Table_Name & "2.txt");

      Interpolation_Table.Singly_Indexed.Write_ASCII (File_Name => Table_Name & "2.txt",
        Table => Table);

      -- Call Read_ASCII to read it back into another table
      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & "2.txt",
        Table => Same_Table);

      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Checking equality of table values..."); Ada.Text_IO.Flush;

      for Value in Test_Table'Range loop

         Good := Interpolation_Table.Singly_Indexed.Interpolate
           (Input => Float(Value),
            Table => Table'access) =
           Interpolation_Table.Singly_Indexed.Interpolate
           (Input => Float(Value),
            Table => Same_Table'access);

         exit when not Good;
      end loop;

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Success => Good, Number => 7);

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (Success => False, Number => 7);
         if Ada.Text_IO.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_8;
   ----------------------------------------------------------------------------

exception
when Event : others =>
   Ada.Text_IO.Put_Line ("unhandled exception:");
   Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
end Interpolation_Table.Singly_Indexed_CTD_2;


