with Ada.Text_IO;
with Ada.Exceptions;
with Class_Test_Result_File;
with Interpolation_Table.Singly_Indexed;

------------------------------------------------------------------------------------------
-- Test driver for the singly-indexed interpolation table.
------------------------------------------------------------------------------------------
procedure Interpolation_Table.Singly_Indexed_CTD_1 is

   Table_File : Ada.Text_IO.File_Type;

   Table      : aliased Interpolation_Table.Singly_Indexed.Instance;
   Same_Table : aliased Interpolation_Table.Singly_Indexed.Instance;

   Table_Name      : constant String := "Temp_Table";

   type Test_Table is array (1..10) of Float;

   Table_Data : constant Test_Table :=
     (0.0, 10.0, 20.0, 30.0, 40.0, 50.0, 60.0, 70.0, 80.0, 90.0);


begin

   ----------------------------------------------------------------------------
   -- Test Case 1
   --
   -- Inputs           : A valid file
   -- Expected Results : An "interpolation" of each independent value should
   --                    match the original depenent values.
   -- Purpose          : Verify that a correct table file is handled properly
   Test_Case_1 : declare

      Good  : Boolean := True;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
           Float'Image(Float(Value)) & "," & Float'Image(Table_Data(Value)));

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & ".txt");

      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifing table values..."); Ada.Text_IO.Flush;

      for Value in Test_Table'Range loop

         Good := Interpolation_Table.Singly_Indexed.Interpolate
           (Input => Float(Value),
            Table => Table'access) = Table_Data(Value);

         exit when not Good;
      end loop;

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_1;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 2
   --
   -- Inputs           : A file w/o a leading comment
   -- Expected Results : A syntax error.
   -- Purpose          : Verify that a leading comment is required for a table
   --                    file.
   Test_Case_2 : declare
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
           Float'Image(Float(Value)) & "," & Float'Image(Table_Data(Value)));

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & ".txt");

      Ada.Text_IO.Open (File => Table_File, Mode => Ada.Text_IO.In_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put ("Verifing leading comment required..."); Ada.Text_IO.Flush;

      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      Class_Test_Result_File.Report_Case_Status (False);
      Ada.Text_IO.Put_Line ("failure!");

   exception
   when others =>
      Class_Test_Result_File.Report_Case_Status (True);
      Ada.Text_IO.Put_Line ("success.");

   end Test_Case_2;

   Ada.Text_IO.Close (Table_File);

   ----------------------------------------------------------------------------
   -- Test Case 3
   --
   -- Inputs           : A file without a float value in the first value set
   -- Expected Results : A syntax error.
   -- Purpose          : Verify that a set of independent/dependent values
   --                    requires the first value to be a float.
   Test_Case_3 : declare

   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & "2.txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
           Integer'Image(Value) & "," & Float'Image(Table_Data(Value)));

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & "2.txt");

      Ada.Text_IO.Open (File => Table_File, Mode => Ada.Text_IO.In_File, Name => Table_Name & "2.txt");

      Ada.Text_IO.Put ("Verifing float independent required..."); Ada.Text_IO.Flush;

      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & "2.txt",
        Table => Table);

      Class_Test_Result_File.Report_Case_Status (False);
      Ada.Text_IO.Put_Line ("failure!");

      Ada.Text_IO.Close (Table_File);

   exception
   when others =>
      Class_Test_Result_File.Report_Case_Status (True);
      Ada.Text_IO.Put_Line ("success.");
      Ada.Text_IO.Close (Table_File);

   end Test_Case_3;

   ----------------------------------------------------------------------------
   -- Test Case 4
   --
   -- Inputs           : A file without a , in the first value set
   -- Expected Results : A syntax error.
   -- Purpose          : Verify that a set of independent/dependent values
   --                    requires ',' separator.
   Test_Case_4 : declare
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & "3.txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
           Float'Image(Float(Value)) & " " & Float'Image(Table_Data(Value)));

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & "3.txt");

      Ada.Text_IO.Open (File => Table_File, Mode => Ada.Text_IO.In_File, Name => Table_Name & "3.txt");

      Ada.Text_IO.Put ("Verifing ',' required..."); Ada.Text_IO.Flush;

      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & "3.txt",
        Table => Table);

      Class_Test_Result_File.Report_Case_Status (False);
      Ada.Text_IO.Put_Line ("failure!");

      Ada.Text_IO.Close (Table_File);

   exception
   when others =>
      Class_Test_Result_File.Report_Case_Status (True);
      Ada.Text_IO.Put_Line ("success.");
      Ada.Text_IO.Close (Table_File);

   end Test_Case_4;

   ----------------------------------------------------------------------------
   -- Test Case 5
   --
   -- Inputs           : A file without a float value for its dependents
   -- Expected Results : A syntax error.
   -- Purpose          : Verify that a set of independent/dependent values
   --                    requires the dependent value to be a float
   Test_Case_5 : declare
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & "4.txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
         Float'Image(Float(Value)) & "," & Integer'Image(Integer(Table_Data(Value))));

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & "4.txt");

      Ada.Text_IO.Open (File => Table_File, Mode => Ada.Text_IO.In_File, Name => Table_Name & "4.txt");

      Ada.Text_IO.Put ("Verifing float dependent required..."); Ada.Text_IO.Flush;

      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & "4.txt",
        Table => Table);

      Class_Test_Result_File.Report_Case_Status (False);
      Ada.Text_IO.Put_Line ("failure!");

      Ada.Text_IO.Close (Table_File);

   exception
   when others =>
      Class_Test_Result_File.Report_Case_Status (True);
      Ada.Text_IO.Put_Line ("success.");
      Ada.Text_IO.Close (Table_File);

   end Test_Case_5;


   ----------------------------------------------------------------------------
   -- Test Case 6
   --
   -- Inputs           : A file with *every* value separated by commas
   -- Expected Results : A syntax error.
   -- Purpose          : Verify that sets of independent/dependent values
   --                    require a blank separator.
   Test_Case_6 : declare
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & "6.txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
           Float'Image(Float(Value)) & "," & Float'Image(Table_Data(Value)) & ",");

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & "6.txt");

      Ada.Text_IO.Put ("Verifing ' ' required..."); Ada.Text_IO.Flush;

      begin
         Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & "6.txt",
           Table => Table);

         Class_Test_Result_File.Report_Case_Status (False);
         Ada.Text_IO.Put_Line ("failure!");


      exception
      when others =>
         Class_Test_Result_File.Report_Case_Status (True);
         Ada.Text_IO.Put_Line ("success.");
      end;

   end Test_Case_6;

   ----------------------------------------------------------------------------
   -- Test Case 7
   --
   -- Inputs           : A file with more than 2 entries per row
   -- Expected Results : Two Tables, which both interpolate correctly
   -- Purpose          : Verify that multiple valid tables are correctly read.
Test_Case_7 : declare

      type Test_Table_List is array (1..2) of Test_Table;
      Table_Data : constant Test_Table_List :=
        ((0.0, 10.0, 20.0, 30.0, 40.0, 50.0, 60.0, 70.0, 80.0, 90.0),
         (0.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0));

      Good  : Boolean := True;

      Table : aliased Interpolation_Table.Singly_Indexed.Table_List(1..2);

   begin

      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File,
                          Name => Table_Name & "7.txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for Value in Test_Table'Range loop

         Ada.Text_IO.Put_Line (File => Table_File, Item =>
            Float'Image(Float(Value)) & "," & Float'Image(Table_Data(1)(Value)) & "," &
            Float'Image(Table_Data(2)(Value)));

      end loop;

      Ada.Text_IO.Close(Table_File);

      -- Call Read_ASCII to read the file in
      Ada.Text_IO.Put_Line ("Reading table from " & Table_Name & "7.txt");

      Interpolation_Table.Singly_Indexed.Read_ASCII (File_Name => Table_Name & "7.txt",
        Table => Table);

      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifing table values..."); Ada.Text_IO.Flush;

      Verification : for Table_Index in Table'Range loop
         for Value in Test_Table'Range loop

            Good := Interpolation_Table.Singly_Indexed.Interpolate
              (Input => Float(Value),
               Table => Table(Table_Index)'access) = Table_Data(Table_Index)(Value);

            exit Verification when not Good;
         end loop;
      end loop Verification;

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_7;


exception
when Event : others =>
   Ada.Text_IO.Put_Line ("unhandled exception:");
   Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
end Interpolation_Table.Singly_Indexed_CTD_1;


