with Ada.Text_IO;
with Ada.Exceptions;
with Class_Test_Result_File;
with Interpolation_Table.Doubly_Indexed;

------------------------------------------------------------------------------------------
-- Test driver for the doubly-indexed interpolation table.
------------------------------------------------------------------------------------------
procedure Interpolation_Table.Doubly_Indexed_CTD_1 is

   Table_File : Ada.Text_IO.File_Type;

   Table      : aliased Interpolation_Table.Doubly_Indexed.Instance;
   Same_Table : aliased Interpolation_Table.Doubly_Indexed.Instance;

   Table_Name      : constant String := "Temp_Table";

   type Test_Vector is array (1..4) of Float;
   type Test_Table is array (1..4) of Test_Vector;

   X_Index : constant Test_Vector := (1.0, 2.0, 3.0, 4.0);
   Y_Index : constant Test_Vector := (1.0, 2.0, 4.0, 8.0);

   Table_Data : constant Test_Table :=
--      1.0    2.0    4.0    8.0
------------------------------------------
     ((  0.0, 10.0,  20.0,  30.0),  -- 1.0
      ( 20.0, 30.0,  50.0,  70.0),  -- 2.0
      ( 30.0, 50.0,  70.0, 100.0),  -- 3.0
      ( 50.0, 70.0, 100.0, 120.0)); -- 4.0

begin

   ----------------------------------------------------------------------------
   -- Test Case 1
   --
   -- Inputs           : A valid file
   -- Expected Results : An "interpolation" of each independent value should
   --                    match the original depenent values.
   -- Purpose          : Verify that a correct table file is handled properly
   Test_Case_1 : declare

      Good  : Boolean := True;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.doubly_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File, Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifing table values..."); Ada.Text_IO.Flush;

      Verification:
      for X_Value in Test_Table'Range loop

         for Y_Value in Test_Table'Range loop

            Good := Interpolation_Table.Doubly_Indexed.Interpolate
              (X     => X_Index(X_Value),
               Y     => Y_Index(Y_Value),
               Table => Table'access) = Table_Data(X_Value)(Y_Value);

            exit Verification when not Good;
         end loop;
      end loop Verification;

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
         Interpolation_Table.Doubly_Indexed.Write_ASCII(Table_Name & "2.txt", Table);
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_1;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 2
   --
   -- Inputs           : A file with more than 2 entries per row
   -- Expected Results : Two Tables, which both interpolate correctly
   -- Purpose          : Verify that multiple valid tables are correctly read.
   Test_Case_2 : declare

      type Test_Table_List is array (1..2) of Test_Table;
      Table_Data : constant Test_Table_List :=
--          1.0   2.0    4.0    8.0
----------------------------------------------
        (((  0.0, 10.0,  20.0,  30.0),   -- 1.0
          ( 20.0, 30.0,  50.0,  70.0),   -- 2.0
          ( 30.0, 50.0,  70.0, 100.0),   -- 3.0
          ( 50.0, 70.0, 100.0, 120.0)),  -- 4.0
         ((  0.0, 20.0,  40.0,  60.0),   -- 1.0
          ( 20.0, 40.0,  60.0,  80.0),   -- 2.0
          ( 40.0, 60.0,  80.0, 100.0),   -- 3.0
          ( 60.0, 80.0, 100.0, 120.0))); -- 4.0

      Good  : Boolean := True;

      Table : aliased Interpolation_Table.Doubly_Indexed.Table_List(1..2);

   begin

      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File,
                          Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.singly_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File,
                                  Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(1)(X_Value)(Y_Value)) & ", " &
              Float'Image(Table_Data(2)(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);

      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifing multi-dependent table values..."); Ada.Text_IO.Flush;

      Verification_2 : for Table_Index in Table'Range loop
         for X_Value in Test_Vector'Range loop

            for Y_Value in Test_Vector'Range loop

               Good := Interpolation_Table.Doubly_Indexed.Interpolate
                 (X     => Float(X_Value),
                  Y     => Y_Index(Y_Value),
                  Table => Table(Table_Index)'access) = Table_Data(Table_Index)(X_Value)(Y_Value);

               if not Good then
                  Ada.Text_Io.Put_Line ("Invalid result for value at location" & Integer'Image(X_Value) &
                                        "," & Float'Image(Y_Index(Y_Value)) & " in table" &
                                        Integer'Image(Table_Index));
               end if;

               exit Verification_2 when not Good;
            end loop;
         end loop;
      end loop Verification_2;

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_2;

   ----------------------------------------------------------------------------
   -- Test Case 3
   --
   -- Inputs           : A valid file with non-planar dependent values.
   --                    Interior point coordinates and precalculated dependent
   --                    results for those interior points.
   -- Expected Results : An interpolation interior points should match the
   --                    precalculated dependent values.
   -- Purpose          : Verify that the interpolation algorthim produces the
   --                    same results as were produced using the multilinear
   --                    algorithm that is used by Raytheon.
   Test_Case_3 : declare

      Good  : Boolean := True;

      subtype Dual_Index_Vector is Interpolation_Table.Index_Vector(1..2);

      type Index_List is array (Positive range <>) of Dual_Index_Vector;

      Indices : constant Index_List := ((1.25, 1.75), (2.5, 6.0));
      Results : constant Interpolation_Table.Index_Vector :=
        (12.5, 72.5);
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.multiply_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File, Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifing complicated no-planar table value..."); Ada.Text_IO.Flush;

      for Result in Results'range loop

         Good := Interpolation_Table.Doubly_Indexed.Interpolate
           (Indices(Result)(1),
            Indices(Result)(2),
            Table => Table'access) = Results(Result);

         if not Good then
            Ada.Text_Io.Put_Line ("Unmatched value for result" & Integer'Image(Result));
            Ada.Text_Io.Put_Line ("Expected" & Float'Image (Results(Result)));
         end if;
         exit when not Good;

      end loop;
      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
         Interpolation_Table.Doubly_Indexed.Write_ASCII(Table_Name & "2.txt", Table);
      end if;


    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
   end Test_Case_3;

   ----------------------------------------------------------------------------
   -- Test Case 4
   --
   -- Inputs           : A valid file
   -- Expected Results : The First_X_Independent result should match the value
   --                    of the first X independent in the file.
   -- Purpose          : Verify that a First_X_Indpendent returns the value of
   --                    the first X independent in the table.
   Test_Case_4 : declare

      Good  : Boolean := True;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.doubly_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File, Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the routine
      Ada.Text_IO.Put ("Verifing First_X_Independent..."); Ada.Text_IO.Flush;

      Good := Interpolation_Table.Doubly_Indexed.First_X_Independent (Table) =
        Float(X_Index (Test_Vector'First));

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_4;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 5
   --
   -- Inputs           : A valid file
   -- Expected Results : The First_Y_Independent result should match the value
   --                    of the first Y independent in the file.
   -- Purpose          : Verify that a First_Y_Indpendent returns the value of
   --                    the first Y independent in the table.
   Test_Case_5 : declare

      Good  : Boolean := True;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.doubly_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File, Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the routine
      Ada.Text_IO.Put ("Verifing First_Y_Independent..."); Ada.Text_IO.Flush;

      Good := Interpolation_Table.Doubly_Indexed.First_Y_Independent (Table) =
        Float(Y_Index (Test_Vector'First));

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_5;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 6
   --
   -- Inputs           : A valid file
   -- Expected Results : The Last_X_Independent result should match the value
   --                    of the last X independent in the file.
   -- Purpose          : Verify that a Last_X_Indpendent returns the value of
   --                    the last x independent in the table.
   Test_Case_6 : declare

      Good  : Boolean := True;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.doubly_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File, Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the routine
      Ada.Text_IO.Put ("Verifing Last_X_Independent..."); Ada.Text_IO.Flush;

      -- Verify the routine
      Good := Interpolation_Table.Doubly_Indexed.Last_X_Independent (Table) =
        Float(X_Index (Test_Vector'Last));

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_6;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 7
   --
   -- Inputs           : A valid file
   -- Expected Results : The Last_Y_Independent result should match the value
   --                    of the first independent in the file.
   -- Purpose          : Verify that a Last_Y_Indpendent returns the value of
   --                    the first independent in the table.
   Test_Case_7 : declare

      Good  : Boolean := True;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary single-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.doubly_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File, Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Verify the routine
      Ada.Text_IO.Put ("Verifing Last_Y_Independent..."); Ada.Text_IO.Flush;

      Good := Interpolation_Table.Doubly_Indexed.Last_Y_Independent (Table) =
        Float(Y_Index (Test_Vector'Last));

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_7;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 8
   --
   -- Inputs           : A valid file
   -- Expected Results : An "interpolation" of each independent value should
   --                    match the original depenent values.
   -- Purpose          : Verify that a correct table object file is handled
   --                    properly
   Test_Case_8 : declare

      Good  : Boolean := True;
   begin
      -- Create a text file matching Table_Data
      Ada.Text_IO.Create (File => Table_File, Mode => Ada.Text_IO.Out_File, Name => Table_Name & ".txt");

      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# Temporary doubly-valued interpolation table created by");
      Ada.Text_IO.Put_Line (File => Table_File, Item =>
        "# interpolation_table.doubly_indexed_ctd_1.adb");

      for X_Value in Test_Vector'Range loop

         for Y_Value in Test_Vector'Range loop

            Ada.Text_IO.Put_Line (File => Table_File, Item =>
              Float'Image(X_Index(X_Value)) & ", " & Float'Image(Y_Index(Y_Value)) & ", " &
              Float'Image(Table_Data(X_Value)(Y_Value)));

         end loop;

      end loop;

      Ada.Text_IO.Close(Table_File);


      -- Call Read_ASCII to read the file in
      Interpolation_Table.Doubly_Indexed.Read_ASCII (File_Name => Table_Name & ".txt",
        Table => Table);

      -- Call Write to write the file out to an object file
      Interpolation_Table.Doubly_Indexed.Write (File_Name => Table_Name & ".ito",
        Table => (1 => Table));

      -- Call Read to read the object file in
      Interpolation_Table.Read (File_Name => Table_Name & ".ito",
        Table => Table);

      -- Verify the values by interpolating at the independent points
      Ada.Text_IO.Put ("Verifing table object file support..."); Ada.Text_IO.Flush;

      Verification_3:
      for X_Value in Test_Table'Range loop

         for Y_Value in Test_Table'Range loop

            Good := Interpolation_Table.Doubly_Indexed.Interpolate
              (X     => X_Index(X_Value),
               Y     => Y_Index(Y_Value),
               Table => Table'access) = Table_Data(X_Value)(Y_Value);

            exit Verification_3 when not Good;
         end loop;
      end loop Verification_3;

      Class_Test_Result_File.Report_Case_Status (Good);

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
         Interpolation_Table.Doubly_Indexed.Write_ASCII(Table_Name & "2.txt", Table);
      end if;

    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Table_File) then
            Ada.Text_IO.Close (Table_File);
         end if;

   end Test_Case_8;
   ----------------------------------------------------------------------------

exception
when Event : others =>
   Ada.Text_IO.Put_Line ("unhandled exception:");
   Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
end Interpolation_Table.Doubly_Indexed_CTD_1;
