-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  James F. Narrin
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Jpats_Reposition.Container;
with Jpats_Reposition.Controller;
with Jpats_Reposition;
with Class_Test_Result_File;
with Jpats_Reposition.Ios_If;
with Jpats_Reposition_Types;
with Interfaces.C;

with Ada.Text_Io;
use Ada.Text_Io;

procedure Jpats_Reposition.Ctd is

   package Jrt renames Jpats_Reposition_Types;
   Io : Container.Io_Interface_Instance renames Container.This_Io_Interface;

   Time : Float := 0.0;
   Iconst : Float := 1.0/10.0;
   Pass : Boolean := False;
   Results : Jrt.Reposition_Command_Type;
   type Rdb_Init_Type is new Interfaces.C.Long;
   N_Rdb_Init_Value : Rdb_Init_Type;



   function N_Rdb_Init return Rdb_Init_Type;
   pragma Import(C, N_Rdb_Init, "n_rdb_init");

begin

   --initialize radio database:
   N_Rdb_Init_Value := N_Rdb_Init;

   -------------------------------------------------------
   -- Test Case 1-- Takeoff Reposition
   --
   -- Inputs:
   --  airport : KTST
   --  runway  : 27
   --  reposition selected : takeoff
   --
   --
   -- Expected Results:
   --   reposition : active
   --   position : 45.0N, 44.957W, 1000.0 ft
   --   heading  : 270.0
   --   airspeed : 0
   --   gear position : down
   --   type reposition : on ground
   --
   -- Purpose:
   --   to test takeoff reposition commanded from IOS
   --

   JPATS_Reposition.Controller.Initialize;

   JPATS_Reposition.Ios_If.Set_Airport("KTST ");
   JPATS_Reposition.Ios_If.Set_Runway ("27   ");
   JPATS_Reposition.Ios_If.Set_Airport_Change(True);
   JPATS_Reposition.Ios_If.Set_Reposition_Select(Jrt.Takeoff);

   JPATS_Reposition.Controller.Update(Iconst);

   --results:
   Results := Container.This_Subsystem.The_Reposition_Command;

   Pass :=  Results.Active and
     abs(Results.Position.Latitude - 45.0) < 0.05 and
     abs(Results.Position.Longitude + 44.958) < 0.05 and
     abs(Results.Position.Altitude - 1000.0) < 10.0 and
     abs(Results.Heading + 90.0) < 0.5 and
     Results.Gear_Position = 1.0 and
     Results.On_Ground;

     if Pass then
        Put_Line("TO Reposition OK");
     else
        Put_Line("TO Reposition Fail");
     end if;
     Class_Test_Result_File.Report_Case_Status(Pass);



     -------------------------------------------------------
   -- Test Case 2 -- ILS 8nm Final
   --
   -- Inputs:
   --  airport : KTST
   --  runway  : 27
   --  reposition selected : ILS 8nm final
   --
   --
   -- Expected Results:
   --   reposition : active
   --   position : 45.0N, 44.773W, 3549.1 ft
   --   heading  : 270.0
   --   airspeed : 110.0
   --   gear position : up
   --   type reposition : in air
   --
   -- Purpose:
   --   to test ILS 8nm  reposition commanded from IOS
   --

     JPATS_Reposition.Controller.Initialize;

     JPATS_Reposition.Ios_If.Set_Airport("KTST ");
     JPATS_Reposition.Ios_If.Set_Runway ("27   ");
     JPATS_Reposition.Ios_If.Set_Airport_Change(True);
     JPATS_Reposition.Ios_If.Set_Reposition_Select(Jrt.Ils_8nm);

     JPATS_Reposition.Controller.Update(Iconst);

     --results:
     Results := Container.This_Subsystem.The_Reposition_Command;

     Pass :=  Results.Active and
       abs(Results.Position.Latitude - 45.0) < 0.05 and
       abs(Results.Position.Longitude + 44.773) < 0.05 and
       abs(Results.Position.Altitude - 3549.1) < 10.0 and
       abs(Results.Heading + 90.0) < 0.5 and
       abs(Results.Airspeed - 110.0) < 1.0 and
       Results.Gear_Position = 0.0 and
       not Results.On_Ground;

     if Pass then
        Put_Line("ILS 8nm Reposition OK");
     else
        Put_Line("ILS 8nm Repostion Fail");
     end if;
     Class_Test_Result_File.Report_Case_Status(Pass);


   -------------------------------------------------------
   -- Test Case 3 -- Lat/Long Insert Reposition
   --
   -- Inputs:
   --  latitude : 45.0N
   --  longitude : 45.0W
   --  aircraft altitude : 5000.0 ft.
   --
   --  reposition selected : lat/long reposition
   --
   --
   -- Expected Results:
   --   reposition : active
   --   position : 45.0N, 45.0, 5000.0 ft
   --   heading  : 270.0
   --   airspeed : 180.0
   --   gear position : down
   --
   -- Purpose:
   --   to test lat/long insert reposition commanded from IOS
   --

     JPATS_Reposition.Controller.Initialize;

     JPATS_Reposition.Ios_If.Set_Latitude(45.0);
     JPATS_Reposition.Ios_If.Set_Longitude(-45.0);
     JPATS_Reposition.Ios_If.Set_Reposition_Select(Jrt.Lat_Lon_Reposition);

     JPATS_Reposition.Controller.Update(Iconst);

     --results:
     Results := Container.This_Subsystem.The_Reposition_Command;

     Pass :=  Results.Active and
       abs(Results.Position.Latitude - 45.0) < 0.05 and
       abs(Results.Position.Longitude + 45.0) < 0.05 and
       abs(Results.Position.Altitude - 5000.0) < 10.0 and
       abs(Results.Heading + 90.0) < 0.5 and
       abs(Results.Airspeed - 180.0) < 1.0 and
       Results.Gear_Position = 1.0;

     if Pass then
        Put_Line("Lat/Long Reposition OK");
     else
        Put_Line("lat/Long Reposition Fail");
     end if;
     Class_Test_Result_File.Report_Case_Status(Pass);

   -------------------------------------------------------
   -- Test Case 4 -- Ident Reposition
   --
   -- Inputs:
   --  ident : NVE
   --  reposition selected : ident reposition
   --
   --
   -- Expected Results:
   --   reposition : active
   --   position : 45.0N,43.586W,5000.0 ft.
   --   heading  : 270.0
   --   airspeed : 180.0
   --
   -- Purpose:
   --   to test ident reposition commanded from IOS
   --

     JPATS_Reposition.Controller.Initialize;

     JPATS_Reposition.Ios_If.Set_Ident("NVE  ");
     JPATS_Reposition.Ios_If.Set_Reposition_Select(Jrt.Ident_Reposition);

     JPATS_Reposition.Controller.Update(Iconst);

     --results:
     Results := Container.This_Subsystem.The_Reposition_Command;

     Pass :=  Results.Active and
       abs(Results.Position.Latitude - 45.0) < 0.05 and
       abs(Results.Position.Longitude + 43.586) < 0.05 and
       abs(Results.Position.Altitude - 5000.0) < 10.0 and
       abs(Results.Heading + 90.0) < 0.5 and
       abs(Results.Airspeed - 180.0) < 1.0 and
       Results.Gear_Position = 1.0;

     if Pass then
        Put_Line("Ident Reposition OK");
     else
        Put_Line("Ident Reposition Fail");
     end if;
     Class_Test_Result_File.Report_Case_Status(Pass);

   -------------------------------------------------------
   -- Test Case 5 -- Formation reposition
   --
   -- Inputs:
   --  ident : NVE
   --  reposition selected : ident reposition
   --
   --
   -- Expected Results:
   --   reposition : active
   --   position : 45.014N,44.981W,6000.0 ft.
   --   heading  : 270.0
   --   airspeed : 200.0
   --
   --
   -- Purpose:
   --   to test formation reposition commanded from IOS
   --

     JPATS_Reposition.Controller.Initialize;

     JPATS_Reposition.Ios_If.Set_Fore_Aft(-5000.0);
     JPATS_Reposition.Ios_If.Set_Left_Right(5000.0);
     JPATS_Reposition.Ios_If.Set_Above_Below(1000.0);
     JPATS_Reposition.Ios_If.Set_Reposition_Select(Jrt.Formation);

     JPATS_Reposition.Controller.Update(Iconst);

     --results:
     Results := Container.This_Subsystem.The_Reposition_Command;

     Pass :=  Results.Active and
       abs(Results.Position.Latitude - 45.014) < 0.05 and
       abs(Results.Position.Longitude + 44.981) < 0.05 and
       abs(Results.Position.Altitude - 6000.0) < 10.0 and
       abs(Results.Heading + 90.0) < 0.5 and
       abs(Results.Airspeed - 200.0) < 1.0 and
       Results.Gear_Position = 1.0;

     if Pass then
        Put_Line("Formation Reposition OK");
     else
        Put_Line("Formation Reposition Fail");
     end if;
     Class_Test_Result_File.Report_Case_Status(Pass);

end;
