-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Text_Io;
with Ada.Float_Text_Io;

with Jpats_Propeller.Scheduler;
with Jpats_Propeller.Controller;
with Jpats_Propeller.Container;
with Jpats_Propeller_Types;


with Geometry;
with Hartzell_4_Blade_Propeller_Geometry;
with Hartzell_4_Blade_Propeller;
with Propeller_Interface_Unit;

with Torque_Types;
with Force_Types;
with Angle_Types;
with Mass_Types;
with Vector_Types;
with Class_Test_Result_File;



procedure Jpats_Propeller.Scheduler_Ctd is


   function "=" (A : in Jpats_Propeller_Types.Coordinate_Type.Instance;
                   B : in Jpats_Propeller_Types.Coordinate_Type.Instance) return Boolean is
   begin
     return A(Vector_Types.X) = B(Vector_Types.X) and A(Vector_Types.Y)=B(Vector_Types.Y) and A(Vector_Types.Z)=B(Vector_Types.Z) ;
   end "=";

   function "=" (A : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
                   B : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance) return Boolean is
   begin
     return A(Vector_Types.Roll) = B(Vector_Types.Roll) and A(Vector_Types.Pitch)=B(Vector_Types.Pitch) and A(Vector_Types.Yaw)=B(Vector_Types.Yaw) ;
   end "=";

   function "=" (A : in Jpats_Propeller_Types.Torque_Vector_Type.Instance;
                   B : in Jpats_Propeller_Types.Torque_Vector_Type.Instance) return Boolean is
   begin
     return A(Vector_Types.X) = B(Vector_Types.X) and A(Vector_Types.Y)=B(Vector_Types.Y) and A(Vector_Types.Z)=B(Vector_Types.Z) ;
   end "=";

   function "=" (A : in Jpats_Propeller_Types.Force_Vector_Type.Instance;
                   B : in Jpats_Propeller_Types.Force_Vector_Type.Instance) return Boolean is
   begin
     return A(Vector_Types.X) = B(Vector_Types.X) and A(Vector_Types.Y)=B(Vector_Types.Y) and A(Vector_Types.Z)=B(Vector_Types.Z) ;
   end "=";



begin

Test_1 :
        -------------------------------------------
        -- Test Case 1
        -- Test the Initialize function
        -- Inputs           : A Valid Propeller.Controller Class
        -- Expected Results : A query of all accessor functions of the
        --                    domain classes with by the controller should
        --                    show that all of the initialized instance
        --                    variables are set to the correct values
        --                    thus, testing the initialize and accessor
        --                    functions.
        -- Purpose          : Test the initialize and accessor functions.
        -- Interface Tested : Power_Management_Unit.Initialize
        --

        declare
          Propeller_System                     : aliased Jpats_Propeller.Scheduler.Instance;
          Test_Result                          : Boolean;
          Init_Result                          : Boolean;
          Test_Number                          : Integer := 1;
          Test_Angle_1                         : Angle_Types.Degrees;
          Axial_Thrust                         : Jpats_Propeller_Types.Force_Vector_Type.Instance;
          Axial_Torque                         : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
          Axial_Coeff                          : Normalized_Types.Normalize;
          Test_Ipgyro                          : Mass_Types.Slugs_Feet_Sq;
          Test_Diameter                        : Length_Types.Feet;
          Test_Area                            : Length_Types.Sq_Feet;
          Test_Low_Pitch                       : Angle_Types.Degrees;
          Test_High_Pitch                      : Angle_Types.Degrees;
          Thrust_Reference_Point               : Jpats_Propeller_Types.Coordinate_Type.Instance;
          Propeller_Rotational_Orientation     : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
          Moment_Arm                           : Jpats_Propeller_Types.Coordinate_Type.Instance;
          Moment_Tensor                        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
          Force_Tensor                         : Jpats_Propeller_Types.Force_Vector_Type.Instance;
          Gyroscopic_Moment_Tensor             : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        begin


          Jpats_Propeller.Scheduler.Initialize
            (An_Instance => Propeller_System );

          -- Use the accessor functions from the components of the
          -- subsystem to check the initialize function from the
          -- components of the Hartzell

            -- Access Delta_Blade_Angle
            Test_Angle_1 := Propeller_Interface_Unit.Delta_Blade_Angle
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_PIU);
            Test_Result := Test_Angle_1 = Angle_Types.Degrees(0.0);
            Init_Result := Test_Result;

            -- Test the Hartzell_4_Blade_Propeller

            -- Access Axial_Thrust
            Axial_Thrust := Hartzell_4_Blade_Propeller.Axial_Thrust
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller);
            Test_Result :=  Axial_Thrust = (0.0,0.0,0.0);
            Init_Result := Init_Result and Test_Result;

            -- Access Axial_Thrust_Coefficient_Limited
            Axial_Coeff := Hartzell_4_Blade_Propeller.Axial_Thrust_Coefficient_Limited
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller);
            Test_Result := Axial_Coeff = Normalized_Types.Normalize(0.0);
            Init_Result := Init_Result and Test_Result;

            -- Access Axial_Torque_Absorbed
            Axial_Torque := Hartzell_4_Blade_Propeller.Axial_Torque_Absorbed
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller);
            Test_Result := Axial_Torque = (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
            Init_Result := Init_Result and Test_Result;


            -- Propeller Geometry Component

            -- Access Propeller_Inertia_Ipgyro
            Test_Ipgyro := Hartzell_4_Blade_Propeller_Geometry.Propeller_Inertia_Ipgyro
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Geometry);
            Test_Result :=  Test_Ipgyro = Mass_Types.Slugs_Feet_Sq(8.97);
            Init_Result := Init_Result and Test_Result;

            -- Access Propeller_Diameter
            Test_Diameter := Hartzell_4_Blade_Propeller_Geometry.Propeller_Diameter
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Geometry);
            Test_Result := Test_Diameter = Length_Types.Feet(8.083);
            Init_Result := Init_Result and Test_Result;

            -- Access Wetted_Surface_Area
            Test_Area := Hartzell_4_Blade_Propeller_Geometry.Wetted_Surface_Area
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Geometry);
            Test_Result := Test_Area = Length_Types.Sq_feet(173.5);
            Init_Result := Init_Result and Test_Result;

            -- Access Low_Pitch_Stop
            Test_Low_Pitch := Hartzell_4_Blade_Propeller_Geometry.Low_Pitch_Stop
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Geometry);
            Test_Result := Test_Low_Pitch = Angle_Types.Degrees(9.9);
            Init_Result := Init_Result and Test_Result;

            -- Access High_Pitch_Stop
            Test_High_Pitch := Hartzell_4_Blade_Propeller_Geometry.High_Pitch_Stop
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Geometry);
            Test_Result := Test_High_Pitch = Angle_Types.Degrees(84.0);
            Init_Result := Init_Result and Test_Result;

            -- Propeller Installation Geometry Component

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Init_Result := Init_Result and Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Init_Result :=  Init_Result and Test_Result;
            --
            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Init_Result :=  Init_Result and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Init_Result :=  Init_Result and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Init_Result :=  Init_Result and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => Jpats_Propeller.Container.Next_Subsystem.The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Init_Result :=  Init_Result and Test_Result;

            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Init_Result, Number => Test_Number);
        end Test_1;

        Test_2:
          declare
            -------------------------------------------
            -- Test Case 2
            -- Test the Update function
            -- Inputs           : A Valid Jpats_Powerplant.Scheduler Class
            -- Expected Results : A query of all accessor functions in Jpats_Propeller
            --                    should show that all of the initialized instance
            --                    variables are set to the correct values thus,
            --                    testing the initialize and accessor functions.
            -- Purpose          : Test the update functions.
            -- Interface Tested : Jpats_Powerplant.Scheduler.Update
            --                    Jpats_Powerplant.Scheduler.Initialize

            Propeller_System                     : aliased Jpats_Propeller.Scheduler.Instance;
            Test_Result                          : Boolean;
            Init_Result                          : Boolean;
            Test_Number                          : Integer := 2;
            Iconst                               : Float;
            Time                                 : Float;
            Axial_Thrust_Coeff_Lim               : Normalized_Types.Normalize;
            Torque_Absorbed                      : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            X_Moment_Arm                         : Length_Types.Feet;
            Moment_Tensor                        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                         : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor             : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            use type Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            use type Jpats_Propeller_Types.Force_Vector_Type.Instance;
          begin
            --             Jpats_Powerplant.Container.This_Io_Interface.Throttle_Control  := Angle_Types.Degrees(30.0);
            --             Jpats_Powerplant.Container.This_Io_Interface.Ignition_On_Switch := True;
            --             Jpats_Powerplant.Container.This_Io_Interface.Ignition_Off_Switch := False;
            --             Jpats_Powerplant.Container.This_Io_Interface.Starter_Man_Switch := False;
            --             Jpats_Powerplant.Container.This_Io_Interface.Starter_Auto_Switch := True;
            --             Jpats_Powerplant.Container.This_Io_Interface.Prop_Feather_Cut_Off_Switch := False;
            --             Jpats_Powerplant.Container.This_Io_Interface.Pmu_Mode_Switch := True;
            Iconst :=  1.0/15.0;
            Time   :=  Iconst;
            Jpats_Propeller.Scheduler.Initialize
              (An_Instance => Propeller_System );

            while Time < 16.0 loop
              Jpats_Propeller.Scheduler.Update
                (An_Instance          => Propeller_System,
                 Integration_Constant => Iconst);
              Time := Time + Iconst;
            end loop;


            -- Access Moment_Arm
            X_Moment_Arm :=  Jpats_Propeller.Propeller_Moment_Arm;
            Test_Result := X_Moment_Arm  = (Length_Types.Feet(-35.41/12.0));
            Init_Result :=  Test_Result;

          -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Jpats_Propeller.Gyroscopic_Moments_Tensor;
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Init_Result := Init_Result and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Jpats_Propeller.Moment_Tensor;
            Ada.Float_Text_Io.Put (Moment_Tensor(Vector_Types.X));
            Ada.Float_Text_Io.Put (Moment_Tensor(Vector_Types.Y));
            Ada.Float_Text_Io.Put (Moment_Tensor(Vector_Types.Z));
            Ada.Text_Io.New_Line;
            Test_Result :=
              (Jpats_Propeller_Types.Torque_Vector_Type.Magnitude
               (Moment_Tensor - (Torque_Types.Ft_Lbf(-987.666),Torque_Types.Ft_Lbf(5262.78),Torque_Types.Ft_Lbf(50.6429)))/5354.895 < 0.0001);
            Init_Result :=  Init_Result and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Jpats_Propeller.Force_Tensor;
            Ada.Float_Text_Io.Put (Force_Tensor(Vector_Types.X));
            Ada.Float_Text_Io.Put (Force_Tensor(Vector_Types.Y));
            Ada.Float_Text_Io.Put (Force_Tensor(Vector_Types.Z));
            Ada.Text_Io.New_Line;
            Test_Result := (Jpats_Propeller_Types.Force_Vector_Type.Magnitude
                            (Force_Tensor -(Force_Types.Lbf(695.144),Force_Types.Lbf(24.2750),Force_Types.Lbf(18.2141)) )/699.64 < 0.0001);
            Init_Result :=  Init_Result and Test_Result;

            -- Axial Thrust Coefficient Limited
            Axial_Thrust_Coeff_Lim := Jpats_Propeller.Axial_Thrust_Coefficient_Limited;
            Ada.Float_Text_Io.Put (Axial_Thrust_Coeff_Lim);
            Ada.Text_Io.New_Line;
            Test_Result := ((Axial_Thrust_Coeff_Lim - Normalized_Types.Normalize(4.04051))/4.04051 < 0.0001);
            Init_Result := Init_Result and Test_Result;

            -- Axial Torque Absorbed
            Torque_Absorbed := Jpats_Propeller.Axial_Torque_Absorbed;
            Ada.Float_Text_Io.Put (Torque_Absorbed(Vector_Types.X));
            Ada.Float_Text_Io.Put (Torque_Absorbed(Vector_Types.Y));
            Ada.Float_Text_Io.Put (Torque_Absorbed(Vector_Types.Z));
            Ada.Text_Io.New_Line;
            Test_Result := (Jpats_Propeller_Types.Torque_Vector_Type.Magnitude
                            (Torque_Absorbed -
                             (Torque_Types.Ft_Lbf(801.795),
                              Torque_Types.Ft_Lbf(0.0),
                              Torque_Types.Ft_Lbf(0.0)))/801.795 < 0.0001);
            Init_Result := Init_Result and Test_Result;

            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Init_Result, Number => Test_Number);

          end Test_2;
     Ada.Text_Io.Put_Line("This is the end.");


end Jpats_Propeller.Scheduler_Ctd;











