-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer: Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0w
with Ada.Text_Io;
with Ada.Float_Text_Io;
with Ada.Numerics;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;
with Geometry;
with Jpats_Propeller_Types;
with Vector_Types;
use  Vector_Types;
with Length_Types;
with Force_Types;
with Torque_Types;
with Angle_Types;
with Mass_Types;
with Class_Test_Result_File;

procedure Geometry_Ctd is
  use type Jpats_Propeller_Types.Force_Vector_Type.Instance;

--  function Pi return Float renames Ada.Numerics.Pi;

  function "=" (A : in Jpats_Propeller_Types.Coordinate_Type.Instance;
                B : in Jpats_Propeller_Types.Coordinate_Type.Instance) return Boolean is
  begin
    return A(X) = B(X) and A(Y)=B(Y) and A(Z)=B(Z) ;
  end "=";

  function "=" (A : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
                B : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance) return Boolean is
  begin
    return A(Roll) = B(Roll) and A(Pitch)=B(Pitch) and A(Yaw)=B(Yaw) ;
  end "=";

  function "=" (A : in Jpats_Propeller_Types.Torque_Vector_Type.Instance;
                B : in Jpats_Propeller_Types.Torque_Vector_Type.Instance) return Boolean is
  begin
    return A(X) = B(X) and A(Y)=B(Y) and A(Z)=B(Z) ;
  end "=";

  function "=" (A : in Jpats_Propeller_Types.Force_Vector_Type.Instance;
                B : in Jpats_Propeller_Types.Force_Vector_Type.Instance) return Boolean is
  begin
    return A(X) = B(X) and A(Y)=B(Y) and A(Z)=B(Z) ;
  end "=";

begin

  Ada.Text_Io.Put_Line ("Geometry_Ctd.exe is executing.");

Test_1:
          -------------------------------------------
          -- Test Case 1
          -- Test the Initialize function
          -- Inputs           : A Geometry  Class
          -- Expected Results : A query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the initialize and accessor
          --                    functions.
          -- Purpose          : Test the initialize and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --
          declare
            Test_Number                          : Integer := 1;
            Test_Result                          : Boolean;
            Test_Init                            : Boolean;
            The_Propeller_Installation_Geometry : Geometry.Instance;
            Thrust_Reference_Point               : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation    : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                           : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                         : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor             : Jpats_Propeller_Types.Torque_Vector_Type.Instance;

          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.float_Text_Io.Put(Thrust_Reference_Point(Vector_Types.X));
--            Ada.float_Text_Io.Put(Thrust_Reference_Point(Vector_Types.Y));
--            Ada.float_Text_Io.Put(Thrust_Reference_Point(Vector_Types.Z));
            Test_Result :=  (Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0)));
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.float_Text_Io.Put(Moment_arm(Vector_Types.X));
--            Ada.float_Text_Io.Put(Moment_arm(Vector_Types.X));
--            Ada.float_Text_Io.Put(Moment_arm(Vector_Types.X));
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);
          end Test_1;

      Test_2:

          -------------------------------------------
          -- Test Case 2
          -- Test the Set function that corespond to the accessor functions.
          -- Inputs           : A Geometry  Class
          -- Expected Results : A call on the set functions should set specific
          --                    values in each of the class attributes.  A
          --                    subsequent query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the Set and accessor
          --                    functions.
          -- Purpose          : Test the Set and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --                    Geometry.Set_Thrust_Reference_Point
          --                    Geometry.Set_Rotational_Orientation
          --                    Geometry.Set_Moment_Arm
          --                    Geometry.Set_Moment_Tensor
          --                    Geometry.Set_Force_Tensor
          --                    Geometry.Set_Gyroscopic_Moments_Tensor
          --
          declare
            Test_Number                             : Integer := 2;
            Test_Result                             : Boolean;
            Test_Init                               : Boolean;
            The_Propeller_Installation_Geometry     : Geometry.Instance;
            Thrust_Reference_Point                  : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                              : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                           : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor                : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Thrust_Reference_Point_Value            : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation_Value  : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm_Value                        : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor_Value                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor_Value                      : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor_Value          : Jpats_Propeller_Types.Torque_Vector_Type.Instance;

          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point_Value :=
              (Length_Types.Feet(Ada.Numerics.Pi),Length_Types.Feet(Ada.Numerics.Pi),Length_Types.Feet(Ada.Numerics.Pi));
            Geometry.Set_Thrust_Reference_Point
              (An_Instance     => The_Propeller_Installation_Geometry,
               Reference_Point => Thrust_Reference_Point_Value);
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = Thrust_Reference_Point_Value;
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation_Value :=
              (Angle_Types.Degrees(Ada.Numerics.Pi),Angle_Types.Degrees(Ada.Numerics.Pi),Angle_Types.Degrees(Ada.Numerics.Pi));
            Geometry.Set_Rotational_Orientation
              (An_Instance  => The_Propeller_Installation_Geometry,
               Rotate       => Propeller_Rotational_Orientation_Value );
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = Propeller_Rotational_Orientation_Value;
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm_Value :=
              (Length_Types.Feet(Ada.Numerics.Pi),Length_Types.Feet(Ada.Numerics.Pi),Length_Types.Feet(Ada.Numerics.Pi));
            Geometry.Set_Moment_Arm
              (An_Instance  => The_Propeller_Installation_Geometry,
               Moment_Arm   => Moment_Arm_Value );
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Moment_Arm  = Moment_Arm_Value;
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(Ada.Numerics.Pi),Torque_Types.Ft_Lbf(Ada.Numerics.Pi),Torque_Types.Ft_Lbf(Ada.Numerics.Pi));
            Geometry.Set_Moment_Tensor
              (An_Instance   => The_Propeller_Installation_Geometry,
               Moment_Tensor => Moment_Tensor_Value );
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = Moment_Tensor_Value;
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor_Value :=
              (Force_Types.Lbf(Ada.Numerics.Pi),Force_Types.Lbf(Ada.Numerics.Pi),Force_Types.Lbf(Ada.Numerics.Pi));
            Geometry.Set_Force_Tensor
              (An_Instance  => The_Propeller_Installation_Geometry,
               Force        => Force_Tensor_Value );
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = Force_Tensor_Value;
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(Ada.Numerics.Pi),Torque_Types.Ft_Lbf(Ada.Numerics.Pi),Torque_Types.Ft_Lbf(Ada.Numerics.Pi));
            Geometry.Set_Gyroscopic_Moment_Tensor
              (An_Instance              => The_Propeller_Installation_Geometry,
               Gyroscopic_Moment_Tensor => Gyroscopic_Moment_Tensor_Value );
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor = Gyroscopic_Moment_Tensor_Value;
            Test_Init :=  Test_Init and Test_Result;

            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);
          end Test_2;

Test_3:
          -------------------------------------------
          -- Test Case 3
          -- Test the Update function
          -- Inputs           : A Geometry Class
          --                  : An aircraft cg position of 0.0.
          --                  :
          --                  : Force vector with X component only.
          --                  : An rotational offset defined in the
          --                  : initialize procedure for force and moment.
          --                  : A translation offset defined in the
          --                  : initialize procedure for the arbiitrary force and moment.
          --                  : Moment Tensor with 0.0 components.
          -- Expected Results : A query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the initialize and accessor
          --                    functions.  Then the update function
          --                    is called producing an equivalent force
          --                    and moment acting at the cg of the aircraft
          --                    from an arbitrarily oriented force and moment
          --                    at an arbitrary point.
          -- Purpose          : Test the initialize and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --                    Geometry.Update;

          declare
            Test_Number                             : Integer := 3;
            Test_Result                             : Boolean;
            Test_Init                               : Boolean;
            The_Propeller_Installation_Geometry    : Geometry.Instance;
            Thrust_Reference_Point                  : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation       : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                              : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                           : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor                : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
--          Thrust_Reference_Point_Value            : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Aircraft_Cg                             : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor_Value                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor_Value                      : Jpats_Propeller_Types.Force_Vector_Type.Instance;
--          Gyroscopic_Moment_Tensor_Value          : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Angular_Rate                   : Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
            Inertia                                 : Mass_Types.Slugs_Feet_Sq;
            Propeller_Rpm_Value                    : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            Force_Tensor_Value :=
              (Force_Types.Lbf(0.0),Force_Types.Lbf(1000.0),Force_Types.Lbf(0.0));
            Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
            Aircraft_Cg :=
              (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
            Inertia := Mass_Types.Slugs_Feet_Sq(1.0);
            Propeller_Rpm_Value :=
              (Angle_Types.Rpm(30000.0),Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0));
            Aircraft_Angular_Rate :=
              (Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0));

            -- Method Update
            Geometry.Update
              (An_Instance            => The_Propeller_Installation_Geometry,
               Axial_Thrust           => Force_Tensor_Value,
               Axial_Torque           => Moment_Tensor_Value,
               Inertia                => Inertia,
               Rpm                    => Propeller_Rpm_Value,
               Aircraft_Angular_Rate  => Aircraft_Angular_Rate,
               Aircraft_Cg            => Aircraft_Cg);

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Init and Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (Length_Types.Feet(-35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(-92.26/12.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.Float_Text_Io.Put (Force_Tensor(X));
--            Ada.Float_Text_Io.Put (Force_Tensor(Y));
--            Ada.Float_Text_Io.Put (Force_Tensor(Z));
--            Ada.Text_Io.New_Line;
            Test_Result := (Jpats_Propeller_Types.Force_Vector_Type.Magnitude
                            (Force_Tensor - (Force_Types.Lbf(-34.899),Force_Types.Lbf(999.391),Force_Types.Lbf(0.0)) )/1000.0 < 0.0001);
            Test_Init := Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=
              ((Moment_Tensor(X) - (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)))/
               (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)) < 0.00001)     and
              ((Moment_Tensor(Y) - (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)))/
               (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)) < 0.00001) and
              ((Moment_Tensor(Z) - (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)))/
               (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)) < 0.00001);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;
            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);


          end Test_3;


Test_4:
          -------------------------------------------
          -- Test Case 4
          -- Test the Update function
          -- Inputs           : A Geometry Class
          --                  : An aircraft cg position of 0.0.
          --                  :
          --                  : Force vector with Y component only.
          --                  : An rotational offset defined in the
          --                  : initialize procedure for force and moment.
          --                  : A translation offset defined in the
          --                  : initialize procedure for the arbiitrary force and moment.
          --                  : Moment Tensor with 0.0 components.
          -- Expected Results : A query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the initialize and accessor
          --                    functions.  Then the update function
          --                    is called producing an equivalent force
          --                    and moment acting at the cg of the aircraft
          --                    from an arbitrarily oriented force and moment
          --                    at an arbitrary point.
          -- Purpose          : Test the initialize and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --                    Geometry.Update;

          declare
            Test_Number                             : Integer := 4;
            Test_Result                             : Boolean;
            Test_Init                               : Boolean;
            The_Propeller_Installation_Geometry     : Geometry.Instance;
            Thrust_Reference_Point                  : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                              : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                           : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor                : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Cg                             : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor_Value                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor_Value                      : Jpats_Propeller_Types.Force_Vector_Type.Instance;
--          Gyroscopic_Moment_Tensor_Value          : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Angular_Rate                   : Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
            Inertia                                 : Mass_Types.Slugs_Feet_Sq;
            Propeller_Rpm_Value                    : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            Force_Tensor_Value :=
              (Force_Types.Lbf(0.0),Force_Types.Lbf(1000.0),Force_Types.Lbf(0.0));
            Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
            Aircraft_Cg :=
              (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
            Inertia := Mass_Types.Slugs_Feet_Sq(1.0);
            Propeller_Rpm_Value :=
              (Angle_Types.Rpm(30000.0),Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0));
            Aircraft_Angular_Rate :=
              (Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0));

            -- Method Update
            Geometry.Update
              (An_Instance            => The_Propeller_Installation_Geometry,
               Axial_Thrust           => Force_Tensor_Value,
               Axial_Torque           => Moment_Tensor_Value,
               Inertia                => Inertia,
               Rpm                    => Propeller_Rpm_Value,
               Aircraft_Angular_Rate  => Aircraft_Angular_Rate,
               Aircraft_Cg            => Aircraft_Cg);

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Init and Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (Length_Types.Feet(-35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(-92.26/12.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.Float_Text_Io.Put (Force_Tensor(X));
--            Ada.Float_Text_Io.Put (Force_Tensor(Y));
--            Ada.Float_Text_Io.Put (Force_Tensor(Z));
--            Ada.Text_Io.New_Line;
            Test_Result := (Jpats_Propeller_Types.Force_Vector_Type.Magnitude
                            (Force_Tensor - (Force_Types.Lbf(-34.8995),Force_Types.Lbf(999.391),Force_Types.Lbf(0.0)) )/1000.0 < 0.0001);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=
              ((Moment_Tensor(X) - (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)))/
               (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)) < 0.00001)     and
              ((Moment_Tensor(Y) - (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)))/
               (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)) < 0.00001) and
              ((Moment_Tensor(Z) - (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)))/
               (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)) < 0.00001);
            Test_Init :=  Test_Init and Test_Result;

          -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;


            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);


          end Test_4;

Test_5:
          -------------------------------------------
          -- Test Case 5
          -- Test the Update function
          -- Inputs           : A Geometry Class
          --                  : An aircraft cg position of 0.0.
          --                  :
          --                  : Force vector with Z component only.
          --                  : An rotational offset defined in the
          --                  : initialize procedure for force and moment.
          --                  : A translation offset defined in the
          --                  : initialize procedure for the arbiitrary force and moment.
          --                  : Moment Tensor with 0.0 components.
          -- Expected Results : A query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the initialize and accessor
          --                    functions.  Then the update function
          --                    is called producing an equivalent force
          --                    and moment acting at the cg of the aircraft
          --                    from an arbitrarily oriented force and moment
          --                    at an arbitrary point.
          -- Purpose          : Test the initialize and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --                    Geometry.Update;

          declare
            Test_Number                             : Integer := 5;
            Test_Result                             : Boolean;
            Test_Init                               : Boolean;
            The_Propeller_Installation_Geometry     : Geometry.Instance;
            Thrust_Reference_Point                  : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                              : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                           : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor                : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Cg                             : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor_Value                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor_Value                      : Jpats_Propeller_Types.Force_Vector_Type.Instance;
--          Gyroscopic_Moment_Tensor_Value          : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Angular_Rate                   : Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
            Inertia                                 : Mass_Types.Slugs_Feet_Sq;
            Propeller_Rpm_Value                     : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            Force_Tensor_Value :=
              (Force_Types.Lbf(0.0),Force_Types.Lbf(0.0),Force_Types.Lbf(1000.0));
            Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
            Aircraft_Cg :=
              (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
            Inertia := Mass_Types.Slugs_Feet_Sq(1.0);
            Propeller_Rpm_Value :=
              (Angle_Types.Rpm(30000.0),Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0));
            Aircraft_Angular_Rate :=
              (Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0));

            -- Method Update
            Geometry.Update
              (An_Instance            => The_Propeller_Installation_Geometry,
               Axial_Thrust           => Force_Tensor_Value,
               Axial_Torque           => Moment_Tensor_Value,
               Inertia                => Inertia,
               Rpm                    => Propeller_Rpm_Value,
               Aircraft_Angular_Rate  => Aircraft_Angular_Rate,
               Aircraft_Cg            => Aircraft_Cg);

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Init and Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (Length_Types.Feet(-35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(-92.26/12.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.Float_Text_Io.Put (Force_Tensor(X));
--            Ada.Float_Text_Io.Put (Force_Tensor(Y));
--            Ada.Float_Text_Io.Put (Force_Tensor(Z));
--            Ada.Text_Io.New_Line;
            Test_Result := (Jpats_Propeller_Types.Force_Vector_Type.Magnitude
                            (Force_Tensor - (Force_Types.Lbf(0.0),Force_Types.Lbf(-0.913562),Force_Types.Lbf(999.657)) )/1000.0 < 0.0001);
--            Test_Result :=
--              (abs((Force_Tensor(X) - (1000.0* Sin(0.0)*Sin(-1.5)))) < 0.000001) and
--              (abs((Force_Tensor(Y) - (1000.0*(Cos(0.0)*Sin(-2.0)*Sin(-1.5) - Sin(-1.5)*Sin(-1.5))))/
--                   (1000.0*(Cos(0.0)*Sin(-2.0)*Sin(-1.5) - Sin(-1.5)*Sin(-1.5)))) < 0.000001) and
--              (abs((Force_Tensor(Z) -(1000.0* Cos(0.0)*Cos(-2.0)))/
--                   (1000.0* Cos(0.0)*Cos(-2.0))) < 0.000001);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=
              ((Moment_Tensor(X) - (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)))/
               (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)) < 0.00001)     and
              ((Moment_Tensor(Y) - (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)))/
               (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)) < 0.00001) and
              ((Moment_Tensor(Z) - (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)))/
               (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)) < 0.00001);
            Test_Init :=  Test_Init and Test_Result;

          -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;


            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);


          end Test_5;
Test_6:
          -------------------------------------------
          -- Test Case 6
          -- Test the Update function
          -- Inputs           : A Geometry Class
          --                  : An aircraft cg position of 0.0.
          --                  :
          --                  : A zero Force vector.
          --                  : An rotational offset defined in the
          --                  : initialize procedure for force and moment.
          --                  : A translation offset defined in the
          --                  : initialize procedure for the arbiitrary force and moment.
          --                  : Moment Tensor with X component Only.
          -- Expected Results : A query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the initialize and accessor
          --                    functions.  Then the update function
          --                    is called producing an equivalent force
          --                    and moment acting at the cg of the aircraft
          --                    from an arbitrarily oriented force and moment
          --                    at an arbitrary point.
          -- Purpose          : Test the initialize and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --                    Geometry.Update;

          declare
            Test_Number                             : Integer := 6;
            Test_Result                             : Boolean;
            Test_Init                               : Boolean;
            The_Propeller_Installation_Geometry     : Geometry.Instance;
            Thrust_Reference_Point                  : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                              : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                           : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor                : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Cg                             : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor_Value                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor_Value                      : Jpats_Propeller_Types.Force_Vector_Type.Instance;
--          Gyroscopic_Moment_Tensor_Value          : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Angular_Rate                   : Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
            Inertia                                 : Mass_Types.Slugs_Feet_Sq;
            Propeller_Rpm_Value                     : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            Force_Tensor_Value :=
              (Force_Types.Lbf(0.0),Force_Types.Lbf(0.0),Force_Types.Lbf(0.0));
            Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(1000.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
            Aircraft_Cg :=
              (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
            Inertia := Mass_Types.Slugs_Feet_Sq(1.0);
            Propeller_Rpm_Value :=
              (Angle_Types.Rpm(30000.0),Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0));
            Aircraft_Angular_Rate :=
              (Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0));

            -- Method Update
            Geometry.Update
              (An_Instance            => The_Propeller_Installation_Geometry,
               Axial_Thrust           => Force_Tensor_Value,
               Axial_Torque           => Moment_Tensor_Value,
               Inertia                => Inertia,
               Rpm                    => Propeller_Rpm_Value,
               Aircraft_Angular_Rate  => Aircraft_Angular_Rate,
               Aircraft_Cg            => Aircraft_Cg);

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Init and Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (Length_Types.Feet(-35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(-92.26/12.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Force_Tensor = (0.0,0.0,0.0);
--              (abs((Force_Tensor(X) - (1000.0* Sin(0.0)*Sin(-1.5)))) < 0.000001) and
--              (abs((Force_Tensor(Y) - (1000.0*(Cos(0.0)*Sin(-2.0)*Sin(-1.5) - Sin(-1.5)*Sin(-1.5))))/
--                  (1000.0*(Cos(0.0)*Sin(-2.0)*Sin(-1.5) - Sin(-1.5)*Sin(-1.5)))) < 0.000001) and
--              (abs((Force_Tensor(Z) -(1000.0* Cos(0.0)*Cos(-2.0)))/
--                   (1000.0* Cos(0.0)*Cos(-2.0))) < 0.000001);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.Float_Text_Io.Put(Moment_Tensor(X));
--            Ada.Float_Text_Io.Put(Moment_Tensor(Y));
--            Ada.Float_Text_Io.Put(Moment_Tensor(Z));
--            Ada.Text_Io.New_Line;
            Test_Result :=
              ((Moment_Tensor(X) + (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)))/
               (1000.0) < 0.00001)     and
              ((Moment_Tensor(Y) + (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)))
               < 0.00001) and
              ((Moment_Tensor(Z) + (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)))
               < 0.00001);
            Test_Init :=  Test_Init and Test_Result;

          -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;


            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);


          end Test_6;
test_7:
          -------------------------------------------
          -- Test Case 7
          -- Test the Update function
          -- Inputs           : A Geometry Class
          --                  : An aircraft cg position of 0.0.
          --                  :
          --                  : A zero Force vector.
          --                  : An rotational offset defined in the
          --                  : initialize procedure for force and moment.
          --                  : A translation offset defined in the
          --                  : initialize procedure for the arbiitrary force and moment.
          --                  : Moment Tensor with Y component only.
          -- Expected Results : A query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the initialize and accessor
          --                    functions.  Then the update function
          --                    is called producing an equivalent force
          --                    and moment acting at the cg of the aircraft
          --                    from an arbitrarily oriented force and moment
          --                    at an arbitrary point.
          -- Purpose          : Test the initialize and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --                    Geometry.Update;

          declare
            Test_Number                             : Integer := 7;
            Test_Result                             : Boolean;
            Test_Init                               : Boolean;
            The_Propeller_Installation_Geometry     : Geometry.Instance;
            Thrust_Reference_Point                  : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                              : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                           : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor                : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Cg                             : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor_Value                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor_Value                      : Jpats_Propeller_Types.Force_Vector_Type.Instance;
--          Gyroscopic_Moment_Tensor_Value          : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Angular_Rate                   : Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
            Inertia                                 : Mass_Types.Slugs_Feet_Sq;
            Propeller_Rpm_Value                     : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            Force_Tensor_Value :=
              (Force_Types.Lbf(0.0),Force_Types.Lbf(0.0),Force_Types.Lbf(0.0));
            Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(1000.0),Torque_Types.Ft_Lbf(0.0));
            Aircraft_Cg :=
              (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
            Inertia := Mass_Types.Slugs_Feet_Sq(1.0);
            Propeller_Rpm_Value :=
              (Angle_Types.Rpm(30000.0),Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0));
            Aircraft_Angular_Rate :=
              (Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0));

            -- Method Update
            Geometry.Update
              (An_Instance            => The_Propeller_Installation_Geometry,
               Axial_Thrust           => Force_Tensor_Value,
               Axial_Torque           => Moment_Tensor_Value,
               Inertia                => Inertia,
               Rpm                    => Propeller_Rpm_Value,
               Aircraft_Angular_Rate  => Aircraft_Angular_Rate,
               Aircraft_Cg            => Aircraft_Cg);

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Init and Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (Length_Types.Feet(-35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(-92.26/12.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Force_Tensor = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.Float_Text_Io.Put(Moment_Tensor(X));
--            Ada.Float_Text_Io.Put(Moment_Tensor(Y));
--            Ada.Float_Text_Io.Put(Moment_Tensor(Z));
--            Ada.Text_Io.New_Line;
            Test_Result :=
              (((Moment_Tensor(X) + (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)))
                - (34.8995))/34.8995 < 0.001)     and
              (((Moment_Tensor(Y) + (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)))
               - (-999.391))/(-999.391) < 0.001) and
              (((Moment_Tensor(Z) + (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y))))
               < 0.00001);
            Test_Init :=  Test_Init and Test_Result;

          -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;


            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);


          end test_7;

      test_8:
            -------------------------------------------
          -- Test Case 8
          -- Test the Update function
          -- Inputs           : A Geometry Class
          --                  : An aircraft cg position of 0.0.
          --                  :
          --                  : A zero Force vector.
          --                  : An rotational offset defined in the
          --                  : initialize procedure for force and moment.
          --                  : A translation offset defined in the
          --                  : initialize procedure for the arbiitrary force and moment.
          --                  : Moment Tensor with Z component only.
          -- Expected Results : A query of all accessor functions should
          --                    show that all of the initialized instance
          --                    variables are set to the correct values
          --                    thus, testing the initialize and accessor
          --                    functions.  Then the update function
          --                    is called producing an equivalent force
          --                    and moment acting at the cg of the aircraft
          --                    from an arbitrarily oriented force and moment
          --                    at an arbitrary point.
          -- Purpose          : Test the initialize and accessor functions.
          -- Interface Tested : Geometry.Initialize
          --                    Geometry.Thrust_Reference_Point
          --                    Geometry.Rotational_Orientation
          --                    Geometry.Moment_Arm
          --                    Geometry.Moment_Tensor
          --                    Geometry.Force_Tensor
          --                    Geometry.Gyroscopic_Moments_Tensor
          --                    Geometry.Update;

          declare
            Test_Number                             : Integer := 8;
            Test_Result                             : Boolean;
            Test_Init                               : Boolean;
            The_Propeller_Installation_Geometry     : Geometry.Instance;
            Thrust_Reference_Point                  : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Propeller_Rotational_Orientation        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
            Moment_Arm                              : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor                           : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
            Gyroscopic_Moment_Tensor                : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Cg                             : Jpats_Propeller_Types.Coordinate_Type.Instance;
            Moment_Tensor_Value                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Force_Tensor_Value                      : Jpats_Propeller_Types.Force_Vector_Type.Instance;
--          Gyroscopic_Moment_Tensor_Value          : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
            Aircraft_Angular_Rate                   : Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
            Inertia                                 : Mass_Types.Slugs_Feet_Sq;
            Propeller_Rpm_Value                     : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
          begin

            -- initialize the instance called The_Propeller_Installation_Geometry
            Geometry.Initialize
              (An_Instance => The_Propeller_Installation_Geometry);

            -- use the accessor function to check the initialize function

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Force_Tensor  = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;

            Force_Tensor_Value :=
              (Force_Types.Lbf(0.0),Force_Types.Lbf(0.0),Force_Types.Lbf(0.0));
            Moment_Tensor_Value :=
              (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(1000.0));
            Aircraft_Cg :=
              (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
            Inertia := Mass_Types.Slugs_Feet_Sq(1.0);
            Propeller_Rpm_Value :=
              (Angle_Types.Rpm(30000.0),Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0));
            Aircraft_Angular_Rate :=
              (Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0),
               Angle_Types.Radians_Per_Sec(0.0));

            -- Method Update
            Geometry.Update
              (An_Instance            => The_Propeller_Installation_Geometry,
               Axial_Thrust           => Force_Tensor_Value,
               Axial_Torque           => Moment_Tensor_Value,
               Inertia                => Inertia,
               Rpm                    => Propeller_Rpm_Value,
               Aircraft_Angular_Rate  => Aircraft_Angular_Rate,
               Aircraft_Cg            => Aircraft_Cg);

            -- Access Thrust_Reference_Point
            Thrust_Reference_Point := Geometry.Thrust_Reference_Point
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=  Thrust_Reference_Point = (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
            Test_Init := Test_Init and Test_Result;

            -- Access Rotational_Orientation
            Propeller_Rotational_Orientation := Geometry.Rotational_Orientation
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result :=   Propeller_Rotational_Orientation = (Angle_Types.Degrees_To_Radians(0.0),Angle_Types.Degrees_To_Radians(-1.5),Angle_Types.Degrees_To_Radians(2.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Arm
            Moment_Arm :=  Geometry.Moment_Arm
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Moment_Arm  = (Length_Types.Feet(-35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(-92.26/12.0));
            Test_Init :=  Test_Init and Test_Result;

            -- Access Force_Tensor
            Force_Tensor := Geometry.Force_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Force_Tensor = (0.0,0.0,0.0);
--              (abs((Force_Tensor(X) - (1000.0* Sin(0.0)*Sin(-1.5)))) < 0.000001) and
--              (abs((Force_Tensor(Y) - (1000.0*(Cos(0.0)*Sin(-2.0)*Sin(-1.5) - Sin(-1.5)*Sin(-1.5))))/
--                  (1000.0*(Cos(0.0)*Sin(-2.0)*Sin(-1.5) - Sin(-1.5)*Sin(-1.5)))) < 0.000001) and
--              (abs((Force_Tensor(Z) -(1000.0* Cos(0.0)*Cos(-2.0)))/
--                   (1000.0* Cos(0.0)*Cos(-2.0))) < 0.000001);
            Test_Init :=  Test_Init and Test_Result;

            -- Access Moment_Tensor
            Moment_Tensor := Geometry.Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
--            Ada.Float_Text_Io.Put(Moment_Tensor(X));
--            Ada.Float_Text_Io.Put(Moment_Tensor(Y));
--            Ada.Float_Text_Io.Put(Moment_Tensor(Z));
--            Ada.Text_Io.New_Line;
            Test_Result :=
              ((Moment_Tensor(X) + (Force_Tensor(Z)*Moment_Arm(Y) - Force_Tensor(Y)*Moment_Arm(Z)))
               < 0.00001)     and
              ((Moment_Tensor(Y) + (Force_Tensor(X)*Moment_Arm(Z) - Force_Tensor(Z)*Moment_Arm(X)))
               - (0.913562)/0.913562 < 0.00001) and
              ((Moment_Tensor(Z) + (Force_Tensor(Y)*Moment_Arm(X) - Force_Tensor(X)*Moment_Arm(Y)))
               - (-999.657)/(-999.657) < 0.00001);
            Test_Init :=  Test_Init and Test_Result;

          -- Access Gyroscopic_Moment_Tensor
            Gyroscopic_Moment_Tensor := Geometry.Gyroscopic_Moment_Tensor
              (An_Instance => The_Propeller_Installation_Geometry);
            Test_Result := Gyroscopic_Moment_Tensor   = (0.0,0.0,0.0);
            Test_Init :=  Test_Init and Test_Result;


            -- Report the result to the test result file.
            Class_Test_Result_File.Report_Case_Status (Success => Test_Init, Number => Test_Number);


         end test_8;

          Ada.Text_Io.Put_Line ("Geometry_Ctd.exe is finished.");

end Geometry_Ctd;







