-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Float_Text_Io;
with Ada.Text_Io;
with Pratt_Whitney_Pt6a_68_Gas_Generator;
with Torque_Types;
with Temperature_Types;
with Force_Types;
with Normalized_Types;
with Length_Types;
with Mass_Types;
with Angle_Types;
with Jpats_Propeller_Types;
with Class_Test_Result_File;

procedure Pratt_Whitney_Pt6a_68_Gas_Generator_Ctd is
  Fuel_Flow : Mass_Types.Pph;
begin

Test_1 :
      -------------------------------------------
      -- Test Case 1
      -- Test the Initialize function
      -- Inputs           : A Valid Pratt_Whitney_Pt6a_68_Gas_Generator Class
      -- Expected Results : A query of all accessor functions should
      --                    show that all of the initialized instance
      --                    variables are set to the correct values
      --                    thus, testing the initialize and accessor
      --                    functions.
      -- Purpose          : Test the initialize and accessor functions.
      -- Interface Tested : Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Power_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Internal_Turbine_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Exhaust_Thrust
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
      --

      declare
        The_Gas_Gen       : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        Torque            : Torque_Types.Ft_Lbf;
        Bleed_Temp        : Temperature_Types.Celsius;
        Bleed_Press       : Force_Types.Psi;
        Test_Result       : Boolean;
        Test_Number       : Integer := 1;
        Init_Result       : Boolean;
        Gas_Gen_Rpm       : Normalized_Types.Normalize;
        Power_Turbine_Rpm : Angle_Types.Rpm;
        Itt               : Temperature_Types.Celsius;
        Fn                : Force_Types.Lbf;
      begin
        Pratt_Whitney_Pt6a_68_Gas_Generator.Read_Tables;

        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => The_Gas_Gen);

        -- Use the accessor function to check the initialize function

        -- Access Gas_Generator_Rpm
        Gas_Gen_Rpm := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          (An_Instance => The_Gas_Gen);
        --  Ada.Float_Text_Io.Put (Gas_Gen_Rpm);
        Test_Result := Gas_Gen_Rpm - Normalized_Types.Normalize(0.0) < 0.000001;
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Test_Result;

        -- Access Power_Turbine_Rpm
        Power_Turbine_Rpm := Pratt_Whitney_Pt6a_68_Gas_Generator.Power_Turbine_Rpm
          (An_Instance => The_Gas_Gen);
        --  Ada.Float_Text_Io.Put (Power_Turbine_Rpm);
        Test_Result := Power_Turbine_Rpm - Angle_Types.Rpm(0.0) < 0.000001;
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Torque_Sensor
        Torque := Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
          ( An_Instance => The_Gas_Gen);
        Test_Result :=  Torque = Torque_Types.Ft_lbf(0.0);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Internal Turbine Temperature
        ITT := Pratt_Whitney_Pt6a_68_Gas_Generator.Internal_Turbine_Temperature_Sensor
          ( An_Instance => The_Gas_Gen);
        Test_Result :=  (ITT - Temperature_Types.Celsius(15.0))/Temperature_Types.Celsius(15.0) < 0.0001;
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Exhaust_Thrust
        Fn := Pratt_Whitney_Pt6a_68_Gas_Generator.Exhaust_Thrust
          ( An_Instance => The_Gas_Gen);
        Test_Result :=  Fn - Force_Types.Lbf(0.0) < 0.000001;
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Bleed Pressure
        Bleed_Press := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
          (An_Instance => The_Gas_Gen);
        Test_Result :=  Bleed_Press = Force_Types.Psi(14.7);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Bleed Temperature
        Bleed_Temp := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
          (An_Instance => The_Gas_Gen);
        Test_Result :=  Bleed_Temp = Temperature_Types.Celsius(15.0);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;
        Class_Test_Result_File.Report_Case_Status
          (Success => Init_Result, Number => Test_Number);

      end Test_1;

  Test_2:
      -------------------------------------------
      -- Test Case 2
      -- Test the Update function
      -- Inputs           : A Valid Pratt_Whitney_Pt6a_68_Gas_Generator Class
      -- Expected Results : A query of all accessor functions should
      --                    show that all of the initialized instance
      --                    variables are set to the correct values
      --                    thus, testing the initialize and accessor
      --                    functions.
      -- Purpose          : Test the update procedure for a standard day
      --                    on the ground at sea level and shutdown.
      -- Interface Tested : Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Power_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Internal_Turbine_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Exhaust_Thrust
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Update
      --
      --------------------------------------------

      declare
        Next_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        This_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        Torque       : Torque_Types.Ft_Lbf;
        Bleed_Temp   : Temperature_Types.Celsius;
        Bleed_Press  : Force_Types.Psi;
        Test_Result  : Boolean;
        Test_Number  : Integer := 2;
        Init_Result  : Boolean;
        Iconst       : Float;
        Propeller_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Accessory_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Air_Conditioner_Torque_Absorbed  : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Result       : Normalized_Types.Normalize;
      begin
        Pratt_Whitney_Pt6a_68_Gas_Generator.Read_Tables;

        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => Next_Gas_Gen);
        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => This_Gas_Gen);

        -- Use the accessor function to check the initialize function

        -- Access Torque_Sensor
        Torque := Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
          ( An_Instance => Next_Gas_Gen);
        Test_Result :=  Torque = Torque_Types.Ft_lbf(0.0);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Test_Result;

        -- Access Bleed Pressure
        Bleed_Press := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Press = Force_Types.Psi(14.7);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Bleed Temperature
        Bleed_Temp := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Temp = Temperature_Types.Celsius(15.0);
        --    Class_Test_Result_File.Report_Case_Status
        --      (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        Propeller_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
        Accessory_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
        Air_Conditioner_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));

        Iconst := 1.0/15.0;
        Pratt_Whitney_Pt6a_68_Gas_Generator.Update
          (Next_Instance                   => Next_Gas_Gen,
           This_Instance                   => This_Gas_Gen,
           Iconst                          => Iconst,
           Ambient_Temperature_Ratio       => Normalized_Types.Normalize(1.0),
           Ambient_Pressure_Ratio          => Normalized_Types.Normalize(1.0),
           Mach_Number                     => Length_Types.Mach(0.0),
           Pressure_Altitude               => Length_Types.Feet(0.0),
           Fuel_Flow_Rate                  => Mass_Types.Pph(0.0), -- lbm/hr
           Bleed_Flow_Rate                 => Mass_Types.Ppm(0.0), -- lbm/min
           Propeller_Torque_Absorbed       => Propeller_Torque_Absorbed,
           Accessory_Torque_Absorbed       => Accessory_Torque_Absorbed,
           Air_Conditioner_Torque_Absorbed => Air_Conditioner_Torque_Absorbed,
           Accessory_Horsepower            => Torque_Types.Horsepower(0.0),
           Ignition_Exciter                => False,
           Hot_Start                       => False,
           No_Light_Request                => False);

        Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          (An_Instance => Next_Gas_Gen);
--      Ada.Text_Io.Put("Test_2");
--      Ada.Float_Text_Io.Put (Result);
        Test_Result := (Result)/100.0 < 1.0;  -- Less than 1 %
        Class_Test_Result_File.Report_Case_Status
          (Success => Init_Result and Test_Result, Number => Test_Number);

      end Test_2;

  Test_3:
      -------------------------------------------
      -- Test Case 3
      -- Test the Update function for Dry Motor Case
      -- Inputs           : A Valid Pratt_Whitney_Pt6a_68_Gas_Generator Class
      -- Expected Results : A query of all accessor functions should
      --                    show that all of the initialized instance
      --                    variables are set to the correct values
      --                    thus, testing the initialize and accessor
      --                    functions.  With any starter torque applied
      --                    to the accessory gear box in the form of a negative
      --                    accessory torque, the starter is engaged and
      --                    under these condition the gas generator speed
      --                    should reach a steady state value of 18 percent
      --                    within 6 or more seconds.
      -- Purpose          : Test the update procedure for a standard day
      --                    on the ground at sea level and dry motoring.
      -- Interface Tested : Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Power_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Internal_Turbine_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Exhaust_Thrust
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Update
      --
      --------------------------------------------

      declare
        Next_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        This_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        Torque       : Torque_Types.Ft_Lbf;
        Bleed_Temp   : Temperature_Types.Celsius;
        Bleed_Press  : Force_Types.Psi;
        Test_Result  : Boolean;
        Test_Number  : Integer := 3;
        Init_Result  : Boolean;
        Iconst       : Float;
        Propeller_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Accessory_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Air_Conditioner_Torque_Absorbed  : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Result       : Normalized_Types.Normalize;
        Time         : Float;
      begin

        Pratt_Whitney_Pt6a_68_Gas_Generator.Read_Tables;
        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => Next_Gas_Gen);
        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => This_Gas_Gen);

        -- Use the accessor function to check the initialize function

        -- Access Torque_Sensor
        Torque := Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
          ( An_Instance => Next_Gas_Gen);
        Test_Result :=  Torque = Torque_Types.Ft_lbf(0.0);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Test_Result;

        -- Access Bleed Pressure
        Bleed_Press := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Press = Force_Types.Psi(14.7);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Bleed Temperature
        Bleed_Temp := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Temp = Temperature_Types.Celsius(15.0);
        --    Class_Test_Result_File.Report_Case_Status
        --      (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;
        Iconst := 1.0/15.0;
        Time := 0.0;
        for I in 1..Integer(8.0/Iconst) loop
          Propeller_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          Accessory_Torque_Absorbed := (Torque_Types.Ft_Lbf(-80.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          Air_Conditioner_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));

          Time := Time + Iconst;
          Pratt_Whitney_Pt6a_68_Gas_Generator.Update
            (Next_Instance                   => Next_Gas_Gen,
             This_Instance                   => This_Gas_Gen,
             Iconst                          => Iconst,
             Ambient_Temperature_Ratio       => Normalized_Types.Normalize(1.0),
             Ambient_Pressure_Ratio          => Normalized_Types.Normalize(1.0),
             Mach_Number                     => Length_Types.Mach(0.0),
             Pressure_Altitude               => Length_Types.Feet(0.0),
             Fuel_Flow_Rate                  => Mass_Types.Pph(65.0), -- minimum lbm/hr
             Bleed_Flow_Rate                 => Mass_Types.Ppm(0.0), -- lbm/min
             Propeller_Torque_Absorbed       => Propeller_Torque_Absorbed,
             Accessory_Torque_Absorbed       => Accessory_Torque_Absorbed,
             Air_Conditioner_Torque_Absorbed => Air_Conditioner_Torque_Absorbed,
             Accessory_Horsepower            => Torque_Types.Horsepower(0.0),
             Ignition_Exciter                => false,
             Hot_Start                       => False,
             No_Light_Request                => False);

          --        Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          --          (An_Instance => Next_Gas_Gen);
          --        Ada.Float_Text_Io.Put (Time);
          --        Ada.Text_Io.Put ("  " );
          --        Ada.Float_Text_Io.Put (Result );
          --        Ada.Text_Io.New_Line;

          Time := Time + Iconst;
          Pratt_Whitney_Pt6a_68_Gas_Generator.Update
            (Next_Instance                   => This_Gas_Gen,
             This_Instance                   => Next_Gas_Gen,
             Iconst                          => Iconst,
             Ambient_Temperature_Ratio       => Normalized_Types.Normalize(1.0),
             Ambient_Pressure_Ratio          => Normalized_Types.Normalize(1.0),
             Mach_Number                     => Length_Types.Mach(0.0),
             Pressure_Altitude               => Length_Types.Feet(0.0),
             Fuel_Flow_Rate                  => Mass_Types.Pph(65.0), -- minimum lbm/hr
             Bleed_Flow_Rate                 => Mass_Types.Ppm(0.0), -- lbm/min
             Propeller_Torque_Absorbed       => Propeller_Torque_Absorbed,
             Accessory_Torque_Absorbed       => Accessory_Torque_Absorbed,
             Air_Conditioner_Torque_Absorbed => Air_Conditioner_Torque_Absorbed,
             Accessory_Horsepower            => Torque_Types.Horsepower(0.0),
             Ignition_Exciter                => false,
             Hot_Start                       => False,
             No_Light_Request                => False);
        end loop;

        Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          (An_Instance => Next_Gas_Gen);
--        Ada.Float_Text_Io.Put (Time);
--        Ada.Text_Io.Put ("  " );
--        Ada.Float_Text_Io.Put (Result );
--        Ada.Text_Io.New_Line;
        Test_Result := abs(Result -18.0)/18.0 < 0.01;
        Class_Test_Result_File.Report_Case_Status
          (Success => Init_Result and Test_Result, Number => Test_Number);

      end Test_3;

  Test_4:
      -------------------------------------------
      -- Test Case 4
      -- Test the Update function for Full Starting Case
      -- Inputs           : A Valid Pratt_Whitney_Pt6a_68_Gas_Generator Class
      -- Expected Results : A query of all accessor functions should
      --                    show that all of the initialized instance
      --                    variables are set to the correct values
      --                    thus, testing the initialize and accessor
      --                    functions.  With any starter torque applied
      --                    to the accessory gear box in the form of a negative
      --                    accessory torque, the starter is engaged and
      --                    under these condition the gas generator speed
      --                    should reach a steady state value of 18 percent
      --                    within 6 or more seconds.
      -- Purpose          : Test the update procedure for a standard day
      --                    on the ground at sea level and dry motoring.
      -- Interface Tested : Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Power_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Internal_Turbine_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Exhaust_Thrust
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Update
      --
      --------------------------------------------

      declare
        Next_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        This_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        Torque       : Torque_Types.Ft_Lbf;
        Bleed_Temp   : Temperature_Types.Celsius;
        Bleed_Press  : Force_Types.Psi;
        Test_Result  : Boolean;
        Test_Number  : Integer := 4;
        Init_Result  : Boolean;
        Iconst       : Float;
        Propeller_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Accessory_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Air_Conditioner_Torque_Absorbed  : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Result       : Normalized_Types.Normalize;
        Time         : Float;
        Ignition_Exciter : Boolean;
      begin

        Pratt_Whitney_Pt6a_68_Gas_Generator.Read_Tables;
        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => Next_Gas_Gen);
        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => This_Gas_Gen);

        -- Use the accessor function to check the initialize function

        -- Access Torque_Sensor
        Torque := Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
          ( An_Instance => Next_Gas_Gen);
        Test_Result :=  Torque = Torque_Types.Ft_lbf(0.0);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Test_Result;

        -- Access Bleed Pressure
        Bleed_Press := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Press = Force_Types.Psi(14.7);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Bleed Temperature
        Bleed_Temp := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Temp = Temperature_Types.Celsius(15.0);
        --    Class_Test_Result_File.Report_Case_Status
        --      (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;
        Iconst := 1.0/15.0;
        Time := 0.0;
        for I in 1..Integer(1500.0/Iconst) loop
          Propeller_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          Air_Conditioner_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));

          Time := Time + Iconst;


          Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
            (An_Instance => Next_Gas_Gen);
          if Result > 16.5 then
            Ignition_Exciter := True;
            Fuel_Flow := 650.0;
          else
            Ignition_Exciter := False;
          end if;

          if Result > 22.0 then
            Accessory_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          else
            Accessory_Torque_Absorbed := (Torque_Types.Ft_Lbf(-80.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          end if;

          Pratt_Whitney_Pt6a_68_Gas_Generator.Update
            (Next_Instance                   => Next_Gas_Gen,
             This_Instance                   => This_Gas_Gen,
             Iconst                          => Iconst,
             Ambient_Temperature_Ratio       => Normalized_Types.Normalize(1.0),
             Ambient_Pressure_Ratio          => Normalized_Types.Normalize(1.0),
             Mach_Number                     => Length_Types.Mach(0.0),
             Pressure_Altitude               => Length_Types.Feet(0.0),
             Fuel_Flow_Rate                  => Fuel_Flow, -- minimum lbm/hr
             Bleed_Flow_Rate                 => Mass_Types.Ppm(0.0), -- lbm/min
             Propeller_Torque_Absorbed       => Propeller_Torque_Absorbed,
             Accessory_Torque_Absorbed       => Accessory_Torque_Absorbed,
             Air_Conditioner_Torque_Absorbed => Air_Conditioner_Torque_Absorbed,
             Accessory_Horsepower            => Torque_Types.Horsepower(0.0),
             Ignition_Exciter                => Ignition_Exciter,
             Hot_Start                       => False,
             No_Light_Request                => False);

          --        Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          --        (An_Instance => Next_Gas_Gen);
          --        Ada.Float_Text_Io.Put (Time);
          --        Ada.Text_Io.Put ("  " );
          --        Ada.Float_Text_Io.Put (Result );
          --        Ada.Text_Io.New_Line;

          Time := Time + Iconst;
          Pratt_Whitney_Pt6a_68_Gas_Generator.Update
            (Next_Instance                   => This_Gas_Gen,
             This_Instance                   => Next_Gas_Gen,
             Iconst                          => Iconst,
             Ambient_Temperature_Ratio       => Normalized_Types.Normalize(1.0),
             Ambient_Pressure_Ratio          => Normalized_Types.Normalize(1.0),
             Mach_Number                     => Length_Types.Mach(0.0),
             Pressure_Altitude               => Length_Types.Feet(0.0),
             Fuel_Flow_Rate                  => Fuel_Flow, -- minimum lbm/hr
             Bleed_Flow_Rate                 => Mass_Types.Ppm(0.0), -- lbm/min
             Propeller_Torque_Absorbed       => Propeller_Torque_Absorbed,
             Accessory_Torque_Absorbed       => Accessory_Torque_Absorbed,
             Air_Conditioner_Torque_Absorbed => Air_Conditioner_Torque_Absorbed,
             Accessory_Horsepower            => Torque_Types.Horsepower(0.0),
             Ignition_Exciter                => Ignition_Exciter,
             Hot_Start                       => False,
             No_Light_Request                => False);

        end loop;
        Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          (An_Instance => Next_Gas_Gen);
--        Ada.Float_Text_Io.Put (Time);
--        Ada.Text_Io.Put ("  " );
--        Ada.Float_Text_Io.Put (Result );
--        Ada.Text_Io.New_Line;
        Test_Result := abs(Result - 97.5522)/97.5522 < 0.00001;
        Class_Test_Result_File.Report_Case_Status
          (Success => Init_Result and Test_Result, Number => Test_Number);

      end Test_4;

  Test_5:
      -------------------------------------------
      -- Test Case 5
      -- Test the Update function for Full Starting Case
      -- Inputs           : A Valid Pratt_Whitney_Pt6a_68_Gas_Generator Class
      -- Expected Results : A query of all accessor functions should
      --                    show that all of the initialized instance
      --                    variables are set to the correct values
      --                    thus, testing the initialize and accessor
      --                    functions.  With any starter torque applied
      --                    to the accessory gear box in the form of a negative
      --                    accessory torque, the starter is engaged and
      --                    under these condition the gas generator speed
      --                    should reach a steady state value of 18 percent
      --                    within 6 or more seconds.
      -- Purpose          : Test the update procedure for a standard day
      --                    on the ground at sea level and dry motoring.
      -- Interface Tested : Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Power_Turbine_Rpm
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Internal_Turbine_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Exhaust_Thrust
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
      --                    Pratt_Whitney_Pt6a_68_Gas_Generator.Update
      --
      --------------------------------------------

      declare
        Next_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        This_Gas_Gen : Pratt_Whitney_Pt6a_68_Gas_Generator.Instance;
        Torque       : Torque_Types.Ft_Lbf;
        Bleed_Temp   : Temperature_Types.Celsius;
        Bleed_Press  : Force_Types.Psi;
        Test_Result  : Boolean;
        Test_Number  : Integer := 5;
        Init_Result  : Boolean;
        Iconst       : Float;
        Propeller_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Accessory_Torque_Absorbed        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Air_Conditioner_Torque_Absorbed  : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
        Result       : Normalized_Types.Normalize;
        Time         : Float;
        Ignition_Exciter : Boolean;
      begin

        Pratt_Whitney_Pt6a_68_Gas_Generator.Read_Tables;
        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => Next_Gas_Gen);
        Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
          (An_Instance => This_Gas_Gen);

        -- Use the accessor function to check the initialize function

        -- Access Torque_Sensor
        Torque := Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
          ( An_Instance => Next_Gas_Gen);
        Test_Result :=  Torque = Torque_Types.Ft_lbf(0.0);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Test_Result;

        -- Access Bleed Pressure
        Bleed_Press := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Pressure
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Press = Force_Types.Psi(14.7);
        --  Class_Test_Result_File.Report_Case_Status
        --    (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;

        -- Access Bleed Temperature
        Bleed_Temp := Pratt_Whitney_Pt6a_68_Gas_Generator.Bleed_Temperature
          (An_Instance => Next_Gas_Gen);
        Test_Result :=  Bleed_Temp = Temperature_Types.Celsius(15.0);
        --    Class_Test_Result_File.Report_Case_Status
        --      (Success => Test_Result, Number => Test_Number);
        Init_Result := Init_Result and Test_Result;


        Iconst := 1.0/15.0;
        Time := 0.0;
        for I in 1..Integer(1500.0/Iconst) loop
          Propeller_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          Air_Conditioner_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));

          Time := Time + Iconst;

          -- Ignition driver
          Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
            (An_Instance => Next_Gas_Gen);
          if Result > 16.5 then
            Ignition_Exciter := True;
          else
            Ignition_Exciter := False;
          end if;
          -- Fuel Flow driver
          if Result > 16.5 then
            if Result > 70.0 then
              Fuel_Flow := 65.0;
            else
              Fuel_Flow := 650.0;
            end if;
          else
            Fuel_Flow := 0.0;
          end if;

          -- Starter Driver
          if Result > 22.0 then
            Accessory_Torque_Absorbed := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          else
            Accessory_Torque_Absorbed := (Torque_Types.Ft_Lbf(-80.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
          end if;

          Pratt_Whitney_Pt6a_68_Gas_Generator.Update
            (Next_Instance                   => Next_Gas_Gen,
             This_Instance                   => This_Gas_Gen,
             Iconst                          => Iconst,
             Ambient_Temperature_Ratio       => Normalized_Types.Normalize(1.0),
             Ambient_Pressure_Ratio          => Normalized_Types.Normalize(1.0),
             Mach_Number                     => Length_Types.Mach(0.0),
             Pressure_Altitude               => Length_Types.Feet(0.0),
             Fuel_Flow_Rate                  => Fuel_Flow, -- minimum lbm/hr
             Bleed_Flow_Rate                 => Mass_Types.Ppm(0.0), -- lbm/min
             Propeller_Torque_Absorbed       => Propeller_Torque_Absorbed,
             Accessory_Torque_Absorbed       => Accessory_Torque_Absorbed,
             Air_Conditioner_Torque_Absorbed => Air_Conditioner_Torque_Absorbed,
             Accessory_Horsepower            => Torque_Types.Horsepower(0.0),
             Ignition_Exciter                => Ignition_Exciter,
             Hot_Start                       => False,
             No_Light_Request                => False);

          --        Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          --        (An_Instance => Next_Gas_Gen);
          --        Ada.Float_Text_Io.Put (Time);
          --        Ada.Text_Io.Put ("  " );
          --        Ada.Float_Text_Io.Put (Result );
          --        Ada.Text_Io.New_Line;

          Time := Time + Iconst;
          Pratt_Whitney_Pt6a_68_Gas_Generator.Update
            (Next_Instance                   => This_Gas_Gen,
             This_Instance                   => Next_Gas_Gen,
             Iconst                          => Iconst,
             Ambient_Temperature_Ratio       => Normalized_Types.Normalize(1.0),
             Ambient_Pressure_Ratio          => Normalized_Types.Normalize(1.0),
             Mach_Number                     => Length_Types.Mach(0.0),
             Pressure_Altitude               => Length_Types.Feet(0.0),
             Fuel_Flow_Rate                  => Fuel_Flow, -- minimum lbm/hr
             Bleed_Flow_Rate                 => Mass_Types.Ppm(0.0), -- lbm/min
             Propeller_Torque_Absorbed       => Propeller_Torque_Absorbed,
             Accessory_Torque_Absorbed       => Accessory_Torque_Absorbed,
             Air_Conditioner_Torque_Absorbed => Air_Conditioner_Torque_Absorbed,
             Accessory_Horsepower            => Torque_Types.Horsepower(0.0),
             Ignition_Exciter                => Ignition_Exciter,
             Hot_Start                       => False,
             No_Light_Request                => False);

        end loop;
        Result := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
          (An_Instance => Next_Gas_Gen);
--        Ada.Float_Text_Io.Put (Time);
--        Ada.Text_Io.Put ("  " );
--        Ada.Float_Text_Io.Put (Result );
--        Ada.Text_Io.New_Line;
        Test_Result := abs(Result - 71.9633)/71.9633 < 0.00001;
        Class_Test_Result_File.Report_Case_Status
          (Success => Init_Result and Test_Result, Number => Test_Number);

      end Test_5;


end Pratt_Whitney_Pt6a_68_Gas_Generator_Ctd;




