-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Gear_Pump;
with Force_Types;
with Ada.Text_Io;
with Angle_Types;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;
with Class_Test_Result_File;

procedure Gear_Pump_Ctd is

   -- This class test driver should instantiate a representative pump package with
   -- appropriate data tables for differential pressure and flow rates.  An object
   -- of the type defined by the package which includes the tables is then declared
   -- in the declare block for each test performed.

   package High_Pressure_Pump is new Gear_Pump
     ( Pump_Pressure_Differential_Table           => "HPDIFF.ito",
       Pump_Flow_Rate_Table                       => "HPFLOW.ito",
       Bypass_Pressure_Regulating_Valve_Set_Point => Force_Types.Psi(50.0));

begin
   Ada.Text_Io.Put_Line ( "Testing gear_pump class ...");
   Ada.Text_Io.New_line;
   ----------------------------------------------------------------------------------------
   -- Test 1
   -- Purpose        : Test the initialize and accessor functions.
   -- Range of Input : There is no range of conditions to test for initialize.
   -- Expected Result: The initialize function should return 0.0 for the Output pressure.
   --                  The output pressure accessor should retreive a force_type.psi value
   --                  of zero.  The compare in the if structure should result in a passing
   --                  test result.
   --
   declare
     Test_Number : Integer := 1;
     Test_Result : Boolean;
     Pump : High_Pressure_Pump.Instance;
   begin

      -- Test initialization function.  Use accessor function to test both
      -- initialization and accessor function.  Since the pump initializes to
      -- zero.  Initialize function is only further corroborated by using the
      -- update function and the accessor function to verify non-zero values for
      -- the output pressure.

      High_Pressure_Pump.Read_Tables;
      High_Pressure_Pump.Initialize (An_Instance => Pump);

      -- Report results to screen.
      Test_Result := High_Pressure_Pump.Outlet_Pressure
        (An_Instance => Pump) = Force_Types.Psi(0.0);
      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
   end;
   --
   --
   -----------------------------------------------------------------------------------------
   -- Test 2
   -- Purpose        : Test the update method.  Check for proper static
   --                  operation of class.  This is the normal flow through the domain
   --                  class. Supply input pressure and rpm to calculate a new state.  The
   --                  state of output pressure is determined by using the accessor function
   --                  for output pressure.  A passing result corroborates the update and
   --                  the accessor function.
   -- Range of Input : Three conditions need to be tested for Inlet Pressure and three
   --                  conditions need to be tested for Rpm.  Those are a subset of normal
   --                  values, out-of-range values, and boundary conditions.  This test
   --                  uses data that is within the set of normal values.
   -- Expected Result: For the subset of normal values for both inlet pressure and rpm,
   --                  the inlet pressure and the pressure across the pump based on rpm
   --                  should be added together.
   --
   --
   declare
     Test_Number : Integer := 2;
     Test_Result : Boolean;
     Pump : High_Pressure_Pump.Instance;
   begin
      -- Read_Tables
      High_Pressure_Pump.Read_Tables;
      -- Initialize pump
      High_Pressure_Pump.Initialize (An_Instance => Pump);

      High_Pressure_Pump.Update (An_Instance    => Pump,
                                 Inlet_Pressure => Force_Types.Psi(20.0),
                                 Rpm            => Angle_Types.Rpm(1430.0) );

      -- Report the result to the screen.

    Test_Result :=  High_Pressure_Pump.Outlet_Pressure
      (An_Instance => Pump) = Force_Types.Psi(95.0);
--    Ada.Text_Io.Put ( Force_Types.Psi'Image(High_Pressure_Pump.Outlet_Pressure(An_Instance => Pump)));
    Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
   end;
   --
--   -----------------------------------------------------------------------------------------
--   -- Test 3
--   -- Purpose        : Test the update method.  Check for proper static
--   --                  operation of class.  This is the normal flow through the domain
--   --                  class. Supply input pressure and rpm to calculate a new state.  The
--   --                  state of output pressure is determined by using the accessor function
--   --                  for output pressure.  A passing result corroborates the update and
--   --                  the accessor function.
--   -- Range of Input : Three conditions need to be tested for Inlet Pressure and three
--   --                  conditions need to be tested for Rpm.  Those are a subset of normal
--   --                  values, out-of-range values, and boundary conditions.  This test
--   --                  uses data that is out of range for the set of normal values.
--   -- Expected Result: For the subset of normal values for both inlet pressure and rpm,
--   --                  the inlet pressure and the pressure across the pump based on rpm
--   --                  should be added together.  This test should yield data at the
--   --                  extreme ranges of the table without creating an exception.
--   --
--   -----------------------------------------------------------------------------------------
--   -- Test the Top end.
--   declare
--     Test_Number : Integer := 3;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--
--      Temp :=  Float'Last;
--      Temp :=  Temp ** Temp;
--
--      -- Read_Tables
--      High_Pressure_Pump.Read_Tables;
--      -- initialize pump
--      High_Pressure_Pump.Initialize (An_Instance => Pump);
--
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(Temp),
--                                 Rpm            => Angle_Types.Rpm(1430.0) );
--      -- Manual Result Input Required.
--      Ada.Text_Io.Put_Line("Input out of range check of positive values for Inlet_Pressure passed.");
--      Test_Result := True;
--      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on one of the input values occurred.");
--   end;
--
--   -- Test the bottom end.
--   declare
--     Test_Number : Integer := 4;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--
--      Temp :=  Float'First;
--      Temp :=  Temp + Temp;
--
--      -- Read_Tables
--      High_Pressure_Pump.Read_Tables;
--      -- initialize pump
--      High_Pressure_Pump.Initialize (An_Instance => Pump);
--
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(Temp),
--                                 Rpm            => Angle_Types.Rpm(1430.0) );
--      -- Manual Result Input required;
--      Ada.Text_Io.Put_Line("Input out of range check of negative values for Inlet_Pressure passed.");
--      Test_Result := True;
--      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on one of the input values occurred.");
--   end;
--
--   -- Test the top end.
--   declare
--     Test_Number : Integer := 5;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--      Temp := Float'Last;
--      Temp :=  Temp ** Temp;
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(20.0),
--                                 Rpm            => Angle_Types.Rpm(Temp) );
--      Test_Result := True;
--      Ada.Text_Io.Put_Line("Input out of range check of positive values for Rpm passed.");
--    Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on one of the input values occurred.");
--   end;
--
--   -- Test the bottom end.
--   declare
--     Test_Number : Integer := 6;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--      Temp := Float'First;
--      Temp :=  Temp + Temp;
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(20.0),
--                                 Rpm            => Angle_Types.Rpm(Temp) );
--      -- Manual Result Input Required
--      Ada.Text_Io.Put_Line("Input out of range check of negative values for Rpm passed.");
--      Test_Result := True;
--      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--     when Constraint_Error =>
--       Ada.Text_Io.Put_Line("A constraint error on one of the input values occurred.");
--   end;
--   --
--   -----------------------------------------------------------------------------------------
--   -- Test 4
--   -- Purpose        : Test the update method.  Check for proper static
--   --                  operation of class.  This is the normal flow through the domain
--   --                  class. Supply input pressure and rpm to calculate a new state.  The
--   --                  state of output pressure is determined by using the accessor function
--   --                  for output pressure.  A passing result corroborates the update and
--   --                  the accessor function.
--   -- Range of Input : Three conditions need to be tested for Inlet Pressure and three
--   --                  conditions need to be tested for Rpm.  Those are a subset of normal
--   --                  values, out-of-range values, and boundary conditions.  This test
--   --                  uses data that is on the boundary for the set of normal values.
--   -- Expected Result: For the subset of normal values for both inlet pressure and rpm,
--   --                  the inlet pressure and the pressure across the pump based on rpm
--   --                  should be added together.  This test should yield data at the
--   --                  extreme ranges of the table without creating an exception.
--   --
--   -----------------------------------------------------------------------------------------
--   -- Test for top boundary conditions for the input pressure.
--   declare
--     Test_Number : Integer := 7;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--      Temp := Float'Last;
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(Temp),
--                                 Rpm            => Angle_Types.Rpm(1430.0) );
--      -- Manual Result Input Required
--      Ada.Text_Io.Put_Line("Input positive boundary check for Inlet Pressure passed.");
--      Test_Result :=  True;
--      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on the boundary of one of the input values occurred.");
--   end;
--
--   -- Test for bottom end boundary conditions for the input pressure.
--   declare
--     Test_Number : Integer := 8;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--      Temp := Float'First;
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(Temp),
--                                 Rpm            => Angle_Types.Rpm(1430.0) );
--      -- Manual Result Input Required
--      Ada.Text_Io.Put_Line("Input negative boundary check for Inlet Pressure passed.");
--      Test_Result :=  True;
--      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on the boundary of one of the input values occurred.");
--   end;
--
--   -- Test for top boundary conditions for the input to rpm.
--   declare
--     Test_Number : Integer := 9;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--      Temp := Float'Last;
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(20.0),
--                                 Rpm            => Angle_Types.Rpm(Temp) );
--      -- Manual Result Input Required
--      Ada.Text_Io.Put_Line("Input positive boundary check for Rpm passed.");
--      Test_Result :=  True;
--      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on the boundary of one of the input values occurred.");
--   end;
--
--   -- Test for bottom boundary conditions for the input to rpm.
--   declare
--     Test_Number : Integer := 10;
--     Test_Result : Boolean;
--     Pump : High_Pressure_Pump.Instance;
--     Temp : Float;
--   begin
--      Temp := Float'First;
--      High_Pressure_Pump.Update (An_Instance    => Pump,
--                                 Inlet_Pressure => Force_Types.Psi(20.0),
--                                 Rpm            => Angle_Types.Rpm(Temp) );
--      -- Manual Result Input Required
--      Ada.Text_Io.Put_Line("Input negative boundary check for Rpm passed.");
--      Test_Result :=  True;
--    Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on the boundary of one of the input values occurred.");
--   end;

end Gear_Pump_Ctd;












