-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Cam;
with Angle_Types;
with Class_Test_Result_File;
with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;
with Ada.Text_Io;

procedure Cam_Ctd is

   subtype New_Cam_Rate_Limit_Type is Angle_Types.Degrees_Per_Sec range -25.0.. 8.0;

   package New_Cam is new Cam
     (Degrees_Type            => Angle_Types.Degrees,
      Degrees_Per_Sec_Type    => New_Cam_Rate_Limit_Type,
      Table_File_Name         => "plapcl.ito");


begin
--   Ada.Text_Io.Put_Line ( "Testing cam class ...");
--   Ada.Text_Io.New_line;

   ----------------------------------------------------------------------------------------
   -- Test 1
   -- Purpose        : Test the initialize and accessor functions.
   -- Range of Input : There is no range of conditions to test for initialize.
   -- Expected Result: The initialize function should set the Cam position to 20.0.
   --                  The position accessor should retreive an Angle_types.degrees value
   --                  of 20.0.  The test result should record a passing test result.
   --
   --
   declare

      Test_Number : Integer := 1;
      Test_Result : Boolean;
      Next_Cam : New_Cam.Instance;
      This_Cam : New_Cam.Instance;
   begin

      New_Cam.Read_Tables;
      New_Cam.Initialize
        (This_Instance => This_Cam,
         Initial_Value => 20.0);
      New_Cam.Initialize
        (This_Instance => Next_Cam,
         Initial_Value => 20.0);

      Test_Result := New_Cam.Position(An_Instance => This_Cam) = Angle_Types.Degrees(20.0);
      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
   end;
   --
   --
   -----------------------------------------------------------------------------------------
   -- Test 2
   -- Purpose        : Test the update method.  Check for proper static
   --                  operation of class.  This is the normal flow through the domain
   --                  class. Supply a demanded position for the cam.  The
   --                  cam follower position should lag if the time rate of change in position
   --                  is greater than those limits specified by the rate limit type that the
   --                  package is declared with.
   -- Range of Input : Three conditions need to be tested for cam position demanded.
   --                  The cam is initialized to zero position. The first two values are values
   --                  that exceed the position that is attainable due to the rate limit
   --                  on the positive and negative ends of the limit.  The third is a value within
   --                  those limits.
   -- Expected Result: The cam position demanded that is beyond the positive and negative limits of
   --                  New_Cam_Rate_Limit_type'Last*Iconst and New_Cam_Rate_Limit_type'First*Iconst
   --                  should equal that value respectively.  The third value should yeild exactly the
   --                  the output value that maps to the input value from the cam table.
   --
   -- Check with rate limit ranges.
   declare
     Test_Number : Integer := 2;
     Test_Result : Boolean;
      Next_Cam : New_Cam.Instance;
      This_Cam : New_Cam.Instance;
      Time     : Float := 0.0;
      Iconst   : Float := 1.0/15.0;
      Pcl      : Angle_Types.Degrees := 0.1839;
      Position : Float;

   begin
      New_Cam.Read_Tables;
      New_Cam.Initialize
        (This_Instance => This_Cam,
         Initial_Value => -5.0);
      New_Cam.Initialize
        (This_Instance => Next_Cam,
         Initial_Value => -5.0);

      Time := Time + Iconst;
      New_Cam.Update(Next_Instance => Next_Cam,
                     This_Instance => This_Cam,
                     Iconst        => Iconst,
                     Power_Control_Lever_Angle => Pcl);
      Position := Float(New_Cam.Position (An_Instance => This_Cam));
      Test_Result :=  Position <= New_Cam_Rate_Limit_Type'Last*Iconst -5.0;
      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);

   end;

   -- Check outside the positive rate limit ranges.
   declare
     Test_Number : Integer := 3;
     Test_Result : Boolean;
      Next_Cam : New_Cam.Instance;
      This_Cam : New_Cam.Instance;
      Time     : Float := 0.0;
      Iconst   : Float := 1.0/15.0;
      Pcl      : Angle_Types.Degrees := 20.0;
      Position : Float := 0.0;

   begin
      New_Cam.Read_Tables;
      New_Cam.Initialize
        (This_Instance => This_Cam,
         Initial_Value => -5.0);
      New_Cam.Initialize
        (This_Instance => Next_Cam,
         Initial_Value => -5.0);

      Time := Time + Iconst;
      New_Cam.Update(Next_Instance => Next_Cam,
                     This_Instance => This_Cam,
                     Iconst        => Iconst,
                     Power_Control_Lever_Angle => Pcl);
      Position := Float(New_Cam.Position (An_Instance => Next_Cam));
      Test_Result := abs(Position - (New_Cam_Rate_Limit_Type'Last*Iconst -5.0))/5.0  < 0.00001;
      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
   end;

   -- Check outside the negative rate limit ranges.
   declare
     Test_Number : Integer := 4;
     Test_Result : Boolean;
      Next_Cam : New_Cam.Instance;
      This_Cam : New_Cam.Instance;
      Time     : Float := 0.0;
      Iconst   : Float := 1.0/15.0;
      Pcl      : Angle_Types.Degrees := 10.0;
      Position : Float;

   begin

      New_Cam.Read_Tables;
      New_Cam.Initialize
        (This_Instance => This_Cam,
         Initial_Value => 80.0);
      New_Cam.Initialize
        (This_Instance => Next_Cam,
         Initial_Value => 80.0);

      Time := Time + Iconst;
      New_Cam.Update(Next_Instance => Next_Cam,
                     This_Instance => This_Cam,
                     Iconst        => Iconst,
                     Power_Control_Lever_Angle => Pcl);
      Position := Float(New_Cam.Position (An_Instance => Next_Cam));
--    Ada.Text_Io.Put_Line(Float'Image(Position));
      Test_Result := abs(Position - (New_Cam_Rate_Limit_Type'First*Iconst + 80.0))/80.0 < 0.00001;
      Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);

   end;
--    --
--    -----------------------------------------------------------------------------------------
--    -- Test 5
--    -- Purpose        : Test the update method.  Check for proper static
--    --                  operation of class.  This is the normal flow through the domain
--    --                  class. Supply input pressure and rpm to calculate a new state.  The
--    --                  state of output pressure is determined by using the accessor function
--    --                  for output pressure.  A passing result corroborates the update and
--    --                  the accessor function.
--    -- Range of Input : Three conditions need to be tested for Inlet Pressure and three
--    --                  conditions need to be tested for Rpm.  Those are a subset of normal
--    --                  values, out-of-range values, and boundary conditions.  This test
--    --                  uses data that is out of range for the set of normal values.
--    -- Expected Result: For the subset of normal values for both inlet pressure and rpm,
--    --                  the inlet pressure and the pressure across the pump based on rpm
--    --                  should be added together.  This test should yield data at the
--    --                  extreme ranges of the table without creating an exception.
--    --
--    -----------------------------------------------------------------------------------------
--
--    declare
--      Test_Number : Integer := 5;
--      Test_Result : Boolean;
--
--       Next_Cam : New_Cam.Instance;
--       This_Cam : New_Cam.Instance;
--       Time     : Float := 0.0;
--       Iconst   : Float := 1.0/15.0;
--       Pcl      : Angle_Types.Degrees := 18.0;
--       Position : Float := 0.0;
--       Temp     : Float;
--    begin
--
--       Temp :=  Float'Last;
--       Temp :=  Temp ** Temp;
--
--       -- Initialize Cam
--       New_Cam.Initialize (This_Instance => Next_Cam,
--                           Initial_Value => Pcl);
--       New_Cam.Initialize (This_Instance => This_Cam,
--                           Initial_Value => Pcl);
--
--       New_Cam.Update(Next_Instance => Next_Cam,
--                      This_Instance => This_Cam,
--                      Iconst        => Iconst,
--                      Power_Control_Lever_Angle => Pcl);
--       Position := Float(New_Cam.Position (An_Instance => This_Cam));
--       Ada.Text_Io.Put(Float'Image(Position));
--       Test_Result := Position <= New_Cam_Rate_Limit_Type'Last*Iconst -5.0;
--       Ada.Text_Io.Put(Float'Image(New_Cam_Rate_Limit_Type'Last));
--       Class_Test_Result_File.Report_Case_Status (Success => Test_Result, Number => Test_Number);
--
--   exception
--      when Constraint_Error =>
--         Ada.Text_Io.Put_Line("A constraint error on one of the input values occurred.");
--   end;

end Cam_Ctd;




