-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  James F. Narrin
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Jpats_Avionics.Container;
with Jpats_Avionics.Controller;
with Jpats_Avionics.Ios_If;
with Jpats_Avionics;
with Class_Test_Result_File;
with Jpats_Simulated_Aircraft;
with Jpats_Radio;
with Jpats_Electrical;
with Jpats_Avionics_Types;

use Jpats_Avionics_Types;

with Ada.Text_Io;
use Ada.Text_Io;

procedure Jpats_Avionics.Ctd is

   package Jat renames Jpats_Avionics_Types;
   Io : Container.Io_Interface_Instance renames Container.This_Io_Interface;

   Time : Float := 0.0;
   Iconst : Float := 1.0/10.0;
   Pass : Boolean := False;

begin


   -------------------------------------------------------
   -- Test Case 1--GPS
   --
   -- Inputs:
   --   cb power: available
   --   aircraft position: 45.N, 45.0W, 5000.0 ft elev
   --   magnetic variation : 0.0
   --
   -- Expected Results:
   --   gps latitude : 45.0N
   --   gps longitude : 45.0W
   --   gps altitude : 5000.0 ft(1524.0 meters)
   --
   --
   -- Purpose:
   --   to test the GPS
   --
   JPATS_Avionics.Controller.Initialize;

   JPATS_Avionics.Controller.Update(Iconst);

   Pass :=
     abs(Io.Gps_Horizontal_Position.Latitude - 45.0) < 0.01 and
     abs(abs(Io.Gps_Horizontal_Position.Longitude) - 45.0) < 0.01 and
     abs(Io.Gps_Vertical_Position.Altitude - 1525.0) < 10.0;

   if Pass then
      Put_Line("GPS OK");
   else
      Put_Line("GPS Fail");
   end if;
   Class_Test_Result_File.Report_Case_Status(Pass);

   -------------------------------------------------------
   -- Test Case 1--AHRS
   --
   -- Inputs:
   --   cb power: available
   --   aircraft position: 45.N, 45.0W, 5000.0 ft elev
   --   magnetic heading : 270.0
   --   aircraft pitch : 30.0
   --   aircraft roll  : 30.0
   --   slaved mode
   --   rapid align : selected
   --
   -- Expected Results:
   --   ahrs pitch : 30.0
   --   ahrs roll  : 30.0
   --   ahrs heading : 270.0
   --
   -- Purpose:
   --   to test the AHRS
   --
   JPATS_Avionics.Controller.Initialize;
   Time := 0.0;

   loop

     Time := Time + Iconst;

     Jpats_Avionics.Ios_If.Set_Rapid_Align(True);
     Io.The_Ahrs_Dg_Sw := True;

     JPATS_Avionics.Controller.Update(Iconst);

     if Time > 1.0 then

        Pass :=
          abs(Io.The_Ahrs_Attitude.Pitch - 30.0) < 0.5 and
          abs(Io.The_Ahrs_Attitude.Roll - 30.0) < 0.5 and
          abs(Io.The_Ahrs_Attitude.Heading + 90.0) < 0.5;

        if Pass then
          Put_Line("AHRS OK");
        else
           Put_Line("AHRS Fail");
        end if;
        Class_Test_Result_File.Report_Case_Status(Pass);
        exit;
     end if;

   end loop;


   -------------------------------------------------------
   -- Test Case 1--Threat Control
   --
   -- Inputs:
   --
   --   aircraft position: 45.N, 45.0W, 5000.0 ft elev
   --   heading : 270.0
   --   aircraft velocity : 200 knots, heading : 270 degrees
   --   IOS Threat Control inputs:
   --     start button : on
   --     bearing      : 0.0 degrees
   --     altitude     : 5000 ft
   --     time         : 5 minutes
   --     vertical position : level
   --     speed        : 200 knots
   --     closure      : 0.0
   --
   -- Expected Results:
   --   threat lat : 45.0N
   --   threat lon : 45.39W
   --
   -- Purpose:
   --   to test the Threat Control
   --
   JPATS_Avionics.Controller.Initialize;

   Jpats_Avionics.Ios_If.Set_Nacws_Page_Start_Button((True,False));
   Jpats_Avionics.Ios_If.Set_Nacws_Page_Bearing((0.0,0.0));
   Jpats_Avionics.Ios_If.Set_Nacws_Page_Distance((0.0,0.0));
   Jpats_Avionics.Ios_If.Set_Nacws_Page_Altitude((5000.0,0.0));
   Jpats_Avionics.Ios_If.Set_Nacws_Page_Time((5.0,0.0));
   Jpats_Avionics.Ios_If.Set_Nacws_Page_Vert_Pos((Jat.Level,Jat.Level));
   Jpats_Avionics.Ios_If.Set_Nacws_Page_Speed((200.0,200.0));
   Jpats_Avionics.Ios_If.Set_Nacws_Page_Closure((0.0,0.0));

   JPATS_Avionics.Controller.Update(Iconst);


   Pass :=
     abs(Container.This_Subsystem.The_Intruder_Drive(1).Lat - 45.0)< 0.01 and
     abs(Container.This_Subsystem.The_Intruder_Drive(1).Lon + 45.39)< 0.01;

   if Pass then
      Put_Line("Threat Control OK");
   else
      Put_Line("Threat Control Fail");
   end if;
   Class_Test_Result_File.Report_Case_Status(Pass);

   -------------------------------------------------------
   -- Test Case 1--NACWS
   --
   -- Inputs:
   --
   --   aircraft position: 45.N, 45.0W, 5000.0 ft elev
   --   heading : 270.0
   --   aircraft velocity : 200 knots, heading : 270 degrees
   --   IOS Threat Control inputs:
   --   intruder position : integrate towards aircraft position
   --   adc corrected altitude : 10000 ft.
   --   NACWS Sw - On
   --
   -- Expected Results:
   --
   --
   -- Purpose:
   --   to test the NACWS
   --
   JPATS_Avionics.Controller.Initialize;
   Time := 0.0;

   Io.The_Nacws_Sw_Off := False;
   Container.This_Subsystem.The_Intruder_Drive(1).Lat := 45.0;
   Container.This_Subsystem.The_Intruder_Drive(1).Lon := -45.1;

   loop
      Time := Time + Iconst;
      Container.This_Subsystem.The_Intruder_Drive(1).Lon :=
        Container.This_Subsystem.The_Intruder_Drive(1).Lon +
        Long_Float(337.5*Iconst/364566.0/0.707);


      JPATS_Avionics.Controller.Update(Iconst);

      if Io.The_Nacws_Data.Intr_Display(1) = Jat.Ta and Time <= 300.0 then
         Put_Line("NACWS OK");
         Class_Test_Result_File.Report_Case_Status (true);
         exit;
      elsif Time > 300.0 then
         Put_Line("NACWS Fail");
         Class_Test_Result_File.Report_Case_Status (false);
         exit;
      end if;
   end loop;

end;
