-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Text_IO;
with Ada.Exceptions;
with Class_Test_Result_File;
with Recorder.Plot_Scale;
with Ada.Strings.Unbounded;
with Ada.Strings.Fixed;
with Ada.Strings;

use type Recorder.Plot_Scale.List;

-------------------------------------------------------------------------------
-- Test driver for the plot_scale package
-------------------------------------------------------------------------------
procedure Recorder.Plot_Scale_Ctd is

   Output_File : Ada.Text_Io.File_Type;
   Output_File_Name : constant String := "recorder-plot_scale_ctd.txt";

begin
   ----------------------------------------------------------------------------
   -- Test Case 1
   --
   -- Inputs           : A valid file with 2 entires
   -- Expected Results : A two-element list with plot scales matching the values
   --                    placed in the file.
   -- Purpose          : Verify that a correct plot scale file is handled
   --                    properly
   Test_Case_1 : declare

      Good  : Boolean := True;

      Output_Plot_Scale_List : constant Recorder.Plot_Scale.List :=
        (1 => ( ("TIME           ", 0.0, 20.0),
                ("AXIS           ", -200.0, 101.5)
                ),
         2 => ( ("TIME           ", 0.0, 60.0),
                ("OTHER AXIS     ", -15.78, 298.576)
                )
         );

      Input_Plot_Scale_List : Recorder.Plot_Scale.List(Output_Plot_Scale_List'Range);
      Last_Plot_Scale : Natural;
   begin
      -- Create a text file matching Output_Plot_Scale_List
      Ada.Text_IO.Create (File => Output_File,
                          Mode => Ada.Text_IO.Out_File,
                          Name => Output_File_Name);

      -- Put out a dummy "test number"
      Ada.Text_Io.Put_Line
        (File => Output_File,
         Item => "t123456"
         );

      for Index in Output_Plot_Scale_List'range loop
         Ada.Text_Io.Put_Line
           (File => Output_File,
            Item =>
              Ada.Strings.Fixed.Trim
            (Source => Output_Plot_Scale_List(Index).Y.Column_1,
             Side   => Ada.Strings.Right
             ) &
            ',' & Float'Image(Output_Plot_Scale_List(Index).Y.Column_2) &
            ',' & Float'Image(Output_Plot_Scale_List(Index).Y.Column_3) &
            ',' & Ada.Strings.Fixed.Trim
            (Source => Output_Plot_Scale_List(Index).X.Column_1,
             Side   => Ada.Strings.Right
             ) &
            ',' & Integer'Image(Integer(Output_Plot_Scale_List(Index).X.Column_2)) &
            ',' & Integer'Image(Integer(Output_Plot_Scale_List(Index).X.Column_3))
            );
      end loop;
      Ada.Text_Io.Close (Output_File);

      Ada.Text_Io.Put_Line ("Testing valid scale file...");

      Recorder.Plot_Scale.Read_List
        (File_Name  => Output_File_Name,
         Scale_List => Input_Plot_Scale_List,
         Last       => Last_Plot_Scale
         );

      -- Make sure we read the expeted number of plot scales
      if Last_Plot_Scale /= Output_Plot_Scale_List'Last then
         Good := False;
         Ada.Text_Io.Put_Line
           ("Error: expected" &
            Integer'Image(Output_Plot_Scale_List'Last) &
            " items, read" &
            Integer'Image(Last_Plot_Scale)
            );
      end if;

      -- Make sure all the plot scales match
      Good := Input_Plot_Scale_List = Output_Plot_Scale_List;

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Good);
    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Output_File) then
            Ada.Text_IO.Close (Output_File);
         end if;

   end Test_Case_1;
   ----------------------------------------------------------------------------

   ----------------------------------------------------------------------------
   -- Test Case 2
   --
   -- Inputs           : A mapping function. A valid file with 1 entry.
   -- Expected Results : A two-element list with plot scales matching the values
   --                    placed in the file.
   -- Purpose          : Verify that a plot scale attribute name is converted
   --                    to the proper value.
   Test_Case_2 : declare

      Good  : Boolean := True;

      Mapping : aliased Recorder.Plot_Scale.Attribute_Map_List :=
        ( (1 => (String => Ada.Strings.Unbounded.To_Unbounded_String ("Fred"),
                 Name   => "TIME           "
                 ),
           2 => (String => Ada.Strings.Unbounded.To_Unbounded_String
                 ("The axis for something or other."),
                 Name   => "AXIS           "
                 )
           )
          );

      Output_Plot_Scale_List : constant Recorder.Plot_Scale.List :=
        (1 => ( ((others => ' '), 0.0, 20.0),
                ((others => ' '), -200.0, 101.5)
                )
         );

      Input_Plot_Scale_List : Recorder.Plot_Scale.List(Output_Plot_Scale_List'Range);
      Last_Plot_Scale : Natural;
   begin
      -- Create a text file matching Output_Plot_Scale_List
      Ada.Text_IO.Create (File => Output_File,
                          Mode => Ada.Text_IO.Out_File,
                          Name => Output_File_Name);

      -- Put out a dummy "test number"
      Ada.Text_Io.Put_Line
        (File => Output_File,
         Item => "t123456"
         );

      Ada.Text_Io.Put_Line
        (File => Output_File,
         Item =>
           Ada.Strings.Unbounded.To_String (Mapping(1).String) &
         ',' & Float'Image(Output_Plot_Scale_List(1).Y.Column_2) &
         ',' & Float'Image(Output_Plot_Scale_List(1).Y.Column_3) &
         ',' & Ada.Strings.Unbounded.To_String (Mapping(2).String) &
         ',' & Integer'Image(Integer(Output_Plot_Scale_List(1).X.Column_2)) &
         ',' & Integer'Image(Integer(Output_Plot_Scale_List(1).X.Column_3))
         );
      Ada.Text_Io.Close (Output_File);

      Ada.Text_Io.Put_Line ("Testing valid scale attribute mapping...");

      Recorder.Plot_Scale.Set_Name_Mapping (Mapping'Unchecked_Access);

      Recorder.Plot_Scale.Read_List
        (File_Name  => Output_File_Name,
         Scale_List => Input_Plot_Scale_List,
         Last       => Last_Plot_Scale
         );

      -- Make sure we read the expeted number of plot scales
      if Last_Plot_Scale /= Output_Plot_Scale_List'Last then
         Good := False;
         Ada.Text_Io.Put_Line
           ("Error: expected" &
            Integer'Image(Output_Plot_Scale_List'Last) &
            " items, read" &
            Integer'Image(Last_Plot_Scale)
            );
      end if;

      -- Make sure all the plot scales attribute names match the mapping
      Good := Input_Plot_Scale_List(1).Y.Column_1 = Mapping(1).Name and
        Input_Plot_Scale_List(1).X.Column_1 = Mapping(2).Name;

      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
         Ada.Text_Io.Put_Line ("Expected """ & Mapping(2).Name & """ and """ &
                               Mapping(1).Name & """");
         Ada.Text_Io.Put_Line ("Got """ & Input_Plot_Scale_List(1).X.Column_1
                               & """ and """ &
                               Input_Plot_Scale_List(1).Y.Column_1 & """");
      end if;

      Class_Test_Result_File.Report_Case_Status (Good);
    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Output_File) then
            Ada.Text_IO.Close (Output_File);
         end if;

   end Test_Case_2;
   ----------------------------------------------------------------------------


exception
when Event : others =>
   Ada.Text_IO.Put_Line ("unhandled exception:");
   Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
end  Recorder.Plot_Scale_Ctd;
