with Ada.Text_IO;
with Ada.Exceptions;
with Class_Test_Result_File;
with Jpats_Auto_Test.Driver_Id;
with Ada.Characters.Latin_1;

use type Jpats_Auto_Test.Driver_Id.Set;

------------------------------------------------------------------------------------------
-- Test driver for the driver_id package
------------------------------------------------------------------------------------------
procedure Jpats_Auto_Test.Driver_Id_Ctd is

   Driver_Map_File : Ada.Text_IO.File_Type;

   Driver_Map_Data : constant String :=
     "  THE_ELEVATOR_TRIM_OPEN_LOOP" & Ada.Characters.Latin_1.Lf &
     "  THE_RIGHT_AILERON_TRIM_OPEN_LOOP" & Ada.Characters.Latin_1.Lf &
     "  THE_TORQUE_TRIM" & Ada.Characters.Latin_1.Lf &
     "  THE_YAW_HOLD" & Ada.Characters.Latin_1.Lf &
     "at  1.00000E+00 use" & Ada.Characters.Latin_1.Lf &
     "  THE_ELEVATOR_TRIM_OPEN_LOOP" & Ada.Characters.Latin_1.Lf &
     "  THE_TORQUE_TRIM" & Ada.Characters.Latin_1.Lf &
     "  THE_YAW_HOLD" & Ada.Characters.Latin_1.Lf &
     "13.0" & Ada.Characters.Latin_1.Lf &
     "  THE_PITCH_HOLD" & Ada.Characters.Latin_1.Lf &
     "  THE_ROLL_HOLD" & Ada.Characters.Latin_1.Lf &
     "  THE_TORQUE_TRIM" & Ada.Characters.Latin_1.Lf &
     "  THE_YAW_HOLD" & Ada.Characters.Latin_1.Lf;
   
   Map_Number : constant := 134210;
   Map_Name : constant String := Jpats_Auto_Test_Types.A_File_path & "Auto_Test_Drivers\" &
     Jpats_Auto_Test.Driver_Id.Test_Path(Map_Number) & "/Run.map";
   
   Mapping : Jpats_Auto_Test.Driver_Id.Test_Mapping;
   
begin

   ----------------------------------------------------------------------------
   -- Test Case 1
   --
   -- Inputs           : A valid file
   -- Expected Results : Values before 0.0 => no drivers
   --                    Values between 0.0 and 1 => 
   --                       THE_ELEVATOR_TRIM_OPEN_LOOP and
   --                       THE_RIGHT_AILERON_TRIM_OPEN_LOOP and
   --                       THE_TORQUE_TRIM and
   --                       THE_YAW_HOLD
   --                    Values between 1 and 13 => 
   --                       THE_ELEVATOR_TRIM_OPEN_LOOP and
   --                       THE_TORQUE_TRIM and
   --                       THE_YAW_HOLD
   --                    Values after 13 =>
   --                        THE_PITCH_HOLD and
   --                        THE_ROLL_HOLD and
   --                        THE_TORQUE_TRIM and
   --                        THE_YAW_HOLD
   -- Purpose          : Verify that a correct map file is handled properly
   Test_Case_1 : declare

      Good  : Boolean := True;
      
      Actual_Set :  Jpats_Auto_Test.Driver_Id.Set;
   
      type Times_List is array (Natural range <>) of Duration;
      Time_To_Check : constant Times_List := (-1.0, 0.2, 5.0, 20.0);
      
      type Value_List is array (Natural range <>) of Jpats_Auto_Test.Driver_Id.Set;
      
      Expected_Value : constant Value_List :=
	((others => False),
	 (Jpats_Auto_Test.Driver_Id.THE_ELEVATOR_TRIM_OPEN_LOOP |
	  Jpats_Auto_Test.Driver_Id.THE_RIGHT_AILERON_TRIM_OPEN_LOOP |
          Jpats_Auto_Test.Driver_Id.THE_TORQUE_TRIM |
          Jpats_Auto_Test.Driver_Id.THE_YAW_Hold => True,
	  others => False),
	 (Jpats_Auto_Test.Driver_Id.THE_ELEVATOR_TRIM_OPEN_LOOP |
          Jpats_Auto_Test.Driver_Id.THE_TORQUE_TRIM |
          Jpats_Auto_Test.Driver_Id.THE_YAW_Hold => True,
	  others => False
	  ),
	 (Jpats_Auto_Test.Driver_Id.THE_PITCH_HOLD |
          Jpats_Auto_Test.Driver_Id.THE_ROLL_HOLD |
          Jpats_Auto_Test.Driver_Id.THE_TORQUE_TRIM |
          Jpats_Auto_Test.Driver_Id.THE_YAW_Hold => True,
	  others => False
	  )
	 );
	 
   begin
      -- Create a text file matching driver_map_data
      begin
	 Ada.Text_IO.Create (File => Driver_Map_File, 
			     Mode => Ada.Text_IO.Out_File, 
			     Name => Map_Name);
      exception
	 when Ada.Text_Io.Name_Error =>
	    Ada.Text_Io.Put_Line ("Error: unable to create file " &
				  Map_Name);
	    Ada.Text_Io.Put_Line ("(you may need to create the directory)");
	    raise;
      end;

      Ada.Text_IO.Put_Line (File => Driver_Map_File, Item =>
        "# Temporary driver map file created by");
      Ada.Text_IO.Put_Line (File => Driver_Map_File, Item =>
        "# Jpats_Auto_Test-Driver_Id_Ctd.adb");

      Ada.Text_IO.Put_Line (File => Driver_Map_File, Item => Driver_Map_Data);
			     
      Ada.Text_IO.Close(Driver_Map_File);

      Ada.Text_Io.Put_Line ("Testing valid map file...");
      
      Jpats_Auto_Test.Driver_Id.Prepare_Mapping
	(Mapping => Mapping,
	 Test    => Map_Number);
      
      for Index in Expected_Value'Range loop
      
	 Actual_Set := Jpats_Auto_Test.Driver_Id.Get
	   (Mapping => Mapping,
	    Time    => Time_To_Check (Index)
	    );
	 
	 if Actual_Set /= Expected_Value(Index) then
	    Good := False;
	    
	    Ada.Text_Io.Put_Line ("Error: values returned for time of" &
				  Duration'Image (Time_To_Check(Index)) &
				  " do not match expected values.");
	    Ada.Text_Io.Put_Line ("Values returned were:");
	    
	    for Index in Actual_Set'Range loop
	       if Actual_Set(Index) then
		  Ada.Text_Io.Put_Line 
		    ("   " & Jpats_Auto_Test.Driver_Id.Instance'Image(Index));
	       end if;
	    end loop;
	    
	 end if;
	      
      end loop;
      
      if Good then
         Ada.Text_IO.Put_Line ("success.");
      else
         Ada.Text_IO.Put_Line ("failure!");
      end if;

      Class_Test_Result_File.Report_Case_Status (Good);
    exception
      when Event : others =>
         Ada.Text_IO.Put_Line ("unhandled exception:");
         Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
         Class_Test_Result_File.Report_Case_Status (False);
         if Ada.Text_Io.Is_Open (Driver_Map_File) then
            Ada.Text_IO.Close (Driver_Map_File);
         end if;

   end Test_Case_1;
   ----------------------------------------------------------------------------


exception
when Event : others =>
   Ada.Text_IO.Put_Line ("unhandled exception:");
   Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information(Event));
end Jpats_Auto_Test.Driver_Id_Ctd;


