-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  James F. Narrin
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;

package body Lightning is

   Thunder_Timer : Float;
   Lightning_Id_N1 : Integer;

-- Access Enabled
   function  Enabled
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Enabled;
   end Enabled;

   procedure Set_Enabled
     (An_Instance : in out Instance;
      Lightning_Selected : in Boolean) is
   begin
      An_Instance.Enabled := Lightning_Selected;
   end Set_Enabled;

-- Access Visible
   function  Visible
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Visible;
   end Visible;

   procedure Set_Visible
     (An_Instance : in out Instance;
      Lightning_Visible : in Boolean) is
   begin
      An_Instance.Visible := Lightning_Visible;
   end Set_Visible;

-- Access Intensity
   function  Intensity
     (An_Instance : in Instance)
      return Integer is
   begin
      return An_Instance.Intensity;
   end Intensity;

   procedure Set_Intensity
     (An_Instance : in out Instance;
      Lightning_Intensity : in Integer) is
   begin
      An_Instance.Intensity := Lightning_Intensity;
   end Set_Intensity;

-- Access Direction
   function  Direction
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Direction;
   end Direction;

   procedure Set_Direction
     (An_Instance : in out Instance;
      Lightning_Direction : in Float) is
   begin
      An_Instance.Direction := Lightning_Direction;
   end Set_Direction;

-- Access Distance
   function Distance
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Distance;
   end Distance;

   procedure Set_Distance
     (An_Instance : in out Instance;
      Lightning_Distance : in Float) is
   begin
      An_Instance.Distance := Lightning_Distance;
   end Set_Distance;

-- Access Thunder
   function Thunder
     (An_Instance : in Instance)
     return Boolean is
   begin
      return An_Instance.Thunder;
   end Thunder;

   procedure Set_Thunder
     (An_Instance : in out Instance;
      Thunder_On : in Boolean) is
   begin
      An_Instance.Thunder := Thunder_On;
   end Set_Thunder;

-- Method Calculate_Direction
   procedure Calculate_Lightning_Direction
     (An_Instance : in out Instance;
      Lightning_Id : in Integer;
      AC_Latitude : in Long_Float;
      AC_Longitude : in Long_Float;
      Ac_Heading : in Float;
      Lightning_Latitude : in Long_Float;
      Lightning_Longitude : in Long_Float;
      Integration_Constant : in Float) is
      Delta_Lat : Float;
      Delta_Lon : Float;
      Coslat : Float;
      A_Distance : Float;
      A_Direction : Float;
   begin
      if An_Instance.Enabled then
         Coslat := Cos(Float(Ac_Latitude),360.0);

         if Lightning_Id /= Lightning_Id_N1 then
            Thunder_Timer := 0.5;

            Delta_Lat := Float((Lightning_Latitude - Ac_Latitude) * 60.0);
            Delta_Lon := Float((Lightning_Longitude - Ac_Longitude)
              * 60.0) * Coslat;
            An_Instance.Distance := Sqrt(Delta_Lat*Delta_Lat +
                                         Delta_Lon*Delta_Lon);
            A_Distance := Sqrt(Delta_Lat*Delta_Lat +
                               Delta_Lon*Delta_Lon);
            A_Direction := Arctan(Delta_Lon, Delta_Lat, 360.0);
            An_Instance.Direction := Arctan(Delta_Lon, Delta_Lat, 360.0);
         elsif Thunder_Timer > 0.0 then
            Thunder_Timer := Thunder_Timer - Integration_Constant;
         end if;

         An_Instance.Thunder := Thunder_Timer > 0.0;

         if An_Instance.Distance > 0.0 then
            An_Instance.Intensity :=
              Integer(1.0/An_Instance.Distance * 1000.0);
            if An_Instance.Intensity > 100 then
               An_Instance.Intensity := 100;
            end if;
            if An_Instance.Intensity < 0 then
               An_Instance.Intensity := 0;
            end if;
         end if;

         An_Instance.Visible := abs(An_Instance.Direction - Ac_Heading) < 90.0;

         Lightning_Id_N1 := Lightning_Id;

      else
         An_Instance.Visible := False;
         An_Instance.Intensity := 0;
         An_Instance.Direction := 0.0;
         An_Instance.Distance := 0.0;
         An_Instance.Thunder := False;
      end if;

   end Calculate_Lightning_Direction;

-- Initialize
   procedure Initialize
     (An_Instance : in out Instance) is
   begin
      An_Instance.Enabled := False;
      An_Instance.Visible := False;
      An_Instance.Intensity := 0;
      An_Instance.Direction := 0.0;
      An_Instance.Distance := 0.0;
      An_Instance.Thunder := False;
      Thunder_Timer := 0.0;
      Lightning_Id_N1 := 0;
   end Initialize;


end Lightning;
