-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  James F. Narrin
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Moving_Models.Position.Cloud_Model;
with Moving_Models.Position.Storm_Model;
with Moving_Models.Position.Host_Hazard;
with Moving_Models.Air_Hazard;
with Moving_Models.Ground_Hazard;
with Traffic_Model;
with Jpats_Visual_Models.Container;
with Jpats_Visual_Models.Ios_If;
with Jpats_Visual_Models_Types;
with Jpats_Avionics;
with Jpats_Avionics_Types;
use Jpats_Avionics_Types;
with Jpats_Simulated_Aircraft;
with Jpats_Formation_Manager;
with Jpats_Visual_Weather;
with Jpats_Visual_Airport;
with Jpats_Visual_Buffer;
with Jpats_Landing_Gear_Types;
with Jpats_Landing_Gear;
with Visual_Airport_Utilities;
with Lat_Long_Types;
with Moving_Models;
with Threat_Control;
with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;
with Ada.Text_Io;
use Ada.Text_Io;
with Simulation_Dictionary;

package body JPATS_Visual_Models.Controller is

   Ios : Container.Ios_Interface_Instance renames Container.This_Ios_Interface;
   package Threat renames Threat_Control;
   package Gear renames Jpats_Landing_Gear;
   package Gear_Types renames Jpats_Landing_Gear_Types;
   package Jat renames Jpats_Avionics_Types;
   package Sim_Ac renames Jpats_Simulated_Aircraft;

   Ff_Test_Active : Boolean;
   Ff_Test_Position : Lat_Long_Types.Position;
   Ff_Test_Heading : Float;
   Ff_Test_Speed_East : Float;
   Ff_Test_Speed_North : Float;

   Visual_Models_Taw6 : Boolean := False;
   Visual_Models_Taw5 : Boolean := False;
   Visual_Models_Taw4 : Boolean := False;
   Visual_Models_Iaf  : Boolean := False;

   Ac_Type_Temp : Jat.Aircraft_Type;

-- Method Initialize
   procedure Initialize is
   begin

      Visual_Models_Taw6 :=
        Simulation_Dictionary.Lookup("Visual_Models") = "TAW6";

      Visual_Models_Taw5 :=
        Simulation_Dictionary.Lookup("Visual_Models") = "TAW5";

      Visual_Models_Taw4 :=
        Simulation_Dictionary.Lookup("Visual_Models") = "TAW4";

      Traffic_Model.Initialize(Container.This_Subsystem.The_Traffic_Model);

      Moving_Models.Air_Hazard.Initialize
        (Container.This_Subsystem.The_Air_Hazard);
      Moving_Models.Air_Hazard.Set_Model_Id
        (Container.This_Subsystem.The_Air_Hazard,30);

      Moving_Models.Ground_Hazard.Initialize
        (Container.This_Subsystem.The_Ground_Hazard);
      Moving_Models.Ground_Hazard.Set_Model_Id
        (Container.This_Subsystem.The_Ground_Hazard,31);

      --Initialize NACWS Threat control attributes
      Threat.Init(Container.This_Subsystem.The_Threat_Control);

      Moving_Models.Position.Host_Hazard.Initialize
        (Container.This_Subsystem.The_Formation_Model);
      Moving_Models.Position.Host_Hazard.Set_Model_Id
        (Container.This_Subsystem.The_Formation_Model,25);
      if Visual_Models_TAW4 then
         Moving_Models.Position.Host_Hazard.Set_Model_Type
           (Container.This_Subsystem.The_Formation_Model,1033);
      elsif Visual_Models_TAW5 then
         Moving_Models.Position.Host_Hazard.Set_Model_Type
           (Container.This_Subsystem.The_Formation_Model,1020);
      elsif Visual_Models_TAW6 then
         Moving_Models.Position.Host_Hazard.Set_Model_Type
           (Container.This_Subsystem.The_Formation_Model,1003);
      else
         Moving_Models.Position.Host_Hazard.Set_Model_Type
           (Container.This_Subsystem.The_Formation_Model,1000);
      end if;

      for I in 1..10 loop
         Moving_Models.Position.Host_Hazard.Initialize
           (Container.This_Subsystem.The_Host_Hazard(I));
         Moving_Models.Position.Host_Hazard.Set_Model_Id
           (Container.This_Subsystem.The_Host_Hazard(I),I);
      end loop;

      for I in 1..5 loop
         Moving_Models.Position.Storm_Model.Initialize
           (Container.This_Subsystem.The_Storm_Model(I));
         Moving_Models.Position.Storm_Model.Set_Model_Id
           (Container.This_Subsystem.The_Storm_Model(I),I+15);
      end loop;

      for I in 1..2 loop
         Moving_Models.Position.Cloud_Model.Initialize
           (Container.This_Subsystem.The_Cloud_Model(I));
         Moving_Models.Position.Cloud_Model.Set_Model_Id
           (Container.This_Subsystem.The_Cloud_Model(I),I+20);
      end loop;

      Container.This_Ios_Interface.Air_Traffic := false;
      Container.This_Ios_Interface.Ground_Traffic := false;
      Container.This_Ios_Interface.Ramp_Traffic := false;
      Container.This_Ios_Interface.Ground_Hazard_On := false;
      Container.This_Ios_Interface.Ground_Hazard_Off := true;
      Container.This_Ios_Interface.Ground_Hazard_Type := 0;
      Container.This_Ios_Interface.Ground_Hazard_Path := 0;
      Container.This_Ios_Interface.Ground_Hazard_Pause := false;

      Container.This_Ios_Interface.Air_Traffic_Was_Set := false;
      Container.This_Ios_Interface.Ground_Traffic_Was_Set := false;
      Container.This_Ios_Interface.Ramp_Traffic_Was_Set := false;
      Container.This_Ios_Interface.Ground_Hazard_On_Was_Set := false;
      Container.This_Ios_Interface.Ground_Hazard_Off_Was_Set := false;
      Container.This_Ios_Interface.Ground_Hazard_Type_Was_Set := false;
      Container.This_Ios_Interface.Ground_Hazard_Path_Was_Set := false;
      Container.This_Ios_Interface.Ground_Hazard_Pause_Was_Set := false;

      Jpats_Visual_Models.Ios_If.Register_Ios_Variables;

   end Initialize;

-- Method Update
   procedure Update
     (Integration_Constant : in Float) is
      Intruder : Jpats_Avionics_Types.Intruder_Drive_Array_Type;
      Model_Position : Lat_Long_Types.Position;
      Rad_To_Deg : constant := 57.29577951308;
      Airport_Position : Lat_Long_Types.Position;
      Aircraft_Position : Lat_Long_Types.Position;
      Lower_Cloud_Model_Type : Integer := 0;
      Upper_Cloud_Model_Type : Integer := 0;
      Storm_Type             : Integer := 0;

   begin

      Airport_Position := Jpats_Visual_Airport.Position;
      Aircraft_Position.Latitude := Sim_Ac.Get_North;
      Aircraft_Position.Longitude := Sim_Ac.Get_East;
      Aircraft_Position.Altitude :=
        abs(Sim_Ac.Get_Aircraft_Geometric_Altitude);
--------THREAT CONTROL---------------------------------------------------
      --The controller interfaces with the IOS and uses ownship data to
      --simulate the flight path of an intruder into the ownship airspace.
      --This intruder data is also used to drive an intruder on the visual
      --system.

      Threat.Set_Ground_Elevation
        (Container.This_Subsystem.The_Threat_Control,
         Jpats_Visual_Buffer.Average_Terrain_Height);
      Threat.Set_Aircraft_Position
        (Container.This_Subsystem.The_Threat_Control,
         Aircraft_Position);
      Threat.Set_True_Heading
        (Container.This_Subsystem.The_Threat_Control,
         Sim_Ac.Get_Hdg_Angle * 57.3);
      Threat.Set_Reposition_Active
        (Container.This_Subsystem.The_Threat_Control,
         Sim_Ac.Reposition_In_Progress);
      Threat.Set_Earth_Axis_Velocity
        (Container.This_Subsystem.The_Threat_Control,
         Sim_Ac.Get_Vel_Ea);
      Threat.Set_Body_Axis_Velocity
        (Container.This_Subsystem.The_Threat_Control,
         Sim_Ac.Get_V_B);
      Threat.Set_Rate_Of_Climb
        (Container.This_Subsystem.The_Threat_Control,
         Sim_Ac.Get_Rate_Of_Climb);
      Threat.Set_Weight_On_Wheels
        (Container.This_Subsystem.The_Threat_Control,
         Gear.Is_Closed(Gear_Types.Left_Gear_Wow_Switch));

      --here are the ios threat control inputs:
      if (Ios.Threat_Scenario_Was_Set(1)) then
         if (Ios.Threat_Scenario(1) > 0) then
            Ios.Threat_Page_Aircraft_Type(2) := Jat.T6a_AF;
            Ios.Threat_Page_Distance(1)      := 0.1;
            Ios.Threat_Page_Altitude(1)      := 0.0;
            Ios.Threat_Page_Time(1)          := 0.2;
            Ios.Threat_Page_Speed(1)         := 200.0;
            if (Ios.Threat_Scenario(1) = 3) then
               Ios.Threat_Page_Vert_Pos(1)   := Jat.Below;
               Ios.Threat_Page_Closure(1)    := 1500.0;
            else
               Ios.Threat_Page_Vert_Pos(1)   := Jat.Level;
               Ios.Threat_Page_Closure(1)    := 0.0;
            end if;
            if (Ios.Threat_Scenario(1) = 1) then
              Ios.Threat_Page_Bearing(1)     := 90.0;
            elsif (Ios.Threat_Scenario(1) = 2) then
              Ios.Threat_Page_Bearing(1)     := -90.0;
            elsif (Ios.Threat_Scenario(1) = 3) then
              Ios.Threat_Page_Bearing(1)     := -30.0;
            end if;
         else
            Ios.Threat_Page_Distance(1)      := 0.0;
            Ios.Threat_Page_Altitude(1)      := 0.0;
            Ios.Threat_Page_Time(1)          := 0.0;
            Ios.Threat_Page_Speed(1)         := 0.0;
            Ios.Threat_Page_Vert_Pos(1)      := Jat.Level;
            Ios.Threat_Page_Closure(1)       := 0.0;
            Ios.Threat_Page_Bearing(1)       := 0.0;
         end if;
         Ios.Threat_Scenario_Was_Set(1) := False;
      end if;

      if (Ios.Threat_Scenario_Was_Set(2)) then
         if (Ios.Threat_Scenario(2) > 0) then
            Ios.Threat_Page_Aircraft_Type(2) := Jat.T6a_AF;
            Ios.Threat_Page_Distance(2)      := 0.1;
            Ios.Threat_Page_Altitude(2)      := 0.0;
            Ios.Threat_Page_Time(2)          := 0.2;
            Ios.Threat_Page_Speed(2)         := 200.0;
            Ios.Threat_Page_Vert_Pos(2)      := Jat.Level;
            Ios.Threat_Page_Closure(2)       := 0.0;
            if (Ios.Threat_Scenario(2) = 1) then
              Ios.Threat_Page_Bearing(2)     := -30.0;
            elsif (Ios.Threat_Scenario(2) = 2) then
              Ios.Threat_Page_Bearing(2)     := 90.0;
            elsif (Ios.Threat_Scenario(2) = 3) then
              Ios.Threat_Page_Bearing(2)     := 0.0;
            end if;
         else
            Ios.Threat_Page_Distance(2)      := 0.0;
            Ios.Threat_Page_Altitude(2)      := 0.0;
            Ios.Threat_Page_Time(2)          := 0.0;
            Ios.Threat_Page_Speed(2)         := 0.0;
            Ios.Threat_Page_Vert_Pos(2)      := Jat.Level;
            Ios.Threat_Page_Closure(2)       := 0.0;
            Ios.Threat_Page_Bearing(2)       := 0.0;
         end if;
         Ios.Threat_Scenario_Was_Set(2) := False;
      end if;


      Threat.Set_Threat_Page_Start_Button
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Start_Button);
      Threat.Set_Threat_Page_Aircraft_Type
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Aircraft_Type);
      Threat.Set_Threat_Page_Bearing
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Bearing);
      Threat.Set_Threat_Page_Distance
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Distance);
      Threat.Set_Threat_Page_Altitude
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Altitude);
      Threat.Set_Threat_Page_Time
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Time);
      Threat.Set_Threat_Page_Vert_Pos
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Vert_Pos);
      Threat.Set_Threat_Page_Speed
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Speed);
      Threat.Set_Threat_Page_Closure
        (Container.This_Subsystem.The_Threat_Control,
         Ios.Threat_Page_Closure);

      Threat.Set_Flight_Freeze
        (Container.This_Subsystem.The_Threat_Control,
         Sim_Ac.Get_Flight_Freeze);

      --call threat control update procedure
      Threat.Update(Integration_Constant,
                    Container.This_Subsystem.The_Threat_Control);

      if (Ios.Threat_Page_Speed_Was_Set(1) or
          Ios.Threat_Page_Speed_Was_Set(2)) then
         Ios.Threat_Page_Speed := Threat.Threat_Page_Speed
           (Container.This_Subsystem.The_Threat_Control);
         Ios.Threat_Page_Speed_Was_Set(1) := False;
         Ios.Threat_Page_Speed_Was_Set(2) := False;
      end if;

      Container.This_Subsystem.The_Intruder_Drive :=
        Threat.Intruder_Drive
        (Container.This_Subsystem.The_Threat_control);

--Ground Traffic Drive
      if (Container.This_Ios_Interface.Ground_Traffic_Was_Set) then
         Traffic_Model.Set_Ground_Traffic
           (Container.This_Subsystem.The_Traffic_Model,
            Container.This_Ios_Interface.Ground_Traffic);
         Container.This_Ios_Interface.Ground_Traffic_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Ground_Hazard_On_Was_Set) then
         if (Container.This_Ios_Interface.Ground_Hazard_On) then
               Moving_Models.Ground_Hazard.Set_Model_Type
                 (Container.This_Subsystem.The_Ground_Hazard,
                  Container.Ground_Model_Number_List
                  (Jpats_Avionics_Types.Aircraft_Type'Val
                   (Container.This_Ios_Interface.Ground_Hazard_Type)));
            Moving_Models.Ground_Hazard.Set_Hazard_Path
              (Container.This_Subsystem.The_Ground_Hazard,
               Container.This_Ios_Interface.Ground_Hazard_Path);
            Container.This_Ios_Interface.Ground_Hazard_Off := false;
         else
            Container.This_Ios_Interface.Ground_Hazard_Pause := false;
            Moving_Models.Ground_Hazard.Set_Hazard_Pause
              (Container.This_Subsystem.The_Ground_Hazard,
               Container.This_Ios_Interface.Ground_Hazard_Pause);
            Container.This_Ios_Interface.Ground_Hazard_Off := true;
         end if;
         Moving_Models.Ground_Hazard.Set_Active
           (Container.This_Subsystem.The_Ground_Hazard,
            Container.This_Ios_Interface.Ground_Hazard_On);
         Container.This_Ios_Interface.Ground_Hazard_On_Was_Set := False;
      elsif (Container.This_Ios_Interface.Ground_Hazard_Pause_Was_Set) then
         if (Moving_Models.Ground_Hazard.Active
             (Container.This_Subsystem.The_Ground_Hazard)) then
            Moving_Models.Ground_Hazard.Set_Hazard_Pause
              (Container.This_Subsystem.The_Ground_Hazard,
               Container.This_Ios_Interface.Ground_Hazard_Pause);
         end if;
         Container.This_Ios_Interface.Ground_Hazard_Pause_Was_Set := False;
      end if;

      Intruder := Container.This_Subsystem.The_Intruder_Drive;

      for I in Jpats_Avionics_Types.Intruders'Range loop
         if (I > 3) then
            exit;
         end if;
         if (Intruder(I).Active) then
            Model_Position.Latitude := Intruder(I).Lat;
            Model_Position.Longitude := Intruder(I).Lon;
            Model_Position.Altitude := Intruder(I).Altitude;
            Moving_Models.Position.Host_Hazard.Set_Position
              (Container.This_Subsystem.The_Host_Hazard(I),
               Model_Position);
            Moving_Models.Position.Host_Hazard.Set_Heading
              (Container.This_Subsystem.The_Host_Hazard(I),
               Visual_Airport_Utilities.Xn360(Intruder(I).Heading));
            Moving_Models.Position.Host_Hazard.Set_Hazard_Pitch
              (Container.This_Subsystem.The_Host_Hazard(I),
               Intruder(I).Pitch);
            Moving_Models.Position.Host_Hazard.Set_Hazard_Roll
              (Container.This_Subsystem.The_Host_Hazard(I),
               Intruder(I).Roll);
               Moving_Models.Position.Host_Hazard.Set_Model_Type
                 (Container.This_Subsystem.The_Host_Hazard(I),
                  Container.Model_Number_List(Intruder(I).Aircraft));
         end if;
         Moving_Models.Position.Host_Hazard.Set_Active
           (Container.This_Subsystem.The_Host_Hazard(I),
            Intruder(I).Active);
      end loop;

-- Formation Flight

      if Jpats_Formation_Manager.Display_Lead then
         Model_Position.Latitude :=
           Jpats_Formation_Manager.Lead_Latitude;
         Model_Position.Longitude :=
           Jpats_Formation_Manager.Lead_Longitude;
         Model_Position.Altitude :=
           Jpats_Formation_Manager.Lead_Geometric_Altitude;
         Moving_Models.Position.Host_Hazard.Set_Position
           (Container.This_Subsystem.The_Formation_Model,
            Model_Position);

         Moving_Models.Position.Host_Hazard.Set_Heading
           (Container.This_Subsystem.The_Formation_Model,
            Visual_Airport_Utilities.Xn360
            (Jpats_Formation_Manager.Lead_Heading));
         Moving_Models.Position.Host_Hazard.Set_Hazard_Pitch
           (Container.This_Subsystem.The_Formation_Model,
            Jpats_Formation_Manager.Lead_Pitch_Angle);
         Moving_Models.Position.Host_Hazard.Set_Hazard_Roll
           (Container.This_Subsystem.The_Formation_Model,
            Jpats_Formation_Manager.Lead_Bank_Angle);
         Moving_Models.Position.Host_Hazard.Set_Active
           (Container.This_Subsystem.The_Formation_Model,True);
      else
         Moving_Models.Position.Host_Hazard.Set_Active
           (Container.This_Subsystem.The_Formation_Model,False);
      end if;

      if (Container.This_Ios_Interface.Ff_Test_Was_Set) then

         if (Container.This_Ios_Interface.Ff_Test) then
--            Ff_Test_Heading := Jpats_Simulated_Aircraft.Get_Hdg_Angle*
--              Rad_To_Deg;
--            Ff_Test_Position :=
--              Jpats_Simulated_Aircraft.Get_Aircraft_Position;
--            Ff_Test_Position.Latitude := Ff_Test_Position.Latitude +
--              800.0/6076.1/60.0*Long_Float(Cos(Ff_Test_Heading,360.0));
--            Ff_Test_Position.Longitude := Ff_Test_Position.Longitude +
--              800.0/6076.1/60.0*Long_Float(Sin(Ff_Test_Heading,360.0))/
--              Long_Float(Cos(Float(Ff_Test_Position.Latitude),360.0));
--            Ff_Test_Speed_East :=
--              (Jpats_Simulated_Aircraft.Get_True_Airspeed-16.9)*
--              Sin(Ff_Test_Heading,360.0);
--            Ff_Test_Speed_North :=
--              (Jpats_Simulated_Aircraft.Get_True_Airspeed-16.9)*
--              Cos(Ff_Test_Heading,360.0);

            Ff_Test_Heading  := 135.0;
                 Ff_Test_Position := Jpats_Simulated_Aircraft.Get_Aircraft_Position;
            Ff_Test_Position.Latitude  := 29.479;
            Ff_Test_Position.Longitude := -100.916;
            Ff_Test_Position.Altitude  := 4000.0;
            Ff_Test_Speed_East  :=  0.0;
            Ff_Test_Speed_North :=  0.0;
            Ff_Test_Active := True;
         else
            Ff_Test_Active := False;
            Moving_Models.Position.Host_Hazard.Set_Active
              (Container.This_Subsystem.The_Formation_Model,False);
         end if;
         Container.This_Ios_Interface.Ff_Test_Was_Set := False;
      elsif (Ff_Test_Active) then
         Ff_Test_Position.Latitude := Ff_Test_Position.Latitude +
           Long_Float(Ff_Test_Speed_North * Integration_Constant)/
           60.0/6076.1;
         Ff_Test_Position.Longitude := Ff_Test_Position.Longitude +
           Long_Float(Ff_Test_Speed_East * Integration_Constant)/
           60.0/6076.1/Long_Float(Cos(Float(Ff_Test_Position.Latitude)));

         Moving_Models.Position.Host_Hazard.Set_Position
           (Container.This_Subsystem.The_Formation_Model,
            Ff_Test_Position);
         Moving_Models.Position.Host_Hazard.Set_Heading
           (Container.This_Subsystem.The_Formation_Model,
            Ff_Test_Heading);
         Moving_Models.Position.Host_Hazard.Set_Hazard_Pitch
           (Container.This_Subsystem.The_Formation_Model,0.0);
         Moving_Models.Position.Host_Hazard.Set_Hazard_Roll
           (Container.This_Subsystem.The_Formation_Model,0.0);
         Moving_Models.Position.Host_Hazard.Set_Model_Type
           (Container.This_Subsystem.The_Formation_Model,457);
         Moving_Models.Position.Host_Hazard.Set_Active
           (Container.This_Subsystem.The_Formation_Model,True);

         Container.This_Ios_Interface.Ff_Test_Lat := Ff_Test_Position.Latitude;
         Container.This_Ios_Interface.Ff_Test_Lon :=
           Ff_Test_Position.Longitude;
         Container.This_Ios_Interface.Ff_Test_Hdg := Ff_Test_Heading;
         Container.This_Ios_Interface.Ff_Test_Alt := Ff_Test_Position.Altitude;
      else
         Container.This_Ios_Interface.Ff_Test_Lat := 0.0;
         Container.This_Ios_Interface.Ff_Test_Lon := 0.0;
         Container.This_Ios_Interface.Ff_Test_Hdg := 0.0;
         Container.This_Ios_Interface.Ff_Test_Alt := 0.0;
      end if;

-- Cloud Models
      if (Jpats_Visual_Weather.Lower_Cloud_Layer_Broken) then
         Lower_Cloud_Model_Type := 60023;
      elsif (Jpats_Visual_Weather.Lower_Cloud_Layer_Scattered) then
         Lower_Cloud_Model_Type := 60022;
      end if;

      if (Jpats_Visual_Weather.Upper_Cloud_Layer_Broken) then
         Upper_Cloud_Model_Type := 60023;
      elsif (Jpats_Visual_Weather.Upper_Cloud_Layer_Scattered) then
         Upper_Cloud_Model_Type := 60022;
      end if;

      if (Jpats_Visual_Weather.Lower_Cloud_Layer_Broken or
          Jpats_Visual_Weather.Lower_Cloud_Layer_Scattered ) then
         Model_Position := Airport_Position;
         Model_Position.Altitude := Airport_Position.Altitude +
           Float(Jpats_Visual_Weather.Lower_Cloud_Ceiling);
         Moving_Models.Position.Cloud_Model.Set_Position
           (Container.This_Subsystem.The_Cloud_Model(1),
            Model_Position);
         Moving_Models.Position.Cloud_Model.Set_Heading
           (Container.This_Subsystem.The_Cloud_Model(1), 0.0);
         Moving_Models.Position.Cloud_Model.Set_Model_Type
           (Container.This_Subsystem.The_Cloud_Model(1),
            Lower_Cloud_Model_Type);
         Moving_Models.Position.Cloud_Model.Set_Active
           (Container.This_Subsystem.The_Cloud_Model(1), True);
      else
         Moving_Models.Position.Cloud_Model.Set_Active
           (Container.This_Subsystem.The_Cloud_Model(1), False);
      end if;

      if (Jpats_Visual_Weather.Upper_Cloud_Layer_Broken or
          Jpats_Visual_Weather.Upper_Cloud_Layer_Scattered) then
         Model_Position := Airport_Position;
         Model_Position.Altitude := Airport_Position.Altitude +
           Float(Jpats_Visual_Weather.Upper_Cloud_Ceiling);
         Moving_Models.Position.Cloud_Model.Set_Position
           (Container.This_Subsystem.The_Cloud_Model(2),
            Model_Position);
         Moving_Models.Position.Cloud_Model.Set_Heading
           (Container.This_Subsystem.The_Cloud_Model(2), 0.0);
         Moving_Models.Position.Cloud_Model.Set_Model_Type
           (Container.This_Subsystem.The_Cloud_Model(2),
            Upper_Cloud_Model_Type);
         Moving_Models.Position.Cloud_Model.Set_Active
           (Container.This_Subsystem.The_Cloud_Model(2), True);
      else
         Moving_Models.Position.Cloud_Model.Set_Active
           (Container.This_Subsystem.The_Cloud_Model(2), False);
      end if;

-- Storm Models
      if (Jpats_Visual_Weather.Storm_1) then
         Storm_Type := 60008;
      elsif (Jpats_Visual_Weather.Storm_2) then
         Storm_Type := 60006;
      elsif (Jpats_Visual_Weather.Storm_3) then
         Storm_Type := 60009;
      elsif (Jpats_Visual_Weather.Storm_4) then
         Storm_Type := 60007;
      elsif (Jpats_Visual_Weather.Storm_5) then
         Storm_Type := 60010;
      elsif (Jpats_Visual_Weather.Storm_6) then
         Storm_Type := 60005;
      elsif (Jpats_Visual_Weather.Storm_7) then
         Storm_Type := 60002;
      elsif (Jpats_Visual_Weather.Storm_8) then
         Storm_Type := 60003;
      elsif (Jpats_Visual_Weather.Storm_9) then
         Storm_Type := 60004;
      elsif (Jpats_Visual_Weather.Storm_10) then
         Storm_Type := 60001;
      end if;


      if (Jpats_Visual_Weather.Storm_1 or
          Jpats_Visual_Weather.Storm_2 or
          Jpats_Visual_Weather.Storm_3 or
          Jpats_Visual_Weather.Storm_4 or
          Jpats_Visual_Weather.Storm_5 or
          Jpats_Visual_Weather.Storm_6 or
          Jpats_Visual_Weather.Storm_7 or
          Jpats_Visual_Weather.Storm_8 or
          Jpats_Visual_Weather.Storm_9 or
          Jpats_Visual_Weather.Storm_10 ) then
         Model_Position := Jpats_Visual_Weather.Storm_Location;
         Model_Position.Altitude := Airport_Position.Altitude +
           Float(Jpats_Visual_Weather.Storm_Ceiling);
         Moving_Models.Position.Storm_Model.Set_Position
           (Container.This_Subsystem.The_Storm_Model(1),
            Model_Position);
         Moving_Models.Position.Storm_Model.Set_Heading
           (Container.This_Subsystem.The_Storm_Model(1), Float(Jpats_Visual_Weather.Storm_Rotation));
         Moving_Models.Position.Storm_Model.Set_Model_Type
           (Container.This_Subsystem.The_Storm_Model(1),
            Storm_Type);
         Moving_Models.Position.Storm_Model.Set_Active
           (Container.This_Subsystem.The_Storm_Model(1), True);
         -- Move Storm (and lightning) for next pass
         Jpats_Visual_Weather.Update_Storms(Integration_Constant);         
      else
         Moving_Models.Position.Storm_Model.Set_Active
           (Container.This_Subsystem.The_Storm_Model(1), False);
      end if;


   end Update;


end JPATS_Visual_Models.Controller;







