-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  James F. Narrin
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with JPATS_IOS_Interface;
with Jpats_Visual_Buffer_Types;
with Jpats_Visual_Buffer.Slow_Data_Container;
with Jpats_Visual_Buffer.Ios_If;
with Jpats_Visual_Airport;
with Jpats_Visual_Airport.Ios_If;
with Jpats_Visual_Weather;
with Jpats_Visual_Weather_Types;
with Jpats_Visual_Weather.Ios_If;
with Jpats_Visual_Models;
with Data_Block.System_Features;
with Data_Block.Airport_Control;
with Data_Block.Runway_Control;
with Data_Block.Atmosphere_Control;
with Data_Block.Celestial_Control;
with Data_Block.Config_Apt_Cntl;
with Data_Block.Test_Pattern;
with Data_Block.Recording_Control;
with Jpats_Simulated_Aircraft;
with Jpats_Atmosphere;
with Jpats_Auto_Test;
with Jpats_Reposition;
with Jpats_Radio_Db_If_Types;
with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;
with Lat_Long_Types;

package body JPATS_Visual_Buffer.Slow_Data_Controller is

   Morning_Readiness_Test_Number : Integer;
   Test_Number                   : Integer;
   Test_Active                   : Boolean := False;

   type Test_Name_Enum is (None, Clr,  Cr,   Dcha, Defc, Fsw8, Geom, Gr8, Gr16,
                           Illv, Jllx, Jlly, Le4k, Lcrc, Oc30, Pe1k, Reso,
                           Setx, Sety, Slc8, Slcc, Slr2, Sphr, Star, Tput,
                           Hlit, Sp15, Sp18, Sp22, Dchb, Fsw0, Fsw1, Fsw2,
                           Fsw3, Fsw4, Fsw5, Fsw6, Fsw7, Le6k, Llit, Lp6, Lp8,
                           Morn, Slc0, Slc1, Slc2, Slc3, Slc4, Slc5, Slc6,
                           Slc7, Sp16, T45b, T45l, T45r, Xlea, Ylea, ld6d,
                           ld4n, Spjp, Sph7);
   Test_Name : constant array (Test_Name_Enum)
     of Jpats_Visual_Buffer_Types.Test_Name_String :=
     ("        ","clr     ","cr      ","dcha    ","defc    ","fsw8    ",
      "geom    ","gr8     ","gr16    ","illv    ","jllx    ","jlly    ",
      "le4k    ","lcrc    ","oc30    ","pe1k    ","reso    ","setx    ",
      "sety    ","slc8    ","slcc    ","slr2    ","sphr    ","star    ",
      "tput    ","hlit    ","sp150   ","sp180   ","sp225   ","dchb    ",
      "fsw0    ","fsw1    ","fsw2    ","fsw3    ","fsw4    ","fsw5    ",
      "fsw6    ","fsw7    ","le6k    ","llit    ","lp6     ","lp8     ",
      "morn    ","slc0    ","slc1    ","slc2    ","slc3    ","slc4    ",
      "slc5    ","slc6    ","slc7    ","sp165   ","t45b    ","t45l    ",
      "t45r    ","xleadlag","yleadlag","ld6d    ","ld4n    ","spjp    ",
      "sphere  ");

   Test_Started : Boolean := False;
   Auto_Test_Timer : Float := 0.0;

   Host_Buffer_Save_Timer : Float := 0.0;

-- Method Initialize
   procedure Initialize is
   begin
      Data_Block.System_Features.Initialize
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features);
      Data_Block.Airport_Control.Initialize
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control);
      Data_Block.Celestial_Control.Initialize
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control);
      Data_Block.Runway_Control.Initialize
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control);
      Data_Block.Atmosphere_Control.Initialize
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control);
      Data_Block.Config_Apt_Cntl.Initialize
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl);
      Data_Block.Test_Pattern.Initialize
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern);

      Slow_Data_Container.This_Ios_Interface.Video_On := true;
      Slow_Data_Container.This_Ios_Interface.Host_Buffer_Save := false;
      Slow_Data_Container.This_Ios_Interface.Record_On := false;
      Slow_Data_Container.This_Ios_Interface.Record_Pause := false;
      Slow_Data_Container.This_Ios_Interface.Record_Reverse := false;
      Slow_Data_Container.This_Ios_Interface.Record_Single_Step := false;
      Slow_Data_Container.This_Ios_Interface.Sea_State := 0;
      Slow_Data_Container.This_Ios_Interface.Visual_Transmit := True;
      Slow_Data_Container.This_Ios_Interface.Config_Apt_Enable := True;
      Slow_Data_Container.This_Ios_Interface.Visual_Com_Reset := False;

      Slow_Data_Container.This_Ios_Interface.Video_On_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.
        Host_Buffer_Save_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.Record_On_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.Record_Pause_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.
        Record_Reverse_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.
        Record_Single_Step_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.Sea_State_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.Visual_Transmit_Was_Set := false;
      Slow_Data_Container.This_Ios_Interface.Visual_Com_Reset_Was_Set := false;

      Jpats_Visual_Buffer.Ios_If.Register_Ios_Variables;

   end Initialize;

-- Method Update
   procedure Update
     (Integration_Constant : in Float) is
      Ac_Position : Lat_Long_Types.Position;
      Ac_Pitch : Float;
      Ac_Roll : Float;
      Ac_Yaw : Float;
      Airport_Position : Lat_Long_Types.Position;
      Crash : Boolean;
      Wind_Speed : Float;
      Active_Airport : Jpats_Radio_Db_If_Types.Apt_Struct_Type;
      Active_Runway : Jpats_Radio_Db_If_Types.Rwy_Struct_Type;
      Runway_Position : Lat_Long_Types.Position;
   begin

      Active_Airport := Jpats_Reposition.Reference_Airfield;
      Active_Runway := Jpats_Reposition.Reference_Runway;

      Ac_Position.Latitude := Jpats_Simulated_Aircraft.Get_North;
      Ac_Position.Longitude := Jpats_Simulated_Aircraft.Get_East;

--      Ac_Position.Latitude := 45.000;
--      Ac_Position.Longitude := -44.957526988;

      Airport_Position := Jpats_Visual_Airport.Position;
      Ac_Pitch := Jpats_Simulated_Aircraft.Get_Pitch_Angle;
      Ac_Roll := Jpats_Simulated_Aircraft.Get_Roll_Angle;
      Ac_Yaw := Jpats_Simulated_Aircraft.Get_Hdg_Angle;
      Crash := Jpats_Simulated_Aircraft.Get_Crash_Sink_Rate or
               Jpats_Simulated_Aircraft.Get_Crash_Vertical_Load or
               Jpats_Simulated_Aircraft.Get_Crash_Excess_Airspeed or
               Jpats_Simulated_Aircraft.Get_Crash_Mountain_Impact;

      Wind_Speed := Jpats_Atmosphere.Surface_Wind_Velocity * 0.0823 /
                                  Sqrt(Jpats_Atmosphere.Density);
      Test_Number := Jpats_Auto_Test.Test_Number;
      if Test_Number > 800 and Test_Number < 1000 then
         Faa_Test_Drive(Test_number, Integration_Constant);
      elsif Slow_Data_Container.This_Ios_Interface.
        Test_Pattern_Number_Was_Set then
         Data_Block.Test_Pattern.Set_Name
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Test_Pattern,
            Test_Name(Test_Name_Enum'Val
                      (Slow_Data_Container.This_Ios_Interface.
                       Test_Pattern_Number)));
         Slow_Data_Container.This_Ios_Interface.
           Test_Pattern_Number_Was_Set := False;
      elsif (Slow_Data_Container.This_Ios_Interface.Test_Pattern_Number = 0) then
         Data_Block.Test_Pattern.Set_Name
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Test_Pattern,Test_Name(None));
      end if;

-- Set System Features Data Block Data
      Data_Block.System_Features.Determine_Visual_On_Off
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features,
         Ac_Position.Latitude, Ac_Position.Longitude,
         Ac_Pitch, Ac_Roll, Ac_Yaw, Integration_Constant);

      if (Slow_Data_Container.This_Ios_Interface.Video_On_Was_Set) then
         Data_Block.System_Features.Set_Visual_On_Off
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features,
            Slow_Data_Container.This_Ios_Interface.Video_On);
      end if;

      Data_Block.System_Features.Set_Morning_Readiness_Tst
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features,
         Morning_Readiness_Test_Number);

      if (Slow_Data_Container.This_Ios_Interface.
          Host_Buffer_Save_Was_Set) then
         Data_Block.System_Features.Set_Host_Buffer_Save
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features,
            Slow_Data_Container.This_Ios_Interface.Host_Buffer_Save);
         Slow_Data_Container.This_Ios_Interface.Host_Buffer_Save_Was_Set := False;
         Slow_Data_Container.This_Ios_Interface.Host_Buffer_Save := False;
         Host_Buffer_Save_Timer := 0.5;
      end if;

      if (Host_Buffer_Save_Timer <= 0.0) then
         Data_Block.System_Features.Set_Host_Buffer_Save
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features,
            False);
      else
          Host_Buffer_Save_Timer := Host_Buffer_Save_Timer - Integration_Constant;
      end if;


      Data_Block.System_Features.Set_Crash_Indicator
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features,
         Crash);

-- Set Airport Control Data Block Data
      Data_Block.Airport_Control.Set_Ramp_Traffic
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Models.Ramp_Traffic);
      Data_Block.Airport_Control.Set_Air_Traffic
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Models.Air_Traffic);
      Data_Block.Airport_Control.Set_Ground_Traffic
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Models.Ground_Traffic);
      Data_Block.Airport_Control.Set_Random_Lighting
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Random_Light_Intensities);
      Data_Block.Airport_Control.Set_Marshaller_Control
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Marshaller_Control);
      Data_Block.Airport_Control.Set_Taxiway_Light_Intensity
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Buffer_Types.Visual_If_Intensity
         (Jpats_Visual_Airport.Taxiway_Light_Intensity));
      Data_Block.Airport_Control.Set_Taxiway_Surface_Intensity
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Buffer_Types.Visual_If_Intensity
         (Jpats_Visual_Airport.Taxiway_Surface_Intensity));
      Data_Block.Airport_Control.Set_Taxiway_Marking_Intensity
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Buffer_Types.Visual_If_Intensity
         (Jpats_Visual_Airport.Taxiway_Markings_Intensity));
      Data_Block.Airport_Control.Set_Cultural_Light_Intensity
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Buffer_Types.Visual_If_Intensity
         (Jpats_Visual_Airport.Cultural_Intensity));
      Data_Block.Airport_Control.Set_Stop_Bar_Light_Select
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Stop_Bar_Active);
      Data_Block.Airport_Control.Set_Runway_Select_Number
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Visual_Runway_Number);
      Data_Block.Airport_Control.Set_Runway_Select_Letter
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Visual_Runway_Letter);
      Data_Block.Airport_Control.Set_Airport_Ident
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Visual_Airport_Code);
      Data_Block.Airport_Control.Set_Special_Switch
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Green_Light_Gun, 4);
      Data_Block.Airport_Control.Set_Special_Switch
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.Red_Light_Gun, 5);
      Data_Block.Airport_Control.Set_Special_Switch
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control,
         Jpats_Visual_Airport.White_Light_Gun, 6);

-- Set Runway Control Data Block Data
      Data_Block.Runway_Control.Set_Runway_Number
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Visual_Runway_Number);
      Data_Block.Runway_Control.Set_Runway_Letter
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Visual_Runway_Letter);
      Data_Block.Runway_Control.Set_Wet_Contamination
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Wet_Runway);
      Data_Block.Runway_Control.Set_Icy_Contamination
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Icy_Runway);
      Data_Block.Runway_Control.Set_Snow_Contamination
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Snowy_Runway);
      Data_Block.Runway_Control.Set_Slush_Contamination
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Slushy_Runway);
      Data_Block.Runway_Control.Set_Patchy_Contamination
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Patchy_Contaminant);
      Data_Block.Runway_Control.Set_Mirl_Hirl_Select
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Mirl_Hirl_Select);
      Data_Block.Runway_Control.Set_Approach_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Approach_Intensity);
      Data_Block.Runway_Control.Set_Runway_Edge_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Runway_Edge_Intensity);
      Data_Block.Runway_Control.Set_Center_Line_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Center_Line_Intensity);
      Data_Block.Runway_Control.Set_Runway_Si
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Surface_Intensity);
      Data_Block.Runway_Control.Set_Runway_Mi
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Markings_Intensity);
      Data_Block.Runway_Control.Set_Runway_Strobe_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Strobe_Intensity);
      Data_Block.Runway_Control.Set_Vasi_Papi_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Vasi_Papi_Intensity);
      Data_Block.Runway_Control.Set_Reil_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Reil_Intensity);
      Data_Block.Runway_Control.Set_Fixed_Distance_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Fixed_Distance_Intensity);
      Data_Block.Runway_Control.Set_Threshold_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Threshold_Intensity);
      Data_Block.Runway_Control.Set_Stop_Bar_Light_Select
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Stop_Bar_Active);
      Data_Block.Runway_Control.Set_Touchdown_Zone_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Td_Zone_Intensity);
      Data_Block.Runway_Control.Set_Agl_Ldin_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Agl_Intensity);
      Data_Block.Runway_Control.Set_Runway_End_Li
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.End_Lights_Intensity);
      Data_Block.Runway_Control.Set_Airport_Ident
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control,
         Jpats_Visual_Airport.Visual_Airport_Code);

-- Set Celestial Control Data Block Data
      Data_Block.Celestial_Control.Set_Manual_Sun_Position_Enable
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Manual_Sun_Position_Enable);
      Data_Block.Celestial_Control.Set_Manual_Moon_Position_Enable
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Manual_Moon_Position_Enable);
      Data_Block.Celestial_Control.Set_Atmospheric_Illuminance
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Atmospheric_Illuminance);
      Data_Block.Celestial_Control.Set_Sun_Azimuth
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Sun_Azimuth);
      Data_Block.Celestial_Control.Set_Sun_Elevation
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Sun_Elevation);
      Data_Block.Celestial_Control.Set_Moon_Phase
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Moon_Phase);
      Data_Block.Celestial_Control.Set_Moon_Elevation
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Moon_Elevation);
      Data_Block.Celestial_Control.Set_Moon_Azimuth
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Moon_Azimuth);
      Data_Block.Celestial_Control.Set_Star_Field
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control,
         Jpats_Visual_Weather.Star_Field);


-- Atmosphere Block Data
      if (Jpats_Visual_Weather.Category_Level =
          Jpats_Visual_Weather_Types.Cavu) then
         Data_Block.Atmosphere_Control.Set_Cavu_Conditions
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,True);
      else
         Data_Block.Atmosphere_Control.Set_Cavu_Conditions
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,False);
      end if;

      if (Jpats_Visual_Weather.Upper_Cloud_Layer_Overcast and
          Jpats_Visual_Weather.Lower_Cloud_Layer_Overcast) then
         Data_Block.Atmosphere_Control.Set_Solid_Cloud_Layer
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control, 2);
      elsif (Jpats_Visual_Weather.Lower_Cloud_Layer_Overcast or
             Jpats_Visual_Weather.Upper_Cloud_Layer_Overcast) then
         Data_Block.Atmosphere_Control.Set_Solid_Cloud_Layer
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control, 1);
      else
         Data_Block.Atmosphere_Control.Set_Solid_Cloud_Layer
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control, 0);
      end if;

      Data_Block.Atmosphere_Control.Set_Winter_Select
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Winter_Scene);
      Data_Block.Atmosphere_Control.Set_Scud_Enable
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Scud_Enabled);
      Data_Block.Atmosphere_Control.Set_Precipitation_Type
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Precipitation_Type);
      Data_Block.Atmosphere_Control.Set_Precipitation_Level
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Precipitation_Level);
      Data_Block.Atmosphere_Control.Set_Cirrus_Cloud_Enable
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Cirrus_Clouds);
      Data_Block.Atmosphere_Control.Set_Fog_Type
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Patch_Fog);
      Data_Block.Atmosphere_Control.Set_Blowing_Contaminant
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Airport.Blowing_Contaminant);
      if (Jpats_Visual_Weather.Lightning_Enabled) then
         Data_Block.Atmosphere_Control.Set_Lightning_Below_Cloud
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Jpats_Visual_Weather.Precipitation_Level);
      else
         Data_Block.Atmosphere_Control.Set_Lightning_Below_Cloud
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control, 0);
      end if;

      if ( Jpats_Visual_Weather.Upper_Cloud_Layer_Overcast and
           not Jpats_Visual_Weather.Lower_Cloud_Layer_Overcast ) then
         Data_Block.Atmosphere_Control.Set_Upper_Cl_Top
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control, 0);
         Data_Block.Atmosphere_Control.Set_Upper_Cl_Bottom
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control, 0);
         Data_Block.Atmosphere_Control.Set_Lower_Cl_Top
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Jpats_Visual_Weather.Upper_Cloud_Top);
         Data_Block.Atmosphere_Control.Set_Lower_Cl_Bottom
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Jpats_Visual_Weather.Upper_Cloud_Ceiling);
      else
         Data_Block.Atmosphere_Control.Set_Upper_Cl_Top
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Jpats_Visual_Weather.Upper_Cloud_Top);
         Data_Block.Atmosphere_Control.Set_Upper_Cl_Bottom
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Jpats_Visual_Weather.Upper_Cloud_Ceiling);
         Data_Block.Atmosphere_Control.Set_Lower_Cl_Top
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Jpats_Visual_Weather.Lower_Cloud_Top);
         if (Jpats_Visual_Weather.Lower_Cloud_Layer_Scattered or
             Jpats_Visual_Weather.Lower_Cloud_Layer_Broken or
             Jpats_Visual_Weather.Lower_Cloud_Layer_Overcast) then
            Data_Block.Atmosphere_Control.Set_Lower_Cl_Bottom
              (Slow_Data_Container.This_Slow_Send_Subsystem.
               The_Atmosphere_Control,
               Jpats_Visual_Weather.Lower_Cloud_Ceiling +
               Integer(Airport_Position.Altitude));
         else
            Data_Block.Atmosphere_Control.Set_Lower_Cl_Bottom
              (Slow_Data_Container.This_Slow_Send_Subsystem.
               The_Atmosphere_Control,
               Jpats_Visual_Weather.Lower_Cloud_Ceiling);
         end if;
      end if;

      Data_Block.Atmosphere_Control.Set_Fl1_Visibility
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Visibility_Nm);
      if (Jpats_Visual_Weather.Fog_Active or
          Jpats_Visual_Weather.Patch_Fog) then
         Data_Block.Atmosphere_Control.Set_Fog_Height
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Jpats_Visual_Weather.Fog_Top +
            Integer(Airport_Position.Altitude));
      else
         Data_Block.Atmosphere_Control.Set_Fog_Height
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control, Jpats_Visual_Weather.Fog_Top);
      end if;
      Data_Block.Atmosphere_Control.Set_Touchdown_RVR
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Rwy_Vis_Range_ft);
      Data_Block.Atmosphere_Control.Set_Midpoint_RVR
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Rwy_Vis_Range_ft);
      Data_Block.Atmosphere_Control.Set_Roll_Out_RVR
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Rwy_Vis_Range_ft);
      Data_Block.Atmosphere_Control.Set_Winds_Aloft_2_Direction
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Atmosphere.Wind_Aloft_Direction);
      Data_Block.Atmosphere_Control.Set_Winds_Aloft_1_Direction
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Atmosphere.Wind_Aloft_Direction);
      Data_Block.Atmosphere_Control.Set_Surface_Winds_Direction
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Atmosphere.Surface_Wind_Direction);
      Data_Block.Atmosphere_Control.Set_Winds_Aloft_2_Speed
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Atmosphere.Wind_Aloft_velocity);
      Data_Block.Atmosphere_Control.Set_Winds_Aloft_1_Speed
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Atmosphere.Wind_aloft_velocity);
      Data_Block.Atmosphere_Control.Set_Surface_Winds_Speed
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         wind_speed);
      Data_Block.Atmosphere_Control.Set_Winds_Aloft_2_Altitude
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         40000);
      if ((Jpats_Visual_Weather.Upper_Cloud_Top = 0) or
          (Jpats_Visual_Weather.Lower_Cloud_Top = 0)) then
         Data_Block.Atmosphere_Control.Set_Winds_Aloft_1_Altitude
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
            39000);
      else
         Data_Block.Atmosphere_Control.Set_Winds_Aloft_1_Altitude
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
            Jpats_Visual_Weather.Upper_Cloud_Top);
      end if;
      if ((Jpats_Visual_Weather.Upper_Cloud_Top = 0) and
          (Jpats_Visual_Weather.Lower_Cloud_Top = 0)) then
         Data_Block.Atmosphere_Control.Set_Surface_Winds_Altitude
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
            38000);
      elsif ((Jpats_Visual_Weather.Upper_Cloud_Top /= 0) and
             (Jpats_Visual_Weather.Lower_Cloud_Top = 0)) then
         Data_Block.Atmosphere_Control.Set_Surface_Winds_Altitude
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
            Jpats_Visual_Weather.Upper_Cloud_Top);
      else
         Data_Block.Atmosphere_Control.Set_Surface_Winds_Altitude
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
            Jpats_Visual_Weather.Lower_Cloud_Top);
      end if;
      Data_Block.Atmosphere_Control.Set_Weather_Elevation
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Integer(Airport_Position.Altitude));
      Data_Block.Atmosphere_Control.Set_Time_Of_Day_Mode
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Time_Of_Day);
      Data_Block.Atmosphere_Control.Set_Continuous_Tod
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control,
         Jpats_Visual_Weather.Current_time);
      if (Slow_Data_Container.This_Ios_Interface.Sea_State_Was_Set) then
         Data_Block.Atmosphere_Control.Set_Sea_States
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,
            Slow_Data_Container.This_Ios_Interface.Sea_State);
      end if;
      if (Data_Block.Atmosphere_Control.Time_Of_Day_Mode
          (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control)
          = 1) then
         Data_Block.Atmosphere_Control.Set_Ambient_Level
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,3);
      elsif (Data_Block.Atmosphere_Control.Time_Of_Day_Mode
          (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control)
          = 2) then
         Data_Block.Atmosphere_Control.Set_Ambient_Level
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,5);
      elsif (Data_Block.Atmosphere_Control.Time_Of_Day_Mode
          (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control)
          = 3) then
         Data_Block.Atmosphere_Control.Set_Ambient_Level
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,3);
      elsif (Data_Block.Atmosphere_Control.Time_Of_Day_Mode
          (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control)
          = 4) then
         Data_Block.Atmosphere_Control.Set_Ambient_Level
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Atmosphere_Control,1);
      end if;

-- Configurable AIrport Data Block
      if (Jpats_Visual_Airport.Visual_Airport_Code = "****" and
          Slow_Data_Container.This_Ios_Interface.Config_Apt_Enable) then
         Data_Block.Config_Apt_Cntl.Set_Use_Data
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            True);
         --         Data_Block.Config_Apt_Cntl.Set_Runway_Number
         --         (Slow_Data_Container.This_Slow_Send_Subsystem.
         --          The_Config_Apt_Cntl,
         --          Jpats_Visual_Airport.Runway_Number);
         Data_Block.Config_Apt_Cntl.Set_Building_Location
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Terminal_Location);
         Data_Block.Config_Apt_Cntl.Set_Reil_Lighting
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.REILS);
         Data_Block.Config_Apt_Cntl.Set_Civil_Rotating_Beacon
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Rotating_Beacon);
         Data_Block.Config_Apt_Cntl.Set_Terrain_Type
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Terrain_Type);
         Data_Block.Config_Apt_Cntl.Set_Runway_Width
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Ca_Runway_Width);
         Data_Block.Config_Apt_Cntl.Set_Runway_Length
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Ca_Runway_Length);
         Data_Block.Config_Apt_Cntl.Set_Approach_Path_Indicator
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Vasi_Type);
         Data_Block.Config_Apt_Cntl.Set_Approach_Light_System
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Approach_Light_System);
         Data_Block.Config_Apt_Cntl.Set_Runway_Markings
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Runway_Markings);
         Data_Block.Config_Apt_Cntl.Set_Cultural_Lights
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Cultural_Lights);
         Data_Block.Config_Apt_Cntl.Set_Primary_Runway_Position
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Runway_Position);
         Data_Block.Config_Apt_Cntl.Set_Default_Runway_True_Hdg
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Runway_True_Heading);
         Data_Block.Config_Apt_Cntl.Set_Default_Runway_Gs_Angle
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Glideslope_Angle);
         Data_Block.Config_Apt_Cntl.Set_Default_Runway_Mag_Hdg
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            Jpats_Visual_Airport.Runway_Mag_Heading);
      else
         Data_Block.Config_Apt_Cntl.Set_Use_Data
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl,
            False);
      end if;

-- Test Pattern Block Setup
      if ( Data_Block.Test_Pattern.Name
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern)
            /= Test_Name(None)) then
         Data_Block.Test_Pattern.Set_Use_Data
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern,
            True);
         Runway_Position.Latitude := Active_Runway.Lat;
         Runway_Position.Longitude := Active_Runway.Lon;
         Runway_Position.Altitude := Active_Airport.Elv;
         Data_Block.Test_Pattern.Set_Position
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern,
            Runway_Position);
         Data_Block.Test_Pattern.Set_Heading
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern,
            Active_Runway.Hdg);
      else
         Data_Block.Test_Pattern.Set_Use_Data
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern,
            False);
      end if;

      Data_Block.System_Features.Pack
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_System_Features);
      Data_Block.Airport_Control.Pack
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Airport_Control);
      Data_Block.Celestial_Control.Pack
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Celestial_Control);
      Data_Block.Runway_Control.Pack
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Runway_Control);
      Data_Block.Atmosphere_Control.Pack
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Atmosphere_Control);
      Data_Block.Config_Apt_Cntl.Pack
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Config_Apt_Cntl);
      Data_Block.Test_Pattern.Pack
        (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern);

   end Update;

-- Method Faa Test Drive
   procedure Faa_Test_Drive
     (Test_Number : in Integer;
      Integration_Constant : in Float) is
      Test_Initializing : Boolean;
      Test_Running : Boolean;
      A_Time_Of_Day : Integer;
      An_Intensity : Integer;
      An_Override : Integer;
      An_Rvr : Integer;
      A_Flag : Boolean;
   begin
 --     Test_Initializing := Jpats_Auto_Test.Get_Stabilizing_Test_Status;
  --    Test_Running := Jpats_Auto_Test.Get_Running_Test_Status;

      case Test_Number is
         when 811 =>
               An_Override := 5;
               Jpats_Visual_Weather.Ios_If.
                 Set_Weather_Override
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Override'Address));
               A_Time_Of_Day := 3;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               An_Intensity := 5;
               Jpats_Visual_Airport.Ios_If.
                 Set_All_Brightness_Intensity
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
               A_Flag := True;
               Jpats_Visual_Airport.Ios_If.
                 Set_All_Brightness_Flag
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Flag'Address));
         when 812 =>
               Morning_Readiness_Test_Number := 1;
         when 813 =>
               A_Time_Of_Day := 3;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               An_Rvr := 1200;
               Jpats_Visual_Weather.Ios_If.Set_Rvr_Ft
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Rvr'Address));
               An_Intensity := 5;
               Jpats_Visual_Airport.Ios_If.
                 Set_All_Brightness_Intensity
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
               A_Flag := True;
               Jpats_Visual_Airport.Ios_If.
                 Set_All_Brightness_Flag
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Flag'Address));
         when 814 =>
               Morning_Readiness_Test_Number := 3;
         when 815 =>
               Data_Block.Test_Pattern.Set_Name
                 (Slow_Data_Container.This_Slow_Send_Subsystem.
                  The_Test_Pattern, Test_Name(Oc30));
         when 821 =>
               A_Time_Of_Day := 2;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
         when 822 =>
               A_Time_Of_Day := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               Data_Block.Test_Pattern.Set_Name
                 (Slow_Data_Container.This_Slow_Send_Subsystem.
                  The_Test_Pattern, Test_Name(cr));
         when 823 =>
               A_Time_Of_Day := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               Data_Block.Test_Pattern.Set_Name
                 (Slow_Data_Container.This_Slow_Send_Subsystem.
                  The_Test_Pattern, Test_Name(Hlit));
         when 824 =>
               A_Time_Of_Day := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               An_Intensity := 1;
               Jpats_Visual_Airport.Ios_If.
                 Set_Runway_Surface_Intensity
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
               An_Intensity := 5;
               Jpats_Visual_Airport.Ios_If.Set_Runway_Markings_Intensity
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
               An_Intensity := 0;
               Jpats_Visual_Airport.Ios_If.
                 Set_Runway_Center_Line_Li
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
               An_Intensity := 0;
               Jpats_Visual_Airport.Ios_If.Set_Td_Zone_Li
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
         when 825 =>
               A_Time_Of_Day := 4;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               A_Flag := True;
               Jpats_Visual_Airport.Ios_If.
                 Set_All_Brightness_Flag
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Flag'Address));
               An_Intensity := 5;
               Jpats_Visual_Airport.Ios_If.Set_All_Brightness_Intensity

                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
               Data_Block.Airport_Control.Set_Special_Switch
                 (Slow_Data_Container.This_Slow_Send_Subsystem.
                  The_Airport_Control, True, 1);
         when 826 =>
               A_Time_Of_Day := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               Data_Block.Test_Pattern.Set_Name
                 (Slow_Data_Container.This_Slow_Send_Subsystem.
                  The_Test_Pattern, Test_Name(Lcrc));
         when 827 =>
               A_Time_Of_Day := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               An_Override := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Weather_Override
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Override'Address));
         when 828 =>
            if not Test_Started then
                  Auto_Test_Timer := 0.0;
                  Test_Started := True;
            end if;
            An_Override := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Weather_Override
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Override'Address));

               if (Auto_Test_Timer <= 30.0) then
                  A_Time_Of_Day := 1;
                  Jpats_Visual_Weather.Ios_If.
                    Set_Time_Of_Day
                    (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
                  Data_Block.Test_Pattern.Set_Name
                    (Slow_Data_Container.This_Slow_Send_Subsystem.
                     The_Test_Pattern, Test_Name(Pe1k));
               elsif (Auto_Test_Timer <= 60.0) then
                  A_Time_Of_Day := 4;
                  Jpats_Visual_Weather.Ios_If.
                    Set_Time_Of_Day
                    (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
                  Data_Block.Test_Pattern.Set_Name
                    (Slow_Data_Container.This_Slow_Send_Subsystem.
                     The_Test_Pattern, Test_Name(Le4k));
               elsif (Auto_Test_Timer > 60.0) then
                  A_Time_Of_Day := 3;
                  Jpats_Visual_Weather.Ios_If.
                    Set_Time_Of_Day
                    (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               end if;
               Auto_Test_Timer := Auto_Test_Timer + Integration_Constant;
         when 829 =>
               A_Time_Of_Day := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Time_Of_Day
                 (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
               An_Override := 1;
               Jpats_Visual_Weather.Ios_If.
                 Set_Weather_Override
                 (JPATS_IOS_Interface.Convert_To_Set_Value (An_Override'Address));
         when 901 | 902 | 903 =>
               Data_Block.Test_Pattern.Set_Name
                 (Slow_Data_Container.This_Slow_Send_Subsystem.
                  The_Test_Pattern, Test_Name(Tput));
         when others =>
            Morning_Readiness_Test_Number := 0;
            Data_Block.Test_Pattern.Set_Name
              (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern,
               Test_Name(None));

      end case;

      if Test_Number = 999 then
         Morning_Readiness_Test_Number := 0;
         Data_Block.Test_Pattern.Set_Name
           (Slow_Data_Container.This_Slow_Send_Subsystem.The_Test_Pattern,
            Test_Name(None));
         A_Time_Of_Day := 1;
         Jpats_Visual_Weather.Ios_If.Set_Time_Of_Day
           (JPATS_IOS_Interface.Convert_To_Set_Value (A_Time_Of_Day'Address));
         An_Override := 1;
         Jpats_Visual_Weather.Ios_If.Set_Weather_Override
           (JPATS_IOS_Interface.Convert_To_Set_Value (An_Override'Address));
         An_Intensity := 1;
         Jpats_Visual_Airport.Ios_If.
           Set_Runway_Contamination_Select
           (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
         A_Flag := True;
         Jpats_Visual_Airport.Ios_If.
           Set_All_Brightness_Flag(JPATS_IOS_Interface.Convert_To_Set_Value (A_Flag'Address));
         An_Intensity := 3;
         Jpats_Visual_Airport.Ios_If.
           Set_All_Brightness_Intensity
           (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
         Jpats_Visual_Airport.Ios_If.
           Set_Runway_Surface_Intensity
           (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
         Jpats_Visual_Airport.Ios_If.
           Set_Runway_Markings_Intensity
           (JPATS_IOS_Interface.Convert_To_Set_Value (An_Intensity'Address));
         Data_Block.Airport_Control.Set_Special_Switch
           (Slow_Data_Container.This_Slow_Send_Subsystem.
            The_Airport_Control, False, 1);
         Test_Started := False;
      end if;
   end Faa_Test_Drive;


end JPATS_Visual_Buffer.Slow_Data_Controller;




