-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  James F. Narrin
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;

package body Data_Block.HOT_Request is

   Gear_X_Offset : Jpats_Visual_Buffer_Types.Hot_Long_Float_Array :=
     (-0.5, -0.5, 7.0, -14.0);
   Gear_Y_Offset : Jpats_Visual_Buffer_Types.Hot_Long_Float_Array :=
     (4.17, -4.17, 0.0, 0.0);
   Predicted_Lat : Long_Float;
   Predicted_Lon : Long_Float;
   Last_Ac_Lat : Long_Float;
   Last_Ac_Lon : Long_Float;
   Ft_To_Deg : constant := 0.000002742988;

-- Access Number_HOT_Points
   function  Number_HOT_Points
     (An_Instance : in Instance)
      return Integer is
   begin
      return An_Instance.Number_HOT_Points;
   end Number_HOT_Points;

   procedure Set_Number_HOT_Points
     (An_Instance : in out Instance;
      A_Number_Of_Points : in Integer) is
   begin
      An_Instance.Number_Hot_Points := A_Number_Of_Points;
   end Set_Number_HOT_Points;

-- Access HOT_ID
   function  HOT_ID
     (An_Instance : in Instance)
      return Integer is
   begin
      return An_Instance.HOT_Id;
   end HOT_ID;

   procedure Set_HOT_ID
     (An_Instance : in out Instance;
      An_Id : in Integer) is
   begin
      An_Instance.Hot_Id := An_Id;
   end Set_HOT_ID;

-- Access Point_Latitude
   function  Point_Latitude
     (An_Instance : in Instance;
      Element : in Integer)
      return Long_Float is
   begin
      return An_Instance.Point_Latitude(Element);
   end Point_Latitude;

   procedure Set_Point_Latitude
     (An_Instance : in out Instance;
      Element : in Integer;
      A_Latitude : in Long_Float) is
   begin
      An_Instance.Point_Latitude(Element) := A_Latitude;
   end Set_Point_Latitude;

-- Access Point_Longitude
   function  Point_Longitude
     (An_Instance : in Instance;
      Element : in Integer)
      return Long_Float is
   begin
      return An_Instance.Point_Longitude(Element);
   end Point_Longitude;

   procedure Set_Point_Longitude
     (An_Instance : in out Instance;
      Element : in Integer;
      A_Longitude : in Long_Float) is
   begin
      An_Instance.Point_Longitude(Element) := A_Longitude;
   end Set_Point_Longitude;

-- Method Predict_AC_Position
   procedure Predict_AC_Position
     (An_Instance : in out Instance;
      Ac_Latitude : in Long_Float;
      Ac_Longitude : in Long_Float;
      Integration_Constant : in Float) is
      Delta_Lat : Long_Float;
      Delta_Lon : Long_Float;
   begin
      Delta_Lat := abs(Ac_Latitude - Last_Ac_Lat);
      Delta_Lon := abs(Ac_Longitude - Last_Ac_Lon);

      if (Delta_Lat > 0.1 or Delta_Lon > 0.1) then
         Last_Ac_Lat := Ac_Latitude;
         Last_Ac_Lon := Ac_Longitude;
      end if;

      Predicted_Lat := Ac_Latitude +
        (Ac_Latitude - Last_Ac_Lat)*2.0;   -- Predict two time steps in the future
      Predicted_Lon := Ac_Longitude +
        (Ac_Longitude - Last_Ac_Lon)*2.0;  -- Predict two time steps in the future

      Last_Ac_Lat := Ac_Latitude;
      Last_Ac_Lon := Ac_Longitude;
   end Predict_AC_Position;

-- Method Gear_Lat_Lon_Position
   procedure Gear_Lat_Lon_Position
     (An_Instance : in out Instance;
      Ac_Latitude : in Long_Float;
      Ac_Heading : in Float) is
      Coslat : Float;
      Sinhdg : Float;
      Coshdg : Float;
   begin
      Coslat := 1.0 / Cos(Float(Ac_Latitude), 360.0);
      Sinhdg := Sin(Ac_Heading, 360.0);
      Coshdg := Cos(Ac_Heading, 360.0);

      for I in 1..An_Instance.Number_Hot_Points loop
         An_Instance.Point_Latitude(I) := Predicted_Lat +
           (Gear_X_Offset(I) * Long_Float(Coshdg) -
            Gear_Y_Offset(I) * Long_Float(Sinhdg))
           * Long_Float(Ft_To_Deg);
         An_Instance.Point_Longitude(I) := Predicted_Lon +
           (Gear_X_Offset(I) * Long_Float(Sinhdg) +
            Gear_Y_Offset(I) * Long_Float(Coshdg))
           * Long_Float(Ft_To_Deg * Coslat);
      end loop;

   end Gear_Lat_Lon_Position;

   procedure Initialize
     (An_Instance : in out Instance) is
   begin
      An_Instance.Word_Count := 9;
      An_Instance.Data_Block_Code := 16#09000000#;
      An_Instance.Use_Data := True;

      An_Instance.Number_Hot_Points := 4;
      An_Instance.Hot_Id := 100;

      for I in 1..An_Instance.Number_Hot_Points loop
         An_Instance.Point_Latitude(I) := 0.0;
         An_Instance.Point_Longitude(I) := 0.0;
      end loop;

      for I in 1..9 loop
         An_Instance.The_Hot_Request_Block(I) := 0;
      end loop;

   end Initialize;

   procedure Pack
     (An_Instance : in out Instance) is
   begin
      for I in 1..An_Instance.Word_Count loop
         An_Instance.The_Hot_Request_Block(I) := 0;
      end loop;

      An_Instance.Hot_Id := An_Instance.Hot_Id + 1;
      if (An_Instance.Hot_Id >= 299) then
         An_Instance.Hot_Id := 100;
      end if;

      An_Instance.The_Hot_Request_Block(1) :=
        Jpats_Visual_Buffer_Types.Unsigned_Int(An_Instance.Data_Block_Code);

      if (An_Instance.Number_Hot_Points > 4) then
         An_Instance.Number_Hot_Points := 4;
      end if;
      if (An_Instance.Number_Hot_Points > 0) then
         An_Instance.The_Hot_Request_Block(1) :=
           An_Instance.The_Hot_Request_Block(1) +
           Jpats_Visual_Buffer_Types.Unsigned_Int
           (An_Instance.Number_Hot_Points - 1) * 16#00010000#;
      end if;

      An_Instance.The_Hot_Request_Block(1) :=
        An_Instance.The_Hot_Request_Block(1) +
        Jpats_Visual_Buffer_Types.Unsigned_Int(An_Instance.Hot_Id);

      for I in 1..An_Instance.Number_Hot_Points loop
         An_Instance.The_Hot_Request_Block(I*2) :=
           Jpats_Visual_Buffer_Types.Unsigned_Int
           (Integer(An_Instance.Point_Latitude(I) *
            Jpats_Visual_Buffer_Types.Bams_32bit));
         An_Instance.The_Hot_Request_Block(I*2+1) :=
           Jpats_Visual_Buffer_Types.Unsigned_Int
           (Integer(An_Instance.Point_Longitude(I) *
            Jpats_Visual_Buffer_Types.Bams_32bit));
      end loop;

      An_Instance.Word_Count := An_Instance.Number_Hot_Points*2+1;
   end Pack;

   function Data_Word
     (An_Instance : in Instance)
      return Jpats_Visual_Buffer_Types.Block_Array is
      Num_Words : Natural;
   begin
      Num_Words := An_Instance.Number_Hot_Points * 2 + 1;
      return An_Instance.The_Hot_Request_Block(1..Num_Words);
   end Data_Word;


end Data_Block.HOT_Request;
