-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Jim F. Narrin
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;
with Ada.Text_Io;
use Ada.Text_Io;
with Ada.Float_Text_Io;
use Ada.Float_Text_Io;
with Visual_Airport_Utilities;
with Ada.Unchecked_Deallocation;
with Simulation_Dictionary;

package body Scene_Collection is

   procedure Delete_Scene is new Ada.Unchecked_Deallocation
     (Scene_Type, Scene_Access);

   -- This procedure reads in the data file and initializes the list
   -- of custom scenes.

   procedure Initialize_List is
      File : Ada.Text_Io.File_Type;
      Current : Scene_Access := null;
      New_Airport : Scene_Access := null;
      I : Integer := 0;
      C : Character := ' ';
      Latitude_In : Float := 0.0;
      Longitude_In : Float := 0.0;
--      File_Name : String(1..100);
   begin

      Scene_List := null;
      Current := null;
      New_Airport := null;

      New_Airport := new Scene_Type;
      New_Airport.Next := null;
      New_Airport.Prev := Current;
      New_Airport.Airport_Code := "    ";
      New_Airport.Runway := "    ";
      New_Airport.Visual_Code := "****";
      New_Airport.Runway_Position.Latitude := 0.0;
      New_Airport.Runway_Position.Longitude := 0.0;
      New_Airport.Runway_Heading := 0.0;
      New_Airport.Runway_Position.Altitude := 0.0;

      Current := New_Airport;
      Scene_List:= New_Airport;

      New_Airport := new Scene_Type;
      New_Airport.Next := null;
      New_Airport.Prev := Current;
      New_Airport.Airport_Code := "KTST";
      New_Airport.Runway := "27  ";
      New_Airport.Visual_Code := "ktst";
      New_Airport.Runway_Position.Latitude := 45.0;
      New_Airport.Runway_Position.Longitude := -44.957526988;
      New_Airport.Runway_Heading := 270.0;
      New_Airport.Runway_Position.Altitude := 1000.0;

      Current.Next := New_Airport;
      Current := New_Airport;

      declare
         File_Name : String := Simulation_Dictionary.
           Lookup("Visual_Airport_Dir") &
           "apttbl.dat";
      begin
         Open (File, Mode=>Ada.Text_Io.In_File,
               Name=>File_Name);

         begin
            while not End_Of_File(File) loop
               New_Airport := new Scene_Type;
               New_Airport.Next := null;
               New_Airport.Prev := Current;
               I := 0;
               Get(File, C);
               while C = ' ' and not End_Of_File(File) loop
                  Get(File, C);
               end loop;
               if End_Of_File(File) then
                  exit;
               end if;
               while I < 4 and then C /= ' ' loop
                  if C /= ' ' then
                     I := I + 1;
                     New_Airport.Airport_Code(I) := C;
                  end if;
                  Get(File, C);
               end loop;
               while I < 4 loop
                  I := I + 1;
                  New_Airport.Airport_Code(I) := ' ';
               end loop;
               loop
                  Get(File, C);
                  exit when C /= ' ';
               end loop;

               I := 1;
               New_Airport.Runway(I) := C;
               while I < 3 and then C /= ' ' loop
                  Get(File, C);
                  if C /= ' ' then
                     I := I + 1;
                     New_Airport.Runway(I) := C;
                  end if;
               end loop;
               while I < 4 loop
                  I := I + 1;
                  New_Airport.Runway(I) := ' ';
               end loop;
               loop
                  Get(File, C);
                  exit when C /= ' ';
               end loop;

               I := 1;
               New_Airport.Visual_Code(I) := C;
               while I < 4 and then C /= ' ' loop
                  Get(File, C);
                  if C /= ' ' then
                     I := I + 1;
                     New_Airport.Visual_Code(I) := C;
                  end if;
               end loop;
               while I < 4 loop
                  I := I + 1;
                  New_Airport.Visual_Code(I) := ' ';
               end loop;

               Get(File, Latitude_In);
               New_Airport.Runway_Position.Latitude := Long_Float(Latitude_In);
               Get(File, Longitude_In);
               New_Airport.Runway_Position.Longitude :=
                 Long_Float(Longitude_In);
               Get(File, New_Airport.Runway_Heading);
               Get(File, New_Airport.Runway_Position.Altitude);

               if (Current = null) then
                  Current := New_Airport;
                  Scene_List := New_Airport;
               else
                  Current.Next := New_Airport;
                  Current := New_Airport;
               end if;
            end loop;

         exception
            when End_Error | Data_Error =>
               Put_Line("Error Reading From File");
               Current := Scene_List.Next.Next;
               while Current /= null loop
                  New_Airport := Current.Next;
                  Delete_Scene(Current);
                  Current := New_Airport;
               end loop;
         end;

         Close(File);
      end;
   Exception
      when Name_Error | Use_Error =>
         Put_Line("Invalid File Name");

   end Initialize_List;

    -- This function searches the list for the desired airport to determine if
    -- the airport is a custom scene.

    procedure Find_Closest_Airport
        (Aircraft_Lat : in Lat_Long_Types.Latitude_Degrees;
         Aircraft_Lon : in Lat_Long_Types.Longitude_Degrees;
         Airport_Name : out Jpats_Visual_Airport_Types.Name_Type;
         Airport_Model : out Jpats_Visual_Airport_Types.Name_Type;
         Runway_Number : out Jpats_Visual_Airport_Types.Name_Type;
         Runway_Position : out Lat_Long_Types.Position;
         Runway_Heading: out Jpats_Visual_Airport_Types.Heading_Type) is
        Scene_Radius : constant := 50.0;
        distance : float;
        last_distance : float;
        coslat : Long_float;
        lat_diff : Long_float;
        lon_diff : Long_float;
        Index : Integer;
        Current : Scene_Access;
        Closest_Airport : Scene_Access;
    begin
       Coslat := Long_Float(cos(Float(Aircraft_Lat * 0.017453292)));
       Last_Distance := 1000000.0;
       Current := Scene_List;
       Closest_Airport := null;

       while (Current /= null) loop

          Lat_Diff := Aircraft_Lat - Current.Runway_Position.Latitude;

          Lon_Diff := Aircraft_Lon - Current.Runway_Position.Longitude;
          Lon_Diff := Visual_Airport_Utilities.Xn180(Lon_Diff);
          Lon_Diff := Lon_Diff * Coslat;

          Distance := sqrt(Float(lat_diff**2 + lon_diff**2));

          if distance < Last_Distance then
             Closest_Airport := Current;
             last_distance := distance;
          end if;

          Current := Current.Next;
       end loop;

       if Closest_Airport /= null then
          Airport_Name := Closest_Airport.Airport_Code;
          Airport_Model := Closest_Airport.Visual_Code;
          Runway_Number := Closest_Airport.Runway;
          Runway_Position.Latitude :=
            Closest_Airport.Runway_Position.Latitude;
          Runway_Position.Longitude :=
            Closest_Airport.Runway_Position.Longitude;
          Runway_Position.Altitude :=
            Closest_Airport.Runway_Position.Altitude;
          Runway_Heading := Closest_Airport.Runway_Heading;
       else
          Airport_Name := "    ";
          Airport_Model := "    ";
          Runway_Number := "    ";
          Runway_Position.Latitude := 0.0;
          Runway_Position.Longitude := 0.0;
          Runway_Position.Altitude := 0.0;
          Runway_Heading := 0.0;
        end if;

    end Find_Closest_Airport;

    procedure Set_Airport
        (Airport_Name : in Jpats_Visual_Airport_Types.Name_Type;
         Airport_Model : out Jpats_Visual_Airport_Types.Name_Type;
         Runway_Number : in Jpats_Visual_Airport_Types.Name_Type;
         Runway_Position : in Lat_Long_Types.Position;
         Runway_Heading : in Jpats_Visual_Airport_Types.Heading_Type) is
       Index : integer;
       Current : Scene_Access;
       Select_Airport : Scene_Access;
    begin
       Current := Scene_List.Next;
       while ((Current /= null) and then
              (Airport_Name /= Current.Airport_Code)) loop
          Current := Current.Next;
       end loop;

       if Current = null then
          Current := Scene_List;
       end if;

       if Current /= null then
          Current.Airport_Code := Airport_Name;
          Current.Runway := Runway_Number;
          Current.Runway_Position.Latitude := Runway_Position.Latitude;
          Current.Runway_Position.Longitude := Runway_Position.Longitude;
          Current.Runway_Position.Altitude := Runway_Position.Altitude;
          Current.Runway_Heading := Runway_Heading;

          Airport_Model := Current.Visual_Code;
       end if;
    end Set_Airport;



end Scene_Collection;



