-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  James F. Narrin
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with JPATS_Visual_Airport.Container;
with JPATS_Visual_Airport_Types;
use Jpats_Visual_Airport_Types;
with Visual_Airport_Utilities;
use Visual_Airport_Utilities;
with Jpats_Visual_Weather;
with Jpats_Simulated_Aircraft;
with Jpats_Reposition;
with Jpats_Radio_Db_If_Types;
with Jpats_Simulated_Aircraft_Types;
with Jpats_Reposition_Types;
with Marshaller;
with Configurable_Airport;
with Taxiway_Lighting;
with Runway;
with Runway_Lighting;
with Physical_Description;
with Surface_Conditions;
with Contamination;
with Scene_Lighting;
with Scene_Collection;
with Config_Collection;
with Jpats_Radio_Db_If_Types;
with Jpats_Reposition;
with Lat_Long_Types;
with Jpats_Visual_Airport.Ios_If;
with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;

package body JPATS_Visual_Airport.Controller is

   Last_Airport : Jpats_Visual_Airport_Types.Name_Type := "    ";
   Last_Repo_Airport : Jpats_Visual_Airport_Types.Name_Type := "    ";
   Last_Repo_Runway : Jpats_Visual_Airport_Types.Name_Type := "    ";
   Last_Pass_Airport : Jpats_Visual_Airport_Types.Name_Type := "    ";
   Last_Pass_Runway : Jpats_Visual_Airport_Types.Name_Type := "    ";
   Last_Time_Of_Day : Natural := 2;
   Last_Ifr : Boolean := false;
   Day_Light_Timer : Float := 0.0;
   Approach_Strobe_Intensity : Jpats_Visual_Airport_Types.Intensity_Type := 0;
   Light_Intensity : Jpats_Visual_Airport_Types.Intensity_Type := 0;
   Night_Intensity : Jpats_Visual_Airport_Types.Intensity_Type := 3;
   Change_Lighting : Boolean := false;
   Generic_Compile_Timer : Float := 0.0;

   Ios : Container.Ios_Interface_Instance renames
     Container.This_Ios_Interface;

-- Method Set_Day_Light_Timer
   procedure Set_Day_Light_Timer
     (Integration_Constant : in Float;
      Time_Of_Day : in Natural;
      IFR_Conditions : in Boolean) is
   begin
      if (Container.This_Subsystem.ICAO_Code /= Last_Airport or
          Time_Of_Day /= Last_Time_Of_Day or
          IFR_Conditions /= Last_IFR) then
         Day_Light_Timer := 5.0;
         Last_Airport := Container.This_Subsystem.ICAO_Code;
         Last_Time_Of_Day := Time_Of_Day;
         Last_IFR := IFR_Conditions;
      end if;

      if Day_Light_Timer > 0.0 then
         Day_Light_Timer :=
           Day_Light_Timer - Integration_Constant;
      end if;
   end Set_Day_Light_Timer;

-- Method Set_Lighting_Level
   procedure Set_Lighting_Level
     (Time_Of_Day: in Natural;
      IFR_Conditions : in Boolean) is
   begin

      if Day_Light_Timer > 0.0 then
         if Time_Of_Day = 1 then  -- Dawn
            if IFR_Conditions then
               Approach_Strobe_Intensity := 4;
               Light_Intensity := 4;
            else
               Approach_Strobe_Intensity := 2;
               Light_Intensity := 2;
            end if;
         elsif Time_Of_Day = 2 then  -- Day
            if IFR_Conditions then
               Approach_Strobe_Intensity := 5;
               Light_Intensity := 5;
            else
               Approach_Strobe_Intensity := 3;
               Light_Intensity := 3;
            end if;
         elsif Time_Of_Day = 3 then  -- Dusk
            if IFR_Conditions then
               Approach_Strobe_Intensity := 4;
               Light_Intensity := 4;
            else
               Approach_Strobe_Intensity := 2;
               Light_Intensity := 2;
            end if;
         else  -- Night
            if IFR_Conditions then
               Approach_Strobe_Intensity := 3;
               Light_Intensity := 3;
            else
               Approach_Strobe_Intensity := 1;
               Light_Intensity := 1;
            end if;
         end if;
         Change_Lighting := True;
      end if;

      if Container.This_Subsystem.All_Bright then
         Approach_Strobe_Intensity :=
           Container.This_Subsystem.All_Bright_Intensity;
         Light_Intensity :=
           Container.This_Subsystem.All_Bright_Intensity;
         Change_Lighting := True;
--         Container.This_Subsystem.All_Bright := False;
      end if;

   end Set_Lighting_Level;


-- Method Initialize
   procedure Initialize is
   begin
      Container.This_Subsystem.Position.Latitude := 29.367450;
      Container.This_Subsystem.Position.Longitude := -100.787897;
      Container.This_Subsystem.Position.Altitude := 1081.0;
      Container.This_Subsystem.Icao_Code := "KDLF";
      Container.This_Subsystem.Visual_Airport_Code := "kdlf";
      Container.This_Subsystem.Random_Light_Intensities_On := False;
      Container.This_Subsystem.All_Bright := False;
      Container.This_Subsystem.All_Bright_Intensity := 0;
      Container.This_Subsystem.Visual_Debug_Position := False;

      Configurable_Airport.Initialize
        (Container.This_Subsystem.The_Configurable_Airport);
      Taxiway_Lighting.Initialize
        (Container.This_Subsystem.The_Taxiway);
      Runway.Initialize
        (Container.This_Subsystem.The_Runway);
      Scene_Lighting.Initialize
        (Container.This_Subsystem.The_Scene_Lighting);
      Marshaller.Initialize
        (Container.This_Subsystem.The_Marshaller);

      Scene_Collection.Initialize_List;
      Config_Collection.Initialize_List;

      Container.This_Ios_Interface.Random_Runway_Lighting := false;
      Container.This_Ios_Interface.MIRL_HIRL_Light_Select := false;
      Container.This_Ios_Interface.Stop_Bar_Lights_Active := false;
      Container.This_Ios_Interface.Approach_LI := 0;
      Container.This_Ios_Interface.Runway_Edge_LI := 3;
      Container.This_Ios_Interface.Runway_Center_Line_LI := 3;
      Container.This_Ios_Interface.Runway_End_Li := 3;
      Container.This_Ios_Interface.Runway_Surface_Intensity := 3;
      Container.This_Ios_Interface.Runway_Markings_Intensity := 3;
      Container.This_Ios_Interface.Strobe_LI := 0;
      Container.This_Ios_Interface.VASI_PAPI_LI := 3;
      Container.This_Ios_Interface.REIL_LI := 3;
      Container.This_Ios_Interface.Runway_Fixed_Distance_LI := 3;
      Container.This_Ios_Interface.Threshold_LI := 3;
      Container.This_Ios_Interface.TD_Zone_LI := 3;
      Container.This_Ios_Interface.Runway_AGL_Lead_In_LI := 3;
      Container.This_Ios_Interface.Taxiway_LI := 3;
      Container.This_Ios_Interface.Taxiway_Surface_Intensity := 3;
      Container.This_Ios_Interface.Taxiway_Markings_Intensity := 3;
      Container.This_Ios_Interface.Cultural_LI := 3;
      Container.This_Ios_Interface.All_Brightness_Flag := false;
      Container.This_Ios_Interface.All_Brightness_Intensity := 3;
      Container.This_Ios_Interface.GM_Terrain_Type := 0;
      Container.This_Ios_Interface.GM_VASI_PAPI_Type := 0;
      Container.This_Ios_Interface.GM_Terminal_Location := false;
      Container.This_Ios_Interface.GM_Cultural_Lights := 1;
      Container.This_Ios_Interface.GM_Runway_Length := 15;
      Container.This_Ios_Interface.Gm_Runway_Width := 0;
      Container.This_Ios_Interface.GM_Runway_Markings := 0;
      Container.This_Ios_Interface.GM_Approach_Light_System := 0;
      Container.This_Ios_Interface.GM_REIL_Lights_Select := false;
      Container.This_Ios_Interface.GM_Rotating_Beacon := false;
      Container.This_Ios_Interface.GM_Autosize_Runway := true;
      Container.This_Ios_Interface.Runway_Contamination_Select := 0;
      Container.This_Ios_Interface.Runway_Braking_Action := 0;
      Container.This_Ios_Interface.Runway_Roughness := 25;
      Container.This_Ios_Interface.Runway_Skid_Marks := false;
      Container.This_Ios_Interface.Runway_Blowing_Contaminant := false;
      Container.This_Ios_Interface.Marshaller_Control := 0;
      Container.This_Ios_Interface.Displaced_Threshold := false;
      Container.This_Ios_Interface.Generic_Data_Update := False;
      Container.This_Ios_Interface.Generic_Data_Compile := 0;
      Container.This_Ios_Interface.Visual_Debug_Position := False;
      Container.This_Ios_Interface.Light_Gun := 0;

      Container.This_Ios_Interface.Random_Runway_Lighting_Was_Set := false;
      Container.This_Ios_Interface.MIRL_HIRL_Light_Select_Was_Set := false;
      Container.This_Ios_Interface.Stop_Bar_Lights_Active_Was_Set := false;
      Container.This_Ios_Interface.Approach_LI_Was_Set := false;
      Container.This_Ios_Interface.Runway_Edge_LI_Was_Set := false;
      Container.This_Ios_Interface.Runway_Center_Line_Li_Was_Set := false;
      Container.This_Ios_Interface.Runway_End_LI_Was_Set := false;
      Container.This_Ios_Interface.Runway_Surface_Intensity_Was_Set
        := false;
      Container.This_Ios_Interface.Runway_Markings_Intensity_Was_Set
        := false;
      Container.This_Ios_Interface.Strobe_LI_Was_Set := false;
      Container.This_Ios_Interface.VASI_PAPI_LI_Was_Set := false;
      Container.This_Ios_Interface.REIL_LI_Was_Set := false;
      Container.This_Ios_Interface.Runway_Fixed_Distance_Li_Was_Set
        := false;
      Container.This_Ios_Interface.Threshold_LI_Was_Set := false;
      Container.This_Ios_Interface.TD_Zone_LI_Was_Set := false;
      Container.This_Ios_Interface.Runway_AGL_Lead_In_Li_Was_Set := false;
      Container.This_Ios_Interface.Taxiway_LI_Was_Set := false;
      Container.This_Ios_Interface.Taxiway_Surface_Intensity_Was_Set
        := false;
      Container.This_Ios_Interface.Taxiway_Markings_Intensity_Was_Set
        := false;
      Container.This_Ios_Interface.Cultural_LI_Was_Set := false;
      Container.This_Ios_Interface.All_Brightness_Flag_Was_Set := false;
      Container.This_Ios_Interface.All_Brightness_Intensity_Was_Set
        := false;
      Container.This_Ios_Interface.GM_Terrain_Type_Was_Set := false;
      Container.This_Ios_Interface.GM_VASI_PAPI_Type_Was_Set := false;
      Container.This_Ios_Interface.GM_Terminal_Location_Was_Set := false;
      Container.This_Ios_Interface.GM_Cultural_Lights_Was_Set := false;
      Container.This_Ios_Interface.GM_Runway_Length_Was_Set := false;
      Container.This_Ios_Interface.Gm_Runway_Width_Was_Set := false;
      Container.This_Ios_Interface.GM_Runway_Markings_Was_Set := false;
      Container.This_Ios_Interface.GM_Approach_Light_System_Was_Set
        := false;
      Container.This_Ios_Interface.GM_REIL_Lights_Select_Was_Set := false;
      Container.This_Ios_Interface.GM_Rotating_Beacon_Was_Set := false;
      Container.This_Ios_Interface.GM_Autosize_Runway_Was_Set := false;
      Container.This_Ios_Interface.Runway_Contamination_Select_Was_Set
        := false;
      Container.This_Ios_Interface.Runway_Braking_Action_Was_Set := false;
      Container.This_Ios_Interface.Runway_Roughness_Was_Set := false;
      Container.This_Ios_Interface.Runway_Skid_Marks_Was_Set := false;
      Container.This_Ios_Interface.Runway_Blowing_Contaminant_Was_Set
        := false;
      Container.This_Ios_Interface.Marshaller_Control_Was_Set := false;
      Container.This_Ios_Interface.Displaced_Threshold_Was_Set := false;
      Container.This_Ios_Interface.Generic_Data_Update_Was_Set := False;
      Container.This_Ios_Interface.Generic_Data_Compile_Was_Set := False;
      Container.This_Ios_Interface.Visual_Debug_Position_Was_Set := False;
      Container.This_Ios_Interface.Light_Gun_Was_Set := False;

      Jpats_Visual_Airport.Ios_If.Register_Ios_Variables;

      Ios.Gm_Terrain_Type := Configurable_Airport.Terrain_Type
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Vasi_Papi_Type := Configurable_Airport.Vasi_Type
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Terminal_Location := Configurable_Airport.Terminal_Location
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Cultural_Lights := Configurable_Airport.Cultural_Lights
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Runway_Length := Configurable_Airport.Runway_Length
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Runway_Width := Configurable_Airport.Runway_Width
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Runway_Markings := Configurable_Airport.Runway_Markings
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Approach_Light_System := Configurable_Airport.
        Approach_Light_System
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Reil_Lights_Select := Configurable_Airport.Reils
        (Container.This_Subsystem.The_Configurable_Airport);
      Ios.Gm_Rotating_Beacon := Configurable_Airport.Rotating_Beacon
        (Container.This_Subsystem.The_Configurable_Airport);


   end Initialize;

-- Method Check_Lighting
   procedure Check_Lighting
     (Integration_Constant : in Float) is
      Time_Of_Day : Natural;
      Ifr_Conditions : Boolean;
   begin

      Time_Of_Day := Jpats_Visual_Weather.Time_Of_Day;
      Ifr_Conditions := Jpats_Visual_Weather.Ifr_Conditions;

      if (Container.This_Ios_Interface.All_Brightness_Flag_Was_Set) then
         Container.This_Subsystem.All_Bright :=
           Container.This_Ios_Interface.All_Brightness_Flag;
         Container.This_Ios_Interface.All_Brightness_Flag_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.
          All_Brightness_Intensity_Was_Set) then
         Container.This_Subsystem.All_Bright_Intensity :=
           Container.This_Ios_Interface.All_Brightness_Intensity;
         Container.This_Ios_Interface.All_Brightness_Intensity_Was_Set :=
           False;
      end if;

      Set_Day_Light_Timer(Integration_Constant,
                          Time_Of_Day,
                          Ifr_Conditions);
      Set_Lighting_Level(Time_Of_Day,
                         Ifr_Conditions);

      if (Change_Lighting) then
         Runway.Set_End_Lights_Intensity
           (Container.This_Subsystem.The_Runway, Light_Intensity);
--         Runway.Set_Markings_Intensity
--           (Container.This_Subsystem.The_Runway, Light_Intensity);
--         Runway.Set_Surface_Intensity
--           (Container.This_Subsystem.The_Runway, Light_Intensity);
         Runway.Set_Strobe_Intensity
           (Container.This_Subsystem.The_Runway, Approach_Strobe_Intensity);
         Runway.Set_Center_Line_Intensity
           (Container.This_Subsystem.The_Runway, Light_Intensity);
         Runway.Set_Runway_Edge_Intensity
           (Container.This_Subsystem.The_Runway, Light_Intensity);
         Runway.Set_Threshold_Intensity
           (Container.This_Subsystem.The_Runway, Light_Intensity);
         Runway.Set_Td_Zone_Intensity
           (Container.This_Subsystem.The_Runway, Light_Intensity);

         Taxiway_Lighting.Set_Light_Intensity
           (An_Instance=>Container.This_Subsystem.The_Taxiway,
            An_Intensity=>Light_Intensity);

         Scene_Lighting.Set_AGL_Intensity
           (An_Instance=>Container.This_Subsystem.The_Scene_Lighting,
            An_Intensity=>Approach_Strobe_Intensity);

         Scene_Lighting.Set_Approach_Intensity
           (An_Instance=>Container.This_Subsystem.The_Scene_Lighting,
            An_Intensity=>Approach_Strobe_Intensity);

         if (Time_Of_Day /= 2 and Light_Intensity < 3) then
            Scene_Lighting.Set_Cultural_Intensity
              (An_Instance=>Container.This_Subsystem.The_Scene_Lighting,
               An_Intensity=>Night_Intensity);
         else
            Scene_Lighting.Set_Cultural_Intensity
              (An_Instance=>Container.This_Subsystem.The_Scene_Lighting,
               An_Intensity=>Light_Intensity);
         end if;

         Scene_Lighting.Set_Fixed_Distance_Intensity
           (An_Instance=>Container.This_Subsystem.The_Scene_Lighting,
            An_Intensity=>Light_Intensity);

         Scene_Lighting.Set_REIL_Intensity
           (An_Instance=>Container.This_Subsystem.The_Scene_Lighting,
            An_Intensity=>Light_Intensity);

         Scene_Lighting.Set_VASI_PAPI_Intensity
           (An_Instance=>Container.This_Subsystem.The_Scene_Lighting,
            An_Intensity=>Light_Intensity);

         Change_Lighting := False;
      end if;

      if (Container.This_Ios_Interface.Approach_Li_Was_Set) then
         Scene_Lighting.Set_Approach_Intensity
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Approach_Li);
         Container.This_Ios_Interface.Approach_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Runway_Edge_Li_Was_Set) then
         Runway.Set_Runway_Edge_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_Edge_Li);
         Container.This_Ios_Interface.Runway_Edge_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Runway_End_Li_Was_Set) then
         Runway.Set_End_Lights_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_End_Li);
         Container.This_Ios_Interface.Runway_End_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Runway_Center_Line_Li_Was_Set) then
         Runway.Set_Center_Line_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_Center_Line_Li);
         Container.This_Ios_Interface.Runway_Center_Line_Li_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.
          Runway_Surface_Intensity_Was_Set) then
         Runway.Set_Surface_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_Surface_Intensity);
         Container.This_Ios_Interface.Runway_Surface_Intensity_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.
          Runway_Markings_Intensity_Was_Set) then
         Runway.Set_Markings_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_Markings_Intensity);
         Container.This_Ios_Interface.Runway_Markings_Intensity_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.Strobe_Li_Was_Set) then
         Runway.Set_Strobe_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Strobe_Li);
         Container.This_Ios_Interface.Strobe_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Vasi_Papi_Li_Was_Set) then
         Scene_Lighting.Set_Vasi_Papi_Intensity
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Vasi_Papi_Li);
         Container.This_Ios_Interface.Vasi_Papi_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Reil_Li_Was_Set) then
         Scene_Lighting.Set_Reil_Intensity
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Reil_Li);
         Container.This_Ios_Interface.Reil_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.
          Runway_Fixed_Distance_Li_Was_Set) then
         Scene_Lighting.Set_Fixed_Distance_Intensity
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Runway_Fixed_Distance_Li);
         Container.This_Ios_Interface.Runway_Fixed_Distance_Li_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.Threshold_Li_Was_Set) then
         Runway.Set_Threshold_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Threshold_Li);
         Container.This_Ios_Interface.Threshold_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Td_Zone_Li_Was_Set) then
         Runway.Set_Td_Zone_Intensity
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Td_Zone_Li);
         Container.This_Ios_Interface.Td_Zone_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Runway_Agl_Lead_In_Li_Was_Set) then
         Scene_Lighting.Set_Agl_Intensity
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Runway_Agl_Lead_In_Li);
         Container.This_Ios_Interface.Runway_Agl_Lead_In_Li_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.Taxiway_Li_Was_Set) then
         Taxiway_Lighting.Set_Light_Intensity
           (Container.This_Subsystem.The_Taxiway,
            Container.This_Ios_Interface.Taxiway_Li);
         Container.This_Ios_Interface.Taxiway_Li_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.
          Taxiway_Surface_Intensity_Was_Set) then
         Taxiway_Lighting.Set_Surface_Intensity
           (Container.This_Subsystem.The_Taxiway,
            Container.This_Ios_Interface.Taxiway_Surface_Intensity);
         Container.This_Ios_Interface.Taxiway_Surface_Intensity_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.
          Taxiway_Markings_Intensity_Was_Set) then
         Taxiway_Lighting.Set_Marking_Intensity
           (Container.This_Subsystem.The_Taxiway,
            Container.This_Ios_Interface.Taxiway_Markings_Intensity);
         Container.This_Ios_Interface.Taxiway_Markings_Intensity_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.Cultural_Li_Was_Set) then
         Scene_Lighting.Set_Cultural_Intensity
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Cultural_Li);
         Container.This_Ios_Interface.Cultural_Li_Was_Set := False;
      end if;

   end Check_Lighting;

-- Method Update
   procedure Update
     (Integration_Constant : in Float) is
      Airport_Name : Jpats_Visual_Airport_Types.Name_Type;
      Runway_Name : Jpats_Visual_Airport_Types.Name_Type;
      Ac_Latitude : Lat_Long_Types.Latitude_Degrees;
      Ac_Longitude : Lat_Long_Types.Longitude_Degrees;
      Runway_Position : Lat_Long_Types.Position;
      Runway_Heading : Jpats_Visual_Airport_Types.Heading_Type;
      Active_Airport : Jpats_Radio_Db_If_Types.Apt_Struct_Type;
      Active_Runway : Jpats_Radio_Db_If_Types.Rwy_Struct_Type;
      Runway_Length : Integer;
      Runway_Width : Integer;
      Runway_Glideslope : Float;
      Runway_Approach_Path : Integer;
      Runway_Vasi_Type : Integer;
      Runway_Markings : Integer;
      Runway_Reil : Boolean;
      Airport_Terrain : Integer;
      Airport_Building : Boolean;
      Lat_Diff : Long_Float;
      Lon_Diff : Long_Float;
      Scene_Distance : Long_Float;
      Configuration_Found : Boolean;
      Reposition_Active : Boolean;
      Active_Distance : Long_Float;
      Reference_Distance : Long_Float;
   begin

      if Container.This_Ios_Interface.Visual_Debug_Position_Was_Set then
         Container.This_Subsystem.Visual_Debug_Position :=
           Container.This_Ios_Interface.Visual_Debug_Position;
         Container.This_Ios_Interface.Visual_Debug_Position_Was_Set := False;
      end if;

      Ac_Latitude := Jpats_Simulated_Aircraft.Get_North;
      Ac_Longitude := Jpats_Simulated_Aircraft.Get_East;

      lat_diff := (Ac_Latitude - Jpats_Reposition.Active_Runway.Lat)*60.0;
      Lon_Diff := Visual_Airport_Utilities.Xn180
			(Ac_Longitude - Jpats_Reposition.Active_Runway.Lon);
      Lon_Diff := Lon_Diff*60.0*
		    Long_Float(cos(Float(Ac_Latitude * 0.017453292)));
      Active_Distance := Long_Float(sqrt(float(Lat_Diff*Lat_Diff + Lon_Diff*Lon_Diff)));

      lat_diff := (Ac_Latitude - Jpats_Reposition.Reference_Runway.Lat)*60.0;
      Lon_Diff := Visual_Airport_Utilities.Xn180
			(Ac_Longitude - Jpats_Reposition.Reference_Runway.Lon);
      Lon_Diff := Lon_Diff*60.0*
		    Long_Float(cos(Float(Ac_Latitude * 0.017453292)));
      Reference_Distance := Long_Float(sqrt(float(Lat_Diff*Lat_Diff + Lon_Diff*Lon_Diff)));

      if (Active_Distance < Reference_Distance) then
         Active_Airport := Jpats_Reposition.Active_Airfield;
         Active_Runway := Jpats_Reposition.Active_Runway;
      else
         Active_Airport := Jpats_Reposition.Reference_Airfield;
         Active_Runway := Jpats_Reposition.Reference_Runway;
      end if;

      if (Container.This_Subsystem.Visual_Debug_Position) then
         Active_Airport.Idnt := "KIAD";
         Active_Airport.Lat := 38.936489;
         Active_Airport.Lon := -77.459588;
         Active_Airport.Elv := 306.0;
         Active_Runway.Idnt := "01L ";
         Active_Runway.Lat := 38.936489;
         Active_Runway.Lon := -77.459588;
         Active_Runway.Hdg := 0.0;
         Active_Runway.Wid := 150.0;
         Active_Runway.Len := 10000.0;
         Ac_Latitude := 38.936489;
         Ac_Longitude := -77.459588;
      end if;

      Airport_Name := Active_Airport.Idnt;
      Runway_Name := Active_Runway.Idnt;

      if (Container.This_Ios_Interface.Light_Gun_Was_Set) then
         Scene_Lighting.Set_Light_Gun_Signal
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Light_Gun);
         Container.This_Ios_Interface.Light_Gun_Was_Set := False;
      end if;

      Scene_Lighting.Light_Gun_Control
        (Container.This_Subsystem.The_Scene_Lighting,
         Integration_Constant);

      if (Container.This_Ios_Interface.Mirl_Hirl_Light_Select_Was_Set) then
         Scene_Lighting.Set_Mirl_Hirl_Select
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Mirl_Hirl_Light_Select);
         Container.This_Ios_Interface.Mirl_Hirl_Light_Select_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.Stop_Bar_Lights_Active_Was_Set) then
         Scene_Lighting.Set_Stop_Bar_Active
           (Container.This_Subsystem.The_Scene_Lighting,
            Container.This_Ios_Interface.Stop_Bar_Lights_Active);
         Container.This_Ios_Interface.Stop_Bar_Lights_Active_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.Random_Runway_Lighting_Was_Set) then
         Container.This_Subsystem.Random_Light_Intensities_On :=
           Container.This_Ios_Interface.Random_Runway_Lighting;
         Container.This_Ios_Interface.Random_Runway_Lighting_Was_Set :=
           False;
      end if;

      if (Container.This_Ios_Interface.
          Runway_Contamination_Select_Was_Set) then
         Runway.Determine_Contaminant
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_Contamination_Select);
         Container.This_Ios_Interface.
           Runway_Contamination_Select_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.
          Runway_Blowing_Contaminant_Was_Set) then
         Runway.Set_Blowing_Contaminant
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_Blowing_Contaminant);
         Container.This_Ios_Interface.
           Runway_Blowing_Contaminant_Was_Set := False;
      end if;

	Reposition_Active := Jpats_Reposition.Reposition_Command.Active;

      if (Airport_Name /= Last_Repo_Airport or Reposition_Active) then

         Container.This_Subsystem.Icao_Code := Airport_Name;
         Container.This_Subsystem.Position.Latitude :=
           Active_Airport.Lat;
         Container.This_Subsystem.Position.Longitude :=
           Active_Airport.Lon;
         Container.This_Subsystem.Position.Altitude :=
           Active_Airport.Elv;
         Runway.Set_Selected_Runway
           (Container.This_Subsystem.The_Runway, Runway_Name);

         Runway_Position.Latitude := Active_Runway.Lat;
         Runway_Position.Longitude := Active_Runway.Lon;
         Runway_Position.Altitude := Float(Active_Runway.Typ2);
         Runway.Set_True_Heading
           (Container.This_Subsystem.The_Runway, Active_Runway.Hdg);
         Runway.Set_Width
           (Container.This_Subsystem.The_Runway, Integer(Active_Runway.Wid));
         Runway.Set_Length
           (Container.This_Subsystem.The_Runway, Integer(Active_Runway.Len));
         Runway.Set_Position
           (Container.This_Subsystem.The_Runway, Runway_Position);

         Scene_Collection.Set_Airport
           (Container.This_Subsystem.Icao_Code,
            Container.This_Subsystem.Visual_Airport_Code,
            Runway.Selected_Runway(Container.This_Subsystem.The_Runway),
            Runway.Position(Container.This_Subsystem.The_Runway),
            Runway.True_Heading(Container.This_Subsystem.The_Runway));

         Last_Repo_Airport := Airport_Name;
         Last_Repo_Runway := Runway_Name;

         if (Container.This_Subsystem.Visual_Airport_Code = "****") then
            Config_Collection.Find_Configuration
              (Container.This_Subsystem.Icao_Code,
               Runway_Name,
               Runway_Length,
               Runway_Width,
               Runway_Glideslope,
               Runway_Approach_Path,
               Runway_Vasi_Type,
               Runway_Markings,
               Runway_Reil,
               Airport_Terrain,
               Airport_Building,
			   Configuration_Found);

            if Configuration_Found then
               Configurable_Airport.Set_Terminal_Location
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Airport_Building);
               Configurable_Airport.Set_Terrain_Type
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Airport_Terrain);
               Configurable_Airport.Set_Vasi_Type
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Vasi_Type);
               Configurable_Airport.Set_Runway_Markings
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Markings);
               Configurable_Airport.Set_Approach_Light_System
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Approach_Path);
               Configurable_Airport.Set_Reils
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Reil);
               Configurable_Airport.Set_Glideslope_Angle
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Glideslope);
               Ios.Gm_Terrain_Type := Configurable_Airport.Terrain_Type
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Vasi_Papi_Type := Configurable_Airport.Vasi_Type
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Terminal_Location :=
                 Configurable_Airport.Terminal_Location
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Cultural_Lights := Configurable_Airport.Cultural_Lights
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Runway_Markings := Configurable_Airport.Runway_Markings
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Approach_Light_System := Configurable_Airport.
                 Approach_Light_System
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Reil_Lights_Select := Configurable_Airport.Reils
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Rotating_Beacon := Configurable_Airport.Rotating_Beacon
                 (Container.This_Subsystem.The_Configurable_Airport);
            end if;
         end if;

      elsif (Runway_Name /= Last_Repo_Runway) then
--             (Container.This_Subsystem.The_Runway)) then
         Runway.Set_Selected_Runway
           (Container.This_Subsystem.The_Runway, Runway_Name);
         Runway_Position.Latitude := Active_Runway.Lat;
         Runway_Position.Longitude := Active_Runway.Lon;
         Runway_Position.Altitude := Float(Active_Runway.Typ2);
         Runway.Set_True_Heading
           (Container.This_Subsystem.The_Runway, Active_Runway.Hdg);
         Runway.Set_Width
           (Container.This_Subsystem.The_Runway, Integer(Active_Runway.Wid));
         Runway.Set_Length
           (Container.This_Subsystem.The_Runway, Integer(Active_Runway.Len));
         Runway.Set_Position
           (Container.This_Subsystem.The_Runway, Runway_Position);

         Scene_Collection.Set_Airport
           (Container.This_Subsystem.Icao_Code,
            Container.This_Subsystem.Visual_Airport_Code,
            Runway.Selected_Runway(Container.This_Subsystem.The_Runway),
            Runway.Position(Container.This_Subsystem.The_Runway),
            Runway.True_Heading(Container.This_Subsystem.The_Runway));

         Last_Repo_Runway := Runway_Name;

         if (Container.This_Subsystem.Visual_Airport_Code = "****") then
            Config_Collection.Find_Configuration
              (Container.This_Subsystem.Icao_Code,
               Runway_Name,
               Runway_Length,
               Runway_Width,
               Runway_Glideslope,
               Runway_Approach_Path,
               Runway_Vasi_Type,
               Runway_Markings,
               Runway_Reil,
               Airport_Terrain,
               Airport_Building,
			   Configuration_Found);

            if Configuration_Found then
               Configurable_Airport.Set_Terminal_Location
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Airport_Building);
               Configurable_Airport.Set_Terrain_Type
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Airport_Terrain);
               Configurable_Airport.Set_Vasi_Type
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Vasi_Type);
               Configurable_Airport.Set_Runway_Markings
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Markings);
               Configurable_Airport.Set_Approach_Light_System
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Approach_Path);
               Configurable_Airport.Set_Reils
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Reil);
               Configurable_Airport.Set_Glideslope_Angle
                 (Container.This_Subsystem.The_Configurable_Airport,
                  Runway_Glideslope);
               Ios.Gm_Terrain_Type := Configurable_Airport.Terrain_Type
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Vasi_Papi_Type := Configurable_Airport.Vasi_Type
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Terminal_Location :=
                 Configurable_Airport.Terminal_Location
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Cultural_Lights := Configurable_Airport.Cultural_Lights
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Runway_Markings := Configurable_Airport.Runway_Markings
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Approach_Light_System := Configurable_Airport.
                 Approach_Light_System
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Reil_Lights_Select := Configurable_Airport.Reils
                 (Container.This_Subsystem.The_Configurable_Airport);
               Ios.Gm_Rotating_Beacon := Configurable_Airport.Rotating_Beacon
                 (Container.This_Subsystem.The_Configurable_Airport);
            end if;
         end if;

      else
	   lat_diff := (Ac_Latitude - Active_Runway.Lat)*60.0;
	   Lon_Diff := Visual_Airport_Utilities.Xn180
			(Ac_Longitude - Active_Runway.Lon);
	   Lon_Diff := Lon_Diff*60.0*
		    Long_Float(cos(Float(Ac_Latitude * 0.017453292)));
	   Scene_Distance := Long_Float(sqrt(float(Lat_Diff*Lat_Diff + Lon_Diff*Lon_Diff)));

	   if (Scene_Distance > 20.0) then
	       Scene_Collection.Find_Closest_Airport
      	     (Ac_Latitude, Ac_Longitude,
            	Container.This_Subsystem.Icao_Code,
           		Container.This_Subsystem.Visual_Airport_Code,
            	Runway_Name,
            	Runway_Position,
            	Runway_Heading);
             Runway.Set_Selected_Runway
           	    (Container.This_Subsystem.The_Runway, Runway_Name);
             Runway.Set_Position
               (Container.This_Subsystem.The_Runway, Runway_Position);
             Runway.Set_True_Heading
               (Container.This_Subsystem.The_Runway, Runway_Heading);

             if (((Container.This_Subsystem.Icao_Code /= Last_Pass_Airport) or
                  (Runway_Name /= Last_Pass_Runway)) and
                 (Container.This_Subsystem.Visual_Airport_Code = "****")) then

                Config_Collection.Find_Configuration
                  (Container.This_Subsystem.Icao_Code,
                   Runway_Name,
                   Runway_Length,
                   Runway_Width,
                   Runway_Glideslope,
                   Runway_Approach_Path,
                   Runway_Vasi_Type,
                   Runway_Markings,
                   Runway_Reil,
                   Airport_Terrain,
                   Airport_Building,
				   Configuration_Found);

                if Configuration_Found then
                   Configurable_Airport.Set_Terminal_Location
                     (Container.This_Subsystem.The_Configurable_Airport,
                      Airport_Building);
                   Configurable_Airport.Set_Terrain_Type
                     (Container.This_Subsystem.The_Configurable_Airport,
                      Airport_Terrain);
                   Configurable_Airport.Set_Vasi_Type
                     (Container.This_Subsystem.The_Configurable_Airport,
                      Runway_Vasi_Type);
                   Configurable_Airport.Set_Runway_Markings
                     (Container.This_Subsystem.The_Configurable_Airport,
                      Runway_Markings);
                   Configurable_Airport.Set_Approach_Light_System
                     (Container.This_Subsystem.The_Configurable_Airport,
                      Runway_Approach_Path);
                   Configurable_Airport.Set_Reils
                     (Container.This_Subsystem.The_Configurable_Airport,
                      Runway_Reil);
                   Configurable_Airport.Set_Glideslope_Angle
                     (Container.This_Subsystem.The_Configurable_Airport,
                      Runway_Glideslope);
                   Ios.Gm_Terrain_Type := Configurable_Airport.Terrain_Type
                     (Container.This_Subsystem.The_Configurable_Airport);
                   Ios.Gm_Vasi_Papi_Type := Configurable_Airport.Vasi_Type
                     (Container.This_Subsystem.The_Configurable_Airport);
                   Ios.Gm_Terminal_Location :=
                     Configurable_Airport.Terminal_Location
                     (Container.This_Subsystem.The_Configurable_Airport);
                   Ios.Gm_Cultural_Lights := Configurable_Airport.Cultural_Lights
                     (Container.This_Subsystem.The_Configurable_Airport);
                   Ios.Gm_Runway_Markings := Configurable_Airport.Runway_Markings
                     (Container.This_Subsystem.The_Configurable_Airport);
                   Ios.Gm_Approach_Light_System := Configurable_Airport.
                     Approach_Light_System
                     (Container.This_Subsystem.The_Configurable_Airport);
                   Ios.Gm_Reil_Lights_Select := Configurable_Airport.Reils
                     (Container.This_Subsystem.The_Configurable_Airport);
                   Ios.Gm_Rotating_Beacon := Configurable_Airport.Rotating_Beacon
                     (Container.This_Subsystem.The_Configurable_Airport);
                end if;
             end if;
	    end if;
      end if;	

      Last_Pass_Airport := Container.This_Subsystem.Icao_Code;
      Last_Pass_Runway := Runway_Name;

      if (Container.This_Ios_Interface.Runway_Skid_Marks_Was_Set) then
         Runway.Set_Wet_Skid_Marks
           (Container.This_Subsystem.The_Runway,
            Container.This_Ios_Interface.Runway_Skid_Marks);
         Container.This_Ios_Interface.Runway_Skid_Marks_Was_Set := False;
      end if;

      if (Container.This_Ios_Interface.Generic_Data_Update_Was_Set) then
         if (Container.This_Ios_Interface.Gm_Cultural_Lights_Was_Set) then
            Configurable_Airport.Set_Cultural_Lights
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Cultural_Lights);
            Container.This_Ios_Interface.Gm_Cultural_Lights_Was_Set :=
              False;
         end if;

         if (Container.This_Ios_Interface.Gm_Terminal_Location_Was_Set) then
            Configurable_Airport.Set_Terminal_Location
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Terminal_Location);
            Container.This_Ios_Interface.Gm_Terminal_Location_Was_Set :=
              False;
         end if;

         if (Container.This_Ios_Interface.Gm_Terrain_Type_Was_Set) then
            Configurable_Airport.Set_Terrain_Type
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Terrain_Type);
            Container.This_Ios_Interface.Gm_Terrain_Type_Was_Set :=
              False;
         end if;

         if (Container.This_Ios_Interface.Gm_Vasi_Papi_Type_Was_Set) then
            Configurable_Airport.Set_Vasi_Type
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Vasi_Papi_Type);
            Container.This_Ios_Interface.Gm_Vasi_Papi_Type_Was_Set :=
              False;
         end if;


         if (Container.This_Ios_Interface.Gm_Runway_Markings_Was_Set) then
            Configurable_Airport.Set_Runway_Markings
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Runway_Markings);
            Container.This_Ios_Interface.Gm_Runway_Markings_Was_Set :=
              False;
         end if;

         if (Container.This_Ios_Interface.
             Gm_Approach_Light_System_Was_Set) then
            Configurable_Airport.Set_Approach_Light_System
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Approach_Light_System);
            Container.This_Ios_Interface.Gm_Approach_Light_System_Was_Set :=
              False;
         end if;

         if (Container.This_Ios_Interface.Gm_Reil_Lights_Select_Was_Set) then
            Configurable_Airport.Set_Reils
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Reil_Lights_Select);
            Container.This_Ios_Interface.Gm_Reil_Lights_Select_Was_Set :=
              False;
         end if;

         if (Container.This_Ios_Interface.Gm_Rotating_Beacon_Was_Set) then
            Configurable_Airport.Set_Rotating_Beacon
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Rotating_Beacon);
            Container.This_Ios_Interface.Gm_Rotating_Beacon_Was_Set :=
              False;
         end if;

         if ((not Configurable_Airport.Autosize_Runway
              (Container.This_Subsystem.The_Configurable_Airport)) and
             (Container.This_Ios_Interface.Gm_Runway_Length_Was_Set)) then
            Configurable_Airport.Set_Runway_Length
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Runway_Length);
            Container.This_Ios_Interface.Gm_Runway_Length_Was_Set :=
              False;
         end if;

         if ((not Configurable_Airport.Autosize_Runway
              (Container.This_Subsystem.The_Configurable_Airport)) and
             (Container.This_Ios_Interface.Gm_Runway_Width_Was_Set)) then
            Configurable_Airport.Set_Runway_Width
              (Container.This_Subsystem.The_Configurable_Airport,
               Container.This_Ios_Interface.Gm_Runway_Width);
            Container.This_Ios_Interface.Gm_Runway_Width_Was_Set :=
              False;
         end if;

         Ios.Gm_Terrain_Type := Configurable_Airport.Terrain_Type
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Vasi_Papi_Type := Configurable_Airport.Vasi_Type
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Terminal_Location := Configurable_Airport.Terminal_Location
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Cultural_Lights := Configurable_Airport.Cultural_Lights
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Runway_Length := Configurable_Airport.Runway_Length
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Runway_Width := Configurable_Airport.Runway_Width
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Runway_Markings := Configurable_Airport.Runway_Markings
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Approach_Light_System := Configurable_Airport.
           Approach_Light_System
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Reil_Lights_Select := Configurable_Airport.Reils
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Rotating_Beacon := Configurable_Airport.Rotating_Beacon
           (Container.This_Subsystem.The_Configurable_Airport);

         Ios.Generic_Data_Compile := 1;
         Generic_Compile_Timer := 60.0;

         Container.This_Ios_Interface.Generic_Data_Update_Was_Set := False;
      end if;

      if ( Generic_Compile_Timer > 0.0 ) then
         Generic_Compile_Timer := Generic_Compile_Timer - Integration_Constant;
      else
         Ios.Generic_Data_Compile := 0;
      end if;

      if (Container.This_Ios_Interface.Gm_Autosize_Runway_Was_Set) then
         Configurable_Airport.Set_Autosize_Runway
           (Container.This_Subsystem.The_Configurable_Airport,
            Container.This_Ios_Interface.Gm_Autosize_Runway);
         Container.This_Ios_Interface.Gm_Autosize_Runway_Was_Set :=
           False;
      end if;

      Runway.Format_Runway_Name
           (Container.This_Subsystem.The_Runway,
            Day_Light_Timer,
            (Container.This_Subsystem.Visual_Airport_Code = "****"));

      Runway.Set_Magnetic_Heading
        (Container.This_Subsystem.The_Runway,
         Float(Runway.Runway_Number
               (Container.This_Subsystem.The_Runway) * 10)+0.049999);

      if (Configurable_Airport.Autosize_Runway
          (Container.This_Subsystem.The_Configurable_Airport) and
          (Container.This_Subsystem.Visual_Airport_Code = "****")) then
         Configurable_Airport.Determine_Runway_Model
           (Container.This_Subsystem.The_Configurable_Airport,
            Runway.Length(Container.This_Subsystem.The_Runway),
            Runway.Width(Container.This_Subsystem.The_Runway));
         Ios.Gm_Runway_Length := Configurable_Airport.Runway_Length
           (Container.This_Subsystem.The_Configurable_Airport);
         Ios.Gm_Runway_Width := Configurable_Airport.Runway_Width
           (Container.This_Subsystem.The_Configurable_Airport);
      end if;


      Check_Lighting(Integration_Constant);

      Ios.Light_Gun := Scene_Lighting.Light_Gun_Signal
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Random_Runway_Lighting :=
        Container.This_Subsystem.Random_Light_Intensities_On;
      Ios.Mirl_Hirl_Light_Select := Scene_Lighting.Mirl_Hirl_Select
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Stop_Bar_Lights_Active := Scene_Lighting.Stop_Bar_Active
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Approach_Li := Scene_Lighting.Approach_Intensity
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Runway_Edge_Li := Runway.Runway_Edge_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Runway_End_Li := Runway.End_Lights_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Runway_Center_Line_Li := Runway.Center_Line_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Runway_Surface_Intensity := Runway.Surface_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Runway_Markings_Intensity := Runway.Markings_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Strobe_Li := Runway.Strobe_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Vasi_Papi_Li := Scene_Lighting.Vasi_Papi_Intensity
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Reil_Li := Scene_Lighting.Reil_Intensity
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Runway_Fixed_Distance_Li := Scene_Lighting.Fixed_Distance_Intensity
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Threshold_Li := Runway.Threshold_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Td_Zone_Li := Runway.Td_Zone_Intensity
        (Container.This_Subsystem.The_Runway);
      Ios.Runway_Agl_Lead_In_Li := Scene_Lighting.Agl_Intensity
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.Taxiway_Li := Taxiway_Lighting.Light_Intensity
        (Container.This_Subsystem.The_Taxiway);
      Ios.Taxiway_Surface_Intensity := Taxiway_Lighting.Surface_Intensity
        (Container.This_Subsystem.The_Taxiway);
      Ios.Taxiway_Markings_Intensity := Taxiway_Lighting.Marking_Intensity
        (Container.This_Subsystem.The_Taxiway);
      Ios.Cultural_Li := Scene_Lighting.Cultural_Intensity
        (Container.This_Subsystem.The_Scene_Lighting);
      Ios.All_Brightness_Flag := Container.This_Subsystem.All_Bright;
      Ios.All_Brightness_Intensity := Container.This_Subsystem.
        All_Bright_Intensity;
      Ios.Gm_Autosize_Runway := Configurable_Airport.Autosize_Runway
        (Container.This_Subsystem.The_Configurable_Airport);
      if ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
           Jpats_Visual_Airport_Types.Dry ) then
         Ios.Runway_Contamination_Select := 0;
      elsif ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
              Jpats_Visual_Airport_Types.Slush ) then
         Ios.Runway_Contamination_Select := 4;
      elsif ( Runway.Patchy_Contaminant
              (Container.This_Subsystem.The_Runway) ) then
         if ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
              Jpats_Visual_Airport_Types.Wet ) then
            Ios.Runway_Contamination_Select := 5;
         elsif ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
                 Jpats_Visual_Airport_Types.Snow ) then
            Ios.Runway_Contamination_Select := 6;
         elsif ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
                 Jpats_Visual_Airport_Types.Ice ) then
            Ios.Runway_Contamination_Select := 7;
         else
            Ios.Runway_Contamination_Select := 5;
         end if;
      elsif ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
              Jpats_Visual_Airport_Types.Wet ) then
         Ios.Runway_Contamination_Select := 1;
      elsif ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
              Jpats_Visual_Airport_Types.Snow ) then
         Ios.Runway_Contamination_Select := 2;
      elsif ( Runway.Contaminant(Container.This_Subsystem.The_Runway) =
              Jpats_Visual_Airport_Types.Ice ) then
         Ios.Runway_Contamination_Select := 3;
      else
         Ios.Runway_Contamination_Select := 0;
      end if;
      Ios.Runway_Blowing_Contaminant := Runway.Blowing_Contaminant
        (Container.This_Subsystem.The_Runway);
      Ios.Runway_Skid_Marks := Runway.Wet_Skid_Marks
        (Container.This_Subsystem.The_Runway);


   end Update;



end JPATS_Visual_Airport.Controller;
