with System;
with Interfaces.C;
package Vx_Ip_Binding is

   type IP_Address is
      array ( Integer range 1 .. 4 ) of Interfaces.C.Unsigned_Char;
   -- Represent IP address of 4 octets using an array of 4 Characters

   type IP_Port is mod 2**16;

   type Socket_Type is new Integer;

   -- Bindings to VxWorks networking routines

   type Protocol_Level is
     ( Ipproto_Ip, Ipproto_Icmp, Ipproto_Tcp,
       Ipproto_Udp, Ipproto_Raw, Sol_Socket );
   for Protocol_Level use
      ( Ipproto_Ip => 0,
        Ipproto_Icmp => 1,
        Ipproto_Tcp => 6,
        Ipproto_Udp => 17,
        Ipproto_Raw => 255,
        Sol_Socket => 65535 );
   for Protocol_Level'Size use 32;

   type Socket_Type_Type is
     ( Sock_Undef, Sock_Stream, Sock_Dgram, Sock_Raw,
       Sock_Rdm, Sock_Seq_Packet );

   type Address_Family_Type is
     ( Af_Unspec, Af_Unix, Af_Inet );
   for Address_Family_Type'Size use 16;

   type Protocol_Family_Type is
     ( Pf_Unspec, Pf_Unix, Pf_Inet );
   for Protocol_Family_Type'Size use 8;

   Sockaddr_Size : constant := 16;
   type Sockaddr_Type is
      record
         Sin_Len       : Interfaces.C.Unsigned_Char := Sockaddr_Size;
         Sin_Family    : Protocol_Family_Type;
         Sin_Port      : Ip_Port;
         Sin_Addr      : Ip_Address;
         Sin_Zero_Low  : Interfaces.C.Unsigned := 0;
         Sin_Zero_High : Interfaces.C.Unsigned := 0;
      end record;

   for Sockaddr_Type use
      record
         Sin_Len       at  0 range  0 .. 7;
         Sin_Family    at  1 range  0 .. 7;
         Sin_Port      at  2 range  0 .. 15;
         Sin_Addr      at  4 range  0 .. 31;
         Sin_Zero_Low  at  8 range  0 .. 31;
         Sin_Zero_High at 12 range  0 .. 31;
      end record;
   for Sockaddr_Type'Size use Sockaddr_Size * 8;

   function Socket
     ( Domain : in Protocol_Family_Type;
       Socktype : in Socket_Type_Type;
       Protocol : in Interfaces.C.Int )
     return Socket_Type;
   pragma Import (C, Socket, "socket");

   function Bind
     ( Socket : in Socket_Type;
       Name : access Sockaddr_Type;
       Namelen : in Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Bind, "bind");

   function Connect
     ( Socket : in Socket_Type;
       Name : access Sockaddr_Type;
       Namelen : in Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Connect, "connect");

   function Listen
     ( S : in Socket_Type;
       Backlog : in Natural )
     return Interfaces.C.Int;
   pragma Import (C, Listen, "listen");

   function Accept_Connection
     ( S : in Socket_Type;
       Addr : access Sockaddr_Type;
       Addrlen : access Interfaces.C.Int )
     return Socket_Type;
   pragma Import (C, Accept_Connection, "accept");

   function Send
     ( S : in Socket_Type;
       Buf : in System.Address;
       Buflen : in Interfaces.C.Int;
       Flags : in Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Send, "send");

   function Recv
     ( S : in Socket_Type;
       Buf : in System.Address;
       Buflen : in Interfaces.C.Int;
       Flags : in Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Recv, "recv");

   function Recvfrom
     ( S : in Socket_Type;
       Buf : in System.Address;
       Buflen : in Interfaces.C.Int;
       Flags : in Interfaces.C.Int;
       From : access Sockaddr_Type;
       FromLen : access Interfaces.C.Int) 
     return Interfaces.C.Int;
   pragma Import (C, Recvfrom, "recvfrom");
   function Setsockopt
     ( S : in Socket_Type;
       Level : in Protocol_Level;
       Optname : in Interfaces.C.Int;
       Optval : access Interfaces.C.Int;
       Optlen : in Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Setsockopt, "setsockopt");

   function Ioctl
     ( S : in Socket_Type;
       Function_Code : in Interfaces.C.Int;
       Arg : access Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Ioctl, "ioctl");

   -- ioctl function codes

   Fionbio : constant Interfaces.C.Int := 16;

   function Htons
     ( X : in Ip_Port )
     return Ip_Port;

   function Htonl
     ( X : in Ip_Address )
     return Ip_Address;

   procedure Close
     ( S : in Socket_Type );
   pragma Import (C, Close, "close");

   -- sockets constants

   Inaddr_Any : constant Ip_Address := (others => 0);
   Ip_multicast_ttl : constant Interfaces.C.Int := 10;

   -- setsockaddr operations

   Tcp_Nodelay  : constant := 1;
   So_Reuseaddr : constant := 4;
   So_Reuseport : constant := 16#0200#;
   So_Sndbuf    : constant := 16#1001#;
   So_Rcvbuf    : constant := 16#1002#;

end Vx_Ip_Binding;
