-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Jpats_Io.Container;
with Io_Interface;
with System;
with Log;
with Ada.Strings.Unbounded;
package body Jpats_Io is

   -- Interface to JPATS I/O software subsystem

   function Interface_Is_Up
     ( An_Interface : in Jpats_Io_Types.Subsystem_Interface_Type )
     return Boolean is

      -- Returns operational status of specified simulator subsystem interface

   begin

      --| Call the operational status function for the specified simulator
      --| subsystem interface, translating the input interface name to
      --| the appropriate interface object.

      return Io_Interface.Is_Up
                ( Container.This_Subsystem.The_Interfaces(An_Interface) );

   end Interface_Is_Up;

   procedure Print_Interface_Status
     ( An_Interface : in Jpats_Io_Types.Subsystem_Interface_Type ) is
   -- Prints operational status of specified simulator subsystem interface
   begin
      Io_Interface.Print_Status
        ( Container.This_Subsystem.The_Interfaces(An_Interface) );
   end Print_Interface_Status;

   procedure Map_To_Icd_Mnemonic
     ( An_Interface       : in Jpats_Io_Types.Subsystem_Interface_Type;
       An_Icd_Mnemonic    : in String;
       A_Direction        : in Io_Types.Direction;
       An_Address         : in System.Address;
       A_Size             : in Natural;
       Replay_Passthrough : in Boolean := False) is

      -- Map an mnemonic listed in the specified Interface Control Document
      -- to the address of a host simulation Ada variable

   begin

      --| Call the map procedure for the specified simulator subsystem interface,
      --| translating the input interface name to the appropriate interface
      --| object.

      -- A_Size argument is in bits; convert to bytes before calling IO_Interface.Map

      if Container.This_Subsystem.The_Interfaces(An_Interface).Is_Valid then

         Io_Interface.Map
           ( Container.This_Subsystem.The_Interfaces(An_Interface),
             An_Icd_Mnemonic,
             A_Direction,
             An_Address,
             A_Size / System.Storage_Unit,
             Replay_Passthrough);

      else

         Log.Report
           ( "Cannot map " & An_Icd_Mnemonic
             & " to "
             & Ada.Strings.Unbounded.To_String
             ( Container.This_Subsystem.The_Interfaces(An_Interface)
               .The_Name )
             & " which was not successfully initialized.",
             Log.Warning );

      end if;

   end Map_To_Icd_Mnemonic;

   function Interface
     ( An_Interface : in Jpats_Io_Types.Subsystem_Interface_Type )
     return Io_Interface.Instance is
   begin
      return Container.This_Subsystem.The_Interfaces(An_Interface);
   end Interface;

   function Mnemonic_Buffer_Address
     ( An_Interface : in Jpats_Io_Types.Subsystem_Interface_Type;
       A_Direction : in Io_Types.Direction;
       A_Mnemonic : in Interfaces.C.Strings.Chars_Ptr )
     return System.Address is
   begin
      return
        Io_Interface.Mnemonic_Buffer_Address
        ( Container.This_Subsystem.The_Interfaces(An_Interface),
          A_Direction,
          Interfaces.C.Strings.Value(A_Mnemonic) );
   end Mnemonic_Buffer_Address;

   function Mnemonic_Host_Address
     ( An_Interface : in Jpats_Io_Types.Subsystem_Interface_Type;
       A_Direction : in Io_Types.Direction;
       A_Mnemonic : in Interfaces.C.Strings.Chars_Ptr )
     return System.Address is
   begin
      return
        Io_Interface.Mnemonic_Host_Address
        ( Container.This_Subsystem.The_Interfaces(An_Interface),
          A_Direction,
          Interfaces.C.Strings.Value(A_Mnemonic) );
   end Mnemonic_Host_Address;

   function Mnemonic_Is_Mapped
     ( An_Interface : in Jpats_Io_Types.Subsystem_Interface_Type;
       A_Direction : in Io_Types.Direction;
       A_Mnemonic : in Interfaces.C.Strings.Chars_Ptr )
     return Boolean is
   begin
      return
        Io_Interface.Mnemonic_Is_Mapped
        ( Container.This_Subsystem.The_Interfaces(An_Interface),
          A_Direction,
          Interfaces.C.Strings.Value(A_Mnemonic) );
   end Mnemonic_Is_Mapped;

   function Mnemonic_Size
     ( An_Interface : in Jpats_Io_Types.Subsystem_Interface_Type;
       A_Direction : in Io_Types.Direction;
       A_Mnemonic : in Interfaces.C.Strings.Chars_Ptr )
     return Natural is
   begin
      return
        Io_Interface.Mnemonic_Size
        ( Container.This_Subsystem.The_Interfaces(An_Interface),
          A_Direction,
          Interfaces.C.Strings.Value(A_Mnemonic) );
   end Mnemonic_Size;


end Jpats_Io;
