-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Log;
with Jpats_Io.Controller;
with Subsystem_Scheduler.Registrar;
with Buffer_Stream.Save;

with Io_Buffer, JPATS_IO.Container;

package body Jpats_IO.Scheduler is

   -- Scheduler interface to JPATS I/O software subsystem

   -- Initialize defines non-real-time initializations run each time the
   -- simulation is started

   procedure Initialize
     (An_Instance : in out Instance) is
   begin
      Log.Report ( "Beginning initialization: JPATS_IO snapshot size is " &
                   Natural'Image (An_Instance.The_Snapshot_Size) );
      Controller.Initialize;
      -- after initialization, the snapshot size is known
      An_Instance.The_Snapshot_Size
        := Controller.Snapshot_Size;
      Log.Report ( "Initialization complete: JPATS_IO snapshot size is " &
                   Natural'Image (An_Instance.The_Snapshot_Size) );
      -- set flag to allow first pass code to be executed
      An_Instance.First_Update_Pass := True;
   end Initialize;

   -- Update defines normal realtime behavior of the subsystem

   procedure Update
     (An_Instance : in Out Instance;
      Integration_Constant : in Float) is

      Open_Succeeded : Boolean;

   begin

      -- On the first time through, make a note of the save stream
      -- created for this scheduler, to use for writing data each
      -- frame.  We have to do this here, rather than in initialize,
      -- because the streams aren't created before the initialize
      -- method is called.

      if An_Instance.First_Update_Pass then
         An_Instance.To_Stream :=
           Subsystem_Scheduler.Save_Stream ( An_Instance );
         An_Instance.From_Stream :=
           Subsystem_Scheduler.Restore_Stream ( An_Instance );
         An_Instance.First_Update_Pass := False;
         Log.Report ( "First pass complete: JPATS_IO snapshot size is " &
                      Natural'Image (An_Instance.The_Snapshot_Size) );
      end if;

      -- Get inputs and send outputs.
      -- Pass flag indicating if restore is in progress,
      -- and pass the stream from which restore data can be read
      -- if needed
      Controller.Update;

      -- Clear flag for next pass
      An_Instance.Restore_In_Progress := False;

   end Update;

   -- Update_In_Freeze defines behavior of the IO subsystem during replay
   procedure Update_In_Freeze
     (An_Instance : in out Instance) is
   begin
      Controller.Replay_Update;
   end Update_In_Freeze;


   -------------------------------------------------------------------------------
   -- Routine to save a snapshot of the instance to the given stream.
   --
   -- Since this is an IO scheduler, this routine will get called immediately
   -- after every Update call. It will not get called for a snapshot.
   -------------------------------------------------------------------------------
   procedure Save
     (An_Instance : in out Instance;
      To_Stream   : access Ada.Streams.Root_Stream_Type'Class) is
   begin
      Controller.Save_Snapshot (To_Stream);
   end Save;

   -------------------------------------------------------------------------------
   -- The restore method is called each time there is data ready to be
   -- played back.  Update will *not* be called afterwards.
   -- If this method is called, we don't want to overwrite
   -- the restored data with data coming in from the subsystems.
   -------------------------------------------------------------------------------
   procedure Restore
     (An_Instance : in out Instance;
      From_Stream : access Ada.Streams.Root_Stream_Type'Class) is
   begin
      Controller.Restore_Snapshot (From_Stream);
   end Restore;

   function Snapshot_Size
     (An_Instance : in Instance)
     return Natural is
   begin
      Log.Report ( "JPATS_IO snapshot size is " &
                   Natural'Image (An_Instance.The_Snapshot_Size),
                   Log.Informational );
      return An_Instance.The_Snapshot_Size;
   end Snapshot_Size;

   This_Scheduler : aliased Instance;

begin

   Subsystem_Scheduler.Registrar.Register (This_Scheduler'Access);

end Jpats_IO.Scheduler;






