-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Jpats_Io.Container, Io_Interface, Io_Types,
  Io_Medium, Io_Medium.Tcp, Io_Medium.Udp, Log;
with JPATS_Io.SimIO_Container;
with Jpats_Io.Container.Ios_Variables;
with JPATS_Simphonics.Container;
with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Io_Map;
--with High_Resolution_Timer;
with Vx_Ip_Binding;
with Simulation_Dictionary;
with Ada.Strings.Fixed;

with Ada.Text_IO,Log;
with Ada.Unchecked_Conversion;

package body Jpats_Io.Controller is

   package SimIO renames JPATS_Io.SimIO_Container;
   Cr_Lf : constant String ( 1 .. 2 )
     := Ada.Characters.Latin_1.Cr
      & Ada.Characters.Latin_1.Lf;

   -- Controls processing of JPATS I/O software subsystem

   use Jpats_Io_Types;
   -- Make values of Subsystem_Interface_Type directly visible, for brevity

   Up_This_Pass : array ( Jpats_Io_Types.Subsystem_Interface_Type ) of Boolean;
   Up_Last_Pass : array ( Jpats_Io_Types.Subsystem_Interface_Type ) of Boolean;

   Counter : Natural := 0;

   Mcast_Transmit : Boolean;

   The_Mcast_Ethernet : Io_Medium.Handle;

   First_Update_Pass : Boolean := True;

--   Wait_Timer, Transmit_Timer : High_Resolution_Timer.Instance;

   Mcast_Foreign_Address : Vx_Ip_Binding.Ip_Address := (10,10,3,2);
   pragma Export ( C, Mcast_Foreign_Address, "SimIO_foreign_address" );

   Mcast_Local_Address : Vx_Ip_Binding.Ip_Address := (10,10,3,1);
   pragma Export ( C, Mcast_Local_Address, "SimIO_local_address" );

   Mcast_Foreign_Port : Natural := 3921;
   pragma Export ( C, Mcast_Foreign_Port, "SimIO_foreign_port" );

   Mcast_Local_Port : Natural := 9001;
   pragma Export ( C, Mcast_Local_Port, "SimIO_local_port" );

   Invalid_Ip_Address : exception;

   -----------------------------------------------------------------------------
   -- Translate the given string into an IP address. The string needs to be a
   -- series of four numbers (from 0 to 255) separated by periods
   -----------------------------------------------------------------------------
   function To_IP_Address (Address : String) return Vx_Ip_Binding.Ip_Address is
      Result : Vx_Ip_Binding.Ip_Address;
      Start : Natural := Address'First;
      Finish : Natural;
   begin
      for Byte in Result'range loop
         -- Locate the end of the next address byte's string.
         if Byte = Result'Last then
            Finish := Address'Last;
         else
            -- Find the character in front of the next period
            Finish := Ada.Strings.Fixed.Index (Source  => Address (Start..Address'Last),
                                               Pattern => ".") - 1;
         end if;

         -- Convert it to a byte in the IP address array
         Result (Byte) := Interfaces.C.Unsigned_Char'Value(Address(Start..Finish));

         -- Advance start past the byte's dot.
         Start := Finish + 2;
      end loop;

      return Result;
   exception
      when Constraint_Error =>
         raise Invalid_Ip_Address;
   end To_Ip_Address;

   procedure Update_ICD_Control_Block_Data ( Tick     : in Integer;
                                             In_Buff  : in Integer;
                                             Out_Buff : in out Integer ) is

      Count  : Container.ICD_Control_Block;
      State  : Container.ICD_Control_Block;
      Temp   : Container.ICD_Control_Block;

      Integer_Form : Integer;

      function Integer_to_ICD_Control_Block is 
         new Ada.Unchecked_Conversion (Integer, Container.ICD_Control_Block);

      function ICD_Control_Block_Byte_to_Integer is 
         new Ada.Unchecked_Conversion (Container.ICD_Control_Block, Integer);

   begin

      Count := Integer_to_ICD_Control_Block ( Tick );
      State := Integer_to_ICD_Control_Block ( In_Buff );

      Temp.Counter := Count.Counter;
      Temp.State   := State.Counter;

      Out_Buff := ICD_Control_Block_Byte_to_Integer ( Temp );

   end Update_ICD_Control_Block_Data ;

   procedure Print_Control_Block ( Buff : in Integer ) is

      package M_IO is new Ada.Text_IO.Modular_IO(Container.ICD_Control_Block_Byte);
      package T_IO renames Ada.Text_IO;

      SIO_Control : Container.ICD_Control_Block;
      Integer_Form : Integer;

      function Integer_to_SIO is 
         new Ada.Unchecked_Conversion (Integer, Container.ICD_Control_Block);

      function SIO_Byte_to_Integer is 
         new Ada.Unchecked_Conversion (Container.ICD_Control_Block_Byte, Integer);

   begin

      SIO_Control := Integer_to_SIO( Buff );

      Log.Report (  "   Counter:" &
                    Natural'Image ( SIO_Byte_to_Integer(SIO_Control.Counter )) &

                    " State:" &
                    Natural'Image ( SIO_Byte_to_Integer(SIO_Control.State) ) &
                    " Acknowledge:" &
                    Natural'Image ( SIO_Byte_to_Integer(SIO_Control.Acknowledge) ) &
                    " Reserved:" &
                    Natural'Image ( SIO_Byte_to_Integer(SIO_Control.Reserved )));

   end Print_Control_Block;


   procedure Initialize is
      -- Set up of I/O interfaces at start-up
      Size : Natural;
   begin

      begin

         --| Create the SimIO Multicast interface using UDP/IP
         Mcast_Transmit :=
           Simulation_Dictionary.Lookup("SimIO_Installed") = "T";

         if Mcast_Transmit then

            Mcast_Foreign_Address := To_Ip_Address
              (Simulation_Dictionary.Lookup
                 (Word    => "SimIO_Foreign_Address", 
                  Default => " 224.0.0.245"
                 )
              );

            Mcast_Local_Address := To_Ip_Address
              (Simulation_Dictionary.Lookup
                  (Word    => "SimIO_Local_Address",
                   Default => "10.10.3.1"
                  )
              );

            The_Mcast_Ethernet :=
              Io_Medium.Udp.Create(A_Local_Port      => 9001,
                                   A_Foreign_Port    => 3921,
                                   A_Local_Address   => Mcast_Local_Address,
                                   A_Foreign_Address => Mcast_Foreign_Address,
                                   Use_Single_Socket => False);

            Io_Medium.Initialize(The_Mcast_Ethernet);
   
         end if;
      exception
         when others =>
            Log.Report
              ( "Unhandled exception when creating SimIO Multicast interface."
                & "Interface disabled.",
                Log.Error );
      end;

      begin
         --| Create the Flight Deck I/O interface using UDP/IP
         Container.This_Subsystem.The_Interfaces ( Flight_Deck )
           := Io_Interface.Create ( A_Name => "Flight Deck",
                                    An_Icd_Filename => "hostfdk.icd",
                                    A_Medium =>
                                      Io_Medium.UDP.Create
                                    ( A_Local_Port    => 5001,
                                      A_Foreign_Port  => 5000,
                                      A_Local_Address   => ( 10, 10, 3, 1),
                                      A_Foreign_Address => ( 10, 10, 3, 2),
                                      Use_Single_Socket => False
                                      ),
                                    Buffer_Is_Network_Order => False );
         JPATS_Simphonics.Container.This_Io_Interface.SIO_Host_Timestamp := Container.This_Subsystem.The_Interfaces ( Flight_Deck ).The_Version;
      exception
         when others =>
            Log.Report
              ( "Unhandled exception when creating Flight Deck interface."
                & "Interface disabled.",
                Log.Error );
      end;

      begin

         --| Create the Control Loading I/O interface using UDP
         Container.This_Subsystem.The_Interfaces(Control_Loading)
               := Io_Interface.Create ( A_Name => "Control Loading",
                                        An_Icd_Filename => "hostecl.icd",
                                        A_Medium =>
                                          Io_Medium.UDP.Create
                                        ( A_Local_Port => 5101,
                                          A_Foreign_Port => 5100,
                                          A_Local_Address => ( 10, 10, 2, 1 ),
                                          A_Foreign_Address => ( 10, 10, 2, 2 ),
                                          Use_Single_Socket => False ),
                                        Buffer_Is_Network_Order => False );

      exception
         when others =>
            Log.Report
              ( "Unhandled exception when creating Control Loading interface."
                & "Interface disabled.",
                Log.Error );
      end;

      begin

         --| Create the aural cue subsystem interface using UDP
          Container.This_Subsystem.The_Interfaces(Aural_Cue)
                := Io_Interface.Create ( A_Name => "Aural Cue",
                                         An_Icd_Filename => "hostacs.icd",
                                         A_Medium =>
                                           Io_Medium.UDP.Create
                                         ( A_Local_Port => 5051,
                                           A_Foreign_Port => 5049,
                                           A_Local_Address => ( 10, 10, 1, 1 ),
                                           A_Foreign_Address => ( 10, 10, 1, 2 ),
                                           Use_Single_Socket => True ),
                                         Buffer_Is_Network_Order => False );

      exception
         when others =>
            Log.Report
              ( "Unhandled exception when creating Aural Cue interface."
                & "Interface disabled.",
                Log.Error );
      end;

      Container.Ios_Variables.Register;


      -- calculate snapshot size

      Size := 0;
      for The_Interface in Jpats_Io_Types.Subsystem_Interface_Type loop
         Size :=
           Size +
           Io_Interface.Snapshot_Size
           ( Container.This_Subsystem.The_Interfaces(The_Interface) );
      end loop;
--      Container.This_Subsystem.The_Snapshot_Size := Size;

      -- Make the buffer big enough so we don't overflow it before
      -- the data_dispatcher can empty it.
      Container.This_Subsystem.The_Snapshot_Size := Size * 60;
      Log.Report
        ( "The actual I/O snapshot size = "
          & Integer'Image ( Size )
          & " * 60 = "
          & Integer'Image ( Container.This_Subsystem.The_Snapshot_Size ),
          Log.Informational );

   end Initialize;

   procedure Update is
--     ( Restore_From_Snapshot : Boolean;
--       From_Stream : access Ada.Streams.Root_Stream_Type'Class ) is
   -- Transfer and distribution of I/O data at scheduled intervals
      Bytes_Written : Natural := 0;
      Read_Count   : Natural := 0;
      Read_Errors  : Natural := 0;
      Write_Count  : Natural := 0;
      Write_Errors : Natural := 0;
   begin

      -- First pass after initialization, convert io_maps to a form
      -- more efficient for real-time processing
      if First_Update_Pass then
         Log.Report ( "JPATS_IO.Controller first update pass");
         for The_Interface in Jpats_Io_Types.Subsystem_Interface_Type
         loop
            declare
               This_Interface_Instance : Io_Interface.Instance
                 renames Container.This_Subsystem.The_Interfaces(The_Interface);
            begin
               for The_Direction in Io_Types.Direction loop
                  begin

                     Io_Map.Convert_For_Real_Time
                       ( This_Interface_Instance.The_Maps ( The_Direction ) );
                  exception
                     when others =>
                        Log.Report
                          ( "Unhandled exception when converting the "
                            & Io_Types.Direction'Image ( The_Direction )
                            & " map for "
                            & Jpats_Io_Types.Subsystem_Interface_Type'Image
                            ( The_Interface )
                            & " interface for real-time operation.",
                            Log.Error );
                  end;
               end loop;
            end;
         end loop;

         Log.Report ( "I/O maps converted for efficient real-time operation.");
         First_Update_Pass := False;
--         High_Resolution_Timer.Start (Wait_Timer);
--         High_Resolution_Timer.Start (Transmit_Timer);
      end if;

      Counter := Counter + 1;
      --| Loop through all interfaces, and do a write and read if for each
      --| if it is up.  If the interface isn't up, reset it.

      for The_Interface in Jpats_Io_Types.Subsystem_Interface_Type
      loop
         if Container.This_Subsystem.The_Interfaces(The_Interface).Is_Valid then
            declare
               This_Interface_Instance : Io_Interface.Instance
                 renames Container.This_Subsystem.The_Interfaces(The_Interface);
            begin
               Up_This_Pass ( The_Interface )
                 := Io_Interface.Is_Up
                 ( This_Interface_Instance );
               if Up_This_Pass ( The_Interface ) then
                  if not Up_Last_Pass ( The_Interface ) then

                     Log.Report
                       ( Jpats_Io_Types.Subsystem_Interface_Type'Image
                           ( The_Interface )
                         & " interface is back up, pass "
                         & Integer'Image ( Counter ) );
                     
                  end if;

--                   if The_Interface = Flight_Deck then
--                      High_Resolution_Timer.Stop (Wait_Timer);

--                      if
--                        High_Resolution_Timer.Milliseconds (Wait_Timer) > 17.0 or
--                        High_Resolution_Timer.Milliseconds (Wait_Timer) < 15.0
--                      then
--                         Log.Report
--                           ("IO took" &
--                            Float'Image(High_Resolution_Timer.Milliseconds (Wait_Timer)) &
--                            "ms between flightdeck sends.",
--                            Log.Warning
--                            );
--                      end if;
--                   end if;

--                   if The_Interface = Flight_Deck then
--                      High_Resolution_Timer.Start (Transmit_Timer);
--                   end if;

                  Io_Interface.Write_Outputs ( This_Interface_Instance );
                  Write_Count := This_Interface_Instance.Transfer_Count(IO_Types.Output);
                  Write_Errors := This_Interface_Instance.Error_Count(IO_Types.Output);

                  Io_Interface.Read_Inputs (This_Interface_Instance);
                  Read_Count := This_Interface_Instance.Transfer_Count(IO_Types.Input);
                  Read_Errors := This_Interface_Instance.Error_Count(IO_Types.Input);

                  if ( The_Interface = Flight_Deck ) then

                     SimIO.SimIO_Read_Count   := Read_Count; 
                     SimIO.SimIO_Read_Errors  := Read_Errors; 
                     SimIO.SimIO_Write_Count  := Write_Count;
                     SimIO.SimIO_Write_Errors := Write_Errors;

                     if Mcast_Transmit then
                        if (Io_Medium.Is_Up(The_Mcast_Ethernet)) then

                           Io_Medium.Write(A_Handle         => The_Mcast_Ethernet,
                                           A_Buffer_Pointer => Mcast_Transmit'Address,
                                           A_Length         => 4,
                                           Bytes_Written    => Bytes_Written);
                        end if;
								 
		     end if;
                  elsif ( The_Interface = Control_Loading ) then

                     SimIO.CLM_Read_Count   := Read_Count; 
                     SimIO.CLM_Read_Errors  := Read_Errors; 
                     SimIO.CLM_Write_Count  := Write_Count;
                     SimIO.CLM_Write_Errors := Write_Errors;

                  elsif ( The_Interface = Aural_Cue ) then

                     SimIO.ACS_Write_Count  := Write_Count;
                     SimIO.ACS_Write_Errors := Write_Errors;

                  end if;


--                   if The_Interface = Flight_Deck then
--                      High_Resolution_Timer.Stop (Transmit_Timer);
--                      if High_Resolution_Timer.Milliseconds (Transmit_Timer) > 0.5 then
--                         Log.Report
--                           ("IO took" &
--                            Float'Image(High_Resolution_Timer.Milliseconds (Transmit_Timer)) &
--                            "ms to perform flightdeck UDP I/O.",
--                            Log.Warning
--                            );
--                      end if;
--                      High_Resolution_Timer.Start (Wait_Timer);
--                   end if;

               else
                  if Up_Last_Pass ( The_Interface ) then

                     Log.Report
                       ( Jpats_Io_Types.Subsystem_Interface_Type'Image
                           ( The_Interface )
                         & " interface is down, pass "
                         & Integer'Image ( Counter ),
                         Severity => Log.Warning );
                     
                  end if;
                  Io_Medium.Reset ( This_Interface_Instance.The_Medium );
               end if;

            exception

               when Error : others =>
                  Log.Report ( Ada.Exceptions.Exception_Information(Error)
                               & Cr_Lf
                               & Ada.Exceptions.Exception_Message(Error)
                               & Cr_Lf
                               & "Unhandled exception during update for "
                               & Jpats_Io_Types.Subsystem_Interface_Type'Image
                                   ( The_Interface )
                               & "; interface will be disabled.",
                               Log.Error );
                  This_Interface_Instance.Is_Valid := False;

            end;

         end if;
         Up_Last_Pass ( The_Interface ) := Up_This_Pass ( The_Interface );
      end loop;

   end Update;

   procedure Replay_Update is
   -- Transfer and distribution of I/O data at scheduled intervals,
   -- but the inputs read are not passed on to the sim.
   begin

      -- First pass after initialization, convert io_maps to a form
      -- more efficient for real-time processing
      if First_Update_Pass then
         Log.Report ( "JPATS_IO.Controller first update pass");
         for The_Interface in Jpats_Io_Types.Subsystem_Interface_Type
         loop
            declare
               This_Interface_Instance : Io_Interface.Instance
                 renames Container.This_Subsystem.The_Interfaces(The_Interface);
            begin
               for The_Direction in Io_Types.Direction loop
                  begin

                     Io_Map.Convert_For_Real_Time
                       ( This_Interface_Instance.The_Maps ( The_Direction ) );
                  exception
                     when others =>
                        Log.Report
                          ( "Unhandled exception when converting the "
                            & Io_Types.Direction'Image ( The_Direction )
                            & " map for "
                            & Jpats_Io_Types.Subsystem_Interface_Type'Image
                            ( The_Interface )
                            & " interface for real-time operation.",
                            Log.Error );
                  end;
               end loop;
            end;
         end loop;

         Log.Report ( "I/O maps converted for efficient real-time operation.");
         First_Update_Pass := False;
      end if;

      Counter := Counter + 1;
      --| Loop through all interfaces, and do a write and read if for each
      --| if it is up.  If the interface isn't up, reset it.

      for The_Interface in Jpats_Io_Types.Subsystem_Interface_Type
      loop
         if Container.This_Subsystem.The_Interfaces(The_Interface).Is_Valid then
            declare
               This_Interface_Instance : Io_Interface.Instance
                 renames Container.This_Subsystem.The_Interfaces(The_Interface);
            begin
               Up_This_Pass ( The_Interface )
                 := Io_Interface.Is_Up
                 ( This_Interface_Instance );
               if Up_This_Pass ( The_Interface ) then
                  if not Up_Last_Pass ( The_Interface ) then
                     Log.Report
                       ( Jpats_Io_Types.Subsystem_Interface_Type'Image
                           ( The_Interface )
                         & " interface is back up, pass "
                         & Integer'Image ( Counter ) );
                  end if;
                  Io_Interface.Write_Outputs ( This_Interface_Instance );
                  Io_Interface.Read_Inputs
                    (An_Instance    => This_Interface_Instance,
                     Replay_Discard => True
                     );

               else
                  if Up_Last_Pass ( The_Interface ) then
                     Log.Report
                       ( Jpats_Io_Types.Subsystem_Interface_Type'Image
                           ( The_Interface )
                         & " interface is down, pass "
                         & Integer'Image ( Counter ),
                         Severity => Log.Warning );
                  end if;
                  Io_Medium.Reset ( This_Interface_Instance.The_Medium );
               end if;

            exception

               when Error : others =>
                  Log.Report ( Ada.Exceptions.Exception_Information(Error)
                               & Cr_Lf
                               & Ada.Exceptions.Exception_Message(Error)
                               & Cr_Lf
                               & "Unhandled exception during update for "
                               & Jpats_Io_Types.Subsystem_Interface_Type'Image
                                   ( The_Interface )
                               & "; interface will be disabled.",
                               Log.Error );
                  This_Interface_Instance.Is_Valid := False;

            end;

         end if;
         Up_Last_Pass ( The_Interface ) := Up_This_Pass ( The_Interface );
      end loop;

   end Replay_Update;

   function Snapshot_Size return Natural is
      -- Returns size of the record/playback snapshot in bytes
   begin
      return Container.This_Subsystem.The_Snapshot_Size;
   end Snapshot_Size;

   procedure Save_Snapshot
     ( To_Stream   : access Ada.Streams.Root_Stream_Type'Class ) is
   begin
      for The_Interface in Jpats_Io_Types.Subsystem_Interface_Type
        range Jpats_Io_Types.Flight_Deck .. Jpats_Io_Types.Control_Loading
      loop
         declare
            This_Interface_Instance : Io_Interface.Instance
              renames Container.This_Subsystem.The_Interfaces(The_Interface);
         begin
            if This_Interface_Instance.Is_Valid then
               Io_Interface.Save_Snapshot
                 ( This_Interface_Instance,
                   To_Stream  );
            end if;
         exception

            when Error : others =>
               Log.Report ( Ada.Exceptions.Exception_Information(Error)
                            & Cr_Lf
                            & Ada.Exceptions.Exception_Message(Error)
                            & Cr_Lf
                            & "Unhandled exception during save_snapshot for "
                            & Jpats_Io_Types.Subsystem_Interface_Type'Image
                            ( The_Interface )
                            & " on iteration "
                            & Integer'Image ( Counter )
                            & "; interface will be disabled.",
                            Log.Error );
               This_Interface_Instance.Is_Valid := False;

         end;
      end loop;
   end Save_Snapshot;

   procedure Restore_Snapshot
     (From_Stream : access Ada.Streams.Root_Stream_Type'Class ) is
   begin
      for The_Interface in Jpats_Io_Types.Subsystem_Interface_Type
        range Jpats_Io_Types.Flight_Deck .. Jpats_Io_Types.Control_Loading
      loop
         declare
            This_Interface_Instance : Io_Interface.Instance
              renames Container.This_Subsystem.The_Interfaces(The_Interface);
         begin
            if This_Interface_Instance.Is_Valid then
               Io_Interface.Restore_Snapshot
                 ( This_Interface_Instance,
                   From_Stream  );
            end if;
         exception

            when Error : others =>
               Log.Report ( Ada.Exceptions.Exception_Information(Error)
                            & Cr_Lf
                            & Ada.Exceptions.Exception_Message(Error)
                            & Cr_Lf
                            & "Unhandled exception during save_snapshot for "
                            & Jpats_Io_Types.Subsystem_Interface_Type'Image
                            ( The_Interface )
                            & " on iteration "
                            & Integer'Image ( Counter )
                            & "; interface will be disabled.",
                            Log.Error );
               This_Interface_Instance.Is_Valid := False;

         end;
      end loop;
   end Restore_Snapshot;

end Jpats_Io.Controller;
