-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Rich Givis and Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Interfaces.C;
with Interfaces.C.Strings;
with Vx_Ip_Binding;
with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;
with Log;

pragma Elaborate_All ( Ada.Unchecked_Deallocation );
package body Io_Medium.Udp is

   use type Interfaces.C.Int;
   use type Vx_Ip_Binding.Socket_Type;
   use type Vx_Ip_Binding.Ip_port;

   type Udp_Handle is access all Io_Medium.Udp.Instance;

   procedure Free is new Ada.Unchecked_Deallocation
     ( Io_Medium.Udp.Instance, Udp_Handle );

   -- Subclass for protocol for communicating with subsystems
   -- via UDP

   -- errno
   Errno : Interfaces.C.Int;
   pragma Import (C, Errno, "errno");

   function Wvevent
     ( User_Event_Id : in Interfaces.C.Unsigned_Short;
       Buffer        : in System.Address := System.Null_Address;
       Buffer_Size   : in Interfaces.C.Unsigned := 0
     ) return Interfaces.C.Int;
   pragma Import (C, Wvevent, "wvEvent");

   procedure Perror
     ( Str : in String ) is
      Saved_Errno : Interfaces.C.Int := Errno;
   begin
      Log.Report ( "UDP "
                   & Str & " errno = "
                   & Interfaces.C.Int'Image (Saved_Errno),
                   Log.Error );
   end Perror;

   Ewouldblock : constant Interfaces.C.Int := 70;

   function Create
     ( A_Local_Port : in Natural;
       A_Foreign_Port : in Natural;
       A_Local_Address : in Vx_Ip_Binding.Ip_Address;
       A_Foreign_Address : in Vx_Ip_Binding.Ip_Address;
       Use_Single_Socket : in Boolean := True; 
       Bind_Local        : in Boolean := False	)
     return Handle is
      -- Create an instance of the interface on the specified port

      A_Handle : Handle :=
       new Io_Medium.Udp.Instance'
        ( The_Local_Port => Vx_Ip_Binding.Ip_Port ( A_Local_Port ),
          The_Foreign_Port => Vx_Ip_Binding.Ip_Port ( A_Foreign_Port ),
          The_Local_Address => A_Local_Address,
          The_Foreign_Address => A_Foreign_Address,
          The_Input_Socket => -1,
          Input_Socket_Valid => False,
          The_Output_Socket => -1,
          Output_Socket_Valid => False,
          Using_Single_Socket => Use_Single_Socket,
          Binding_Local       => Bind_Local,			  
          Send_Buffer_Size    => Default_Udp_Sndbuf_Size,
          Receive_Buffer_Size => Default_Udp_Rcvbuf_Size);

   begin

      return A_Handle;

   end Create;

   procedure Close_Sockets

     ( A_Handle : access Instance ) is
      -- Close the sockets

   begin

      if A_Handle.Input_Socket_Valid then
         Vx_Ip_Binding.Close ( A_Handle.The_Input_Socket );
      end if;

      if A_Handle.Output_Socket_Valid and not A_Handle.Using_Single_Socket then
         Vx_Ip_Binding.Close ( A_Handle.The_Output_Socket );
      end if;

      A_Handle.Input_Socket_Valid := False;
      A_Handle.Output_Socket_Valid := False;
      A_Handle.The_Input_Socket := -1;
      A_Handle.The_Output_Socket := -1;

   end Close_Sockets;

   procedure Delete
     ( A_Handle : access Instance ) is
      -- Delete an instance of the interface

      The_Udp_Handle : Udp_Handle := Udp_Handle ( A_Handle );

   begin

      Close_Sockets ( A_Handle );
      Free ( The_Udp_Handle );

   end Delete;

   procedure Initialize
     ( A_Handle : access Instance ) is
   -- Initialize the medium, creating any data structures necessary
   begin
      -- Initialize is the same as Reset.
      Reset ( A_Handle );
   end Initialize;

   procedure Open_Sockets
     ( A_Handle : access Instance ) is

      An_Input_Socket : Vx_Ip_Binding.Socket_Type;
      An_Output_Socket : Vx_Ip_Binding.Socket_Type;

      Local_Sockaddr : aliased Vx_Ip_Binding.Sockaddr_Type;
      Foreign_Sockaddr : aliased Vx_Ip_Binding.Sockaddr_Type;
      Addrlen : Interfaces.C.Int := Vx_Ip_Binding.Sockaddr_Type'Size / 8;

      Option_Value : aliased Interfaces.C.Int;
      Result_1 : Interfaces.C.Int;

      Bad_Socket : exception;

      Send_Buffer_Size    : aliased Interfaces.C.Int := Interfaces.C.Int(A_Handle.Send_Buffer_Size);
      Receive_Buffer_Size : aliased Interfaces.C.Int := Interfaces.C.Int(A_Handle.Receive_Buffer_Size);

   begin

      -- Assemble a local address to bind to the socket.  If
      -- the local address is INADDR_ANY, we don't care which network
      -- interface is involved.  If we provide a specific local
      -- address, we only want to exchange data via that interface.

      -- Note that IP addresses are assumed to be in network order
      -- already, with the most-significant octet in element 1 of the
      -- 4 byte array.

      Local_Sockaddr := ( Sin_Len    => Vx_IP_Binding.Sockaddr_Size,
                          Sin_Family => Vx_Ip_Binding.Pf_Inet,
                          Sin_Port =>
                            Vx_Ip_Binding.Htons ( A_Handle.The_Local_Port ),
                          Sin_Addr =>
                            A_Handle.The_Local_Address,
                          Sin_Zero_Low => 0,
                          Sin_Zero_High => 0 );

      -- Assemble a foreign address to connect to the socket.

      Foreign_Sockaddr := ( Sin_Len    => Vx_IP_Binding.Sockaddr_Size,
                            Sin_Family => Vx_Ip_Binding.Pf_Inet,
                            Sin_Port =>
                              Vx_Ip_Binding.Htons ( A_Handle.The_Foreign_Port ),
                            Sin_Addr =>
                              A_Handle.The_Foreign_Address,
                            Sin_Zero_Low => 0,
                            Sin_Zero_High => 0 );

      -- Create an output socket

      An_Output_Socket := Vx_Ip_Binding.Socket ( Vx_Ip_Binding.Pf_Inet,
                                         Vx_Ip_Binding.Sock_Dgram, 0 );

      if An_Output_Socket < 0 then
         Perror ( "Error on socket (output) for port "
                  & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Foreign_Port ) );
         raise Bad_Socket;
      end if;

      -- Set socket option SO_REUSEPORT
      Option_Value := 1;
      Result_1 := Vx_Ip_Binding.Setsockopt ( An_Output_Socket,
                                             Vx_Ip_Binding.Sol_Socket,
                                             Vx_Ip_Binding.So_Reuseport,
                                             Option_Value'Access,
                                             Option_Value'Size / 8 );

      if Result_1 < 0 then
         Perror ( "Error on setsockopt (output) for port "
                  & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Foreign_Port ) );
         raise Bad_Socket;
      end if;

      if ( A_Handle.Using_Single_Socket ) then

         -- use same socket for input and output; no need to make new
         -- one, just copy the file descriptor

         An_Input_Socket := An_Output_Socket;

      else

         -- use two sockets, one for input, one for output
         -- Create an input socket

         An_Input_Socket := Vx_Ip_Binding.Socket ( Vx_Ip_Binding.Pf_Inet,
                                                   Vx_Ip_Binding.Sock_Dgram, 0 );
         if An_Input_Socket < 0 then
            Perror ( "Error on socket (input) for port "
                     & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Local_Port ) );
            raise Bad_Socket;
         end if;

         -- Set socket option SO_REUSEPORT
         Option_Value := 1;
         Result_1 := Vx_Ip_Binding.Setsockopt ( An_Input_Socket,
                                                Vx_Ip_Binding.Sol_Socket,
                                                Vx_Ip_Binding.So_Reuseport,
                                                Option_Value'Access,
                                                Option_Value'Size / 8 );

         if Result_1 < 0 then
            Perror ( "Error on setsockopt (input) for port "
                     & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Local_Port ) );
            raise Bad_Socket;
         end if;

         -- Bind read socket to local address

         Result_1 := Vx_Ip_Binding.Bind ( An_Input_Socket,
                                          Local_Sockaddr'Access,
                                          Addrlen );

         if Result_1 < 0 then
            Perror ( "Error on bind for port "
                       & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Local_Port ) );
            raise Bad_Socket;
         end if;

      end if;

      -- Bind output socket to local address
      if ( A_Handle.Binding_Local ) then

   	     Result_1 := Vx_Ip_Binding.Bind ( An_Output_Socket,
                                          Local_Sockaddr'Access,
                                          Addrlen );
         if Result_1 < 0 then
            Perror ( "Error on bind for port "
                     & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Local_Port ) );
            raise Bad_Socket;
         end if;
	  end if;
      -- Connect output socket

      Result_1 := Vx_Ip_Binding.Connect ( An_Output_Socket,
                                          Foreign_Sockaddr'access,
                                          Addrlen );

      if Result_1 < 0 then
         Perror ( "Error on connect for port "
                  & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Foreign_Port ) );
         raise Bad_Socket;
      end if;

      -- Make socket non-blocking

      Result_1 := Vx_Ip_Binding.Ioctl ( An_Input_Socket,
                                        Vx_Ip_Binding.Fionbio,
                                        Option_Value'Access );

      if Result_1 < 0 then
         Perror ( "Error on ioctl (input) for port "
                  & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Local_Port ) );
         raise Bad_Socket;
      end if;

      -- Set the socket send buffer size
      if
        Vx_Ip_Binding.Setsockopt ( An_Output_Socket,
                                   Vx_Ip_Binding.Sol_Socket,
                                   Vx_Ip_Binding.So_Sndbuf,
                                   Send_Buffer_Size'Access,
                                   (Send_Buffer_Size'Size + 7) / 8 ) < 0
      then
         Perror ( "Error on setsockopt send buffer increase" );
         raise Bad_Socket;
      end if;

      -- Set the socket's receive buffer size
      if
        Vx_Ip_Binding.Setsockopt ( An_Input_Socket,
                                   Vx_Ip_Binding.Sol_Socket,
                                   Vx_Ip_Binding.So_Rcvbuf,
                                   Receive_Buffer_Size'Access,
                                   (Receive_Buffer_Size'Size + 7) / 8 ) < 0
      then
         Perror ( "Error on setsockopt receive buffer increase" );
         raise Bad_Socket;
      end if;

      A_Handle.Output_Socket_Valid := True;
      A_Handle.The_Output_Socket := An_Output_Socket;
      A_Handle.Input_Socket_Valid := True;
      A_Handle.The_Input_Socket := An_Input_Socket;

   exception

      -- If any of the above fail, close the sockets and mark them invalid
      when Bad_Socket =>
         if An_Input_Socket >=0 then
            Vx_Ip_Binding.Close ( An_Input_Socket );
         end if;
         if An_Output_Socket >=0 and not A_Handle.Using_Single_Socket then
            Vx_Ip_Binding.Close ( An_Output_Socket );
         end if;
         A_Handle.Input_Socket_Valid := False;
         A_Handle.Output_Socket_Valid := False;
         A_Handle.The_Output_Socket := -1;
         A_Handle.The_Input_Socket := -1;

   end Open_Sockets;

   procedure Reset
     ( A_Handle : access Instance ) is
      -- Reset the medium, reinitializing as needed
   begin

      if A_Handle.Input_Socket_Valid or A_Handle.Output_Socket_Valid then

         Close_Sockets ( A_Handle );

      end if;

      Open_Sockets ( A_Handle );

   end Reset;

   function Is_Up
     ( A_Handle : access Instance )
      return Boolean is
      -- Report status of the medium
   begin
      return ( A_Handle.Input_Socket_Valid and A_Handle.Output_Socket_Valid );
   end Is_Up;

   procedure Read
     ( A_Handle : access Instance;
       A_Buffer_Pointer : in System.Address;
       A_Length : in Natural;
       Bytes_Read : out Natural ) is

      -- Receive I/O buffer via the medium

      Result : Interfaces.C.Int;
      Recvfrom_Flags : Interfaces.C.Int := 0;
      Recvfrom_Address : aliased Vx_Ip_Binding.Sockaddr_Type;
      Recvfrom_Addrlen : aliased Interfaces.C.Int;

   begin

      -- initialize output variable
      Bytes_Read := 0;

      if A_Handle.Input_Socket_Valid then

         if Wvevent (6) /= 0 then
            Log.Report ("Unable to mark recv event");
         end if;
		 if ( A_Handle.Binding_Local ) then
            -- Call recv to read data into the buffer
            Result := Vx_Ip_Binding.Recv ( A_Handle.The_Input_Socket,
                                           A_Buffer_Pointer,
                                           Interfaces.C.Int(A_Length),
                                           0 );
         else
         -- Call recvfrom to read data into the buffer

         Result := Vx_Ip_Binding.Recvfrom
                      ( S => A_Handle.The_Input_Socket,
                        Buf => A_Buffer_Pointer,
                        Buflen => Interfaces.C.Int(A_Length),
                        Flags => Recvfrom_Flags,
                        From => Recvfrom_Address'Access,
                        FromLen => Recvfrom_Addrlen'Access);
		 end if;
         if Wvevent (7) /= 0 then
            Log.Report ("Unable to mark end recv event");
         end if;

         -- Check for errors -- flag if no data is read, or a
         -- non-EWOULDBLOCK error occurs. If a non-EWOULDBLOCK error log
         -- an error and close the Socket. If EWOULDBLOCK log a
         -- warning and continue.

         if Result < 0 then
            case Errno is
               when Ewouldblock =>
                  null;
               when others =>
                  Perror ( "Error on read from port "
                           & Vx_Ip_Binding.Ip_Port'Image
                             ( A_Handle.The_Local_Port ) );
                  Close_Sockets ( A_Handle );
            end case;
         elsif Result = 0 then
            Close_Sockets ( A_Handle );
         else
            Bytes_Read := Natural ( Result );

         end if;

      end if;

   end Read;

   procedure Write
     ( A_Handle : access Instance;
       A_Buffer_Pointer : in System.Address;
       A_Length : in Natural;
       Bytes_Written : out Natural ) is

      -- Transmit I/O buffer via the medium

      Result : Interfaces.C.Int;

   begin

      Bytes_Written := 0;

      if A_Handle.Output_Socket_Valid then

         if Wvevent (8) /= 0 then
            Log.Report ("Unable to mark send event");
         end if;
         Result := Vx_Ip_Binding.Send ( A_Handle.The_Output_Socket,
                                        A_Buffer_Pointer,
                                        Interfaces.C.Int(A_Length),
                                        0 );
         if Wvevent (9) /= 0 then
            Log.Report ("Unable to mark send event");
         end if;

         -- Check for errors -- flag if no data is written, or a
         -- non-EWOULDBLOCK error occurs. If a non-EWOULDBLOCK error log
         -- an error and close the Socket. If EWOULDBLOCK log a
         -- warning and continue.

         if Result < 0 then
            case Errno is
               when Ewouldblock =>
                  null;
               when others =>
                  Perror ( "Error on write to port "
                           & Vx_Ip_Binding.Ip_Port'Image
                           ( A_Handle.The_Foreign_Port ) );
                  Close_Sockets ( A_Handle );
            end case;
         elsif Result = 0 then
            Close_Sockets ( A_Handle );
         else
            Bytes_Written := Natural ( Result );
         end if;
      end if;

   end Write;

   procedure Print_Status
     ( A_Handle : access Instance ) is
   -- Print the state of the IO_Medium object
   begin
      Log.Report ( "IO_Medium.UDP Local "
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Local_Address(1) )
                   & "."
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Local_Address(2) )
                   & "."
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Local_Address(3) )
                   & "."
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Local_Address(4) )
                   & "."
                   & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Local_Port ) );

      Log.Report ( "IO_Medium.UDP Foreign "
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Foreign_Address(1) )
                   & "."
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Foreign_Address(2) )
                   & "."
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Foreign_Address(3) )
                   & "."
                   & Interfaces.C.Unsigned_Char'Image
                   ( A_Handle.The_Foreign_Address(4) )
                   & "."
                   & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Foreign_Port ) );

      Log.Report ( "IO_Medium.UDP Input_Socket "
                   & Vx_Ip_Binding.Socket_Type'Image
                   ( A_Handle.The_Input_Socket ) & " "
                   & Boolean'Image ( A_Handle.Input_Socket_Valid ) );

      Log.Report ( "IO_Medium.UDP Output_Socket "
                   & Vx_Ip_Binding.Socket_Type'Image
                   ( A_Handle.The_Output_Socket ) & " "
                   & Boolean'Image ( A_Handle.Output_Socket_Valid ) );

   end Print_Status;

   procedure Set_Buffers
     ( A_Handle    : access Instance;
       Output_Size : in     Natural;
       Input_Size  : in     Natural
     ) is
   -- Set the maximum size of the medium's internal send and receive buffers
   begin

      A_Handle.Send_Buffer_Size := Output_Size;
      A_Handle.Receive_Buffer_Size := Input_Size;

   end Set_Buffers;

   procedure Set_Multicast_TTL
     ( A_Handle    : access Instance;
       TTL         : in Natural
     ) is
     Bad_Socket : exception;

     Result_1  : Interfaces.C.Int;
     TTL_Value : aliased Interfaces.C.Int := 8;

   begin

      Result_1 := Vx_Ip_Binding.Setsockopt ( S       => A_Handle.The_Output_Socket,
                                             Level   => Vx_Ip_Binding.Ipproto_Ip,
                                             Optname => Vx_Ip_Binding.Ip_multicast_ttl,
                                             Optval  => TTL_Value'Access,
                                             Optlen  => TTL_Value'Size / 8 );

      if Result_1 < 0 then
         Perror ( "Error on setsockopt (multicast) for TTL "
                  & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Local_Port ) );
         raise Bad_Socket;
      end if;

   end Set_Multicast_TTL;
   -- Set the maximum size of the medium's internal send and receive buffers

end Io_Medium.Udp;
