-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Rich Givis and Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Interfaces.C;
with Interfaces.C.Strings;
with Vx_Ip_Binding;
with Log;
with System.Storage_Elements;
package body Io_Medium.Tcp is

   use System.Storage_Elements;
   use type Interfaces.C.Int;
   use type Vx_Ip_Binding.Socket_Type;
   use type Vx_Ip_Binding.Ip_port;

   -- Subclass for protocol for communicating with subsystems
   -- via raw Ethernet

   -- errno
   Errno : Interfaces.C.Int;
   pragma Import (C, Errno, "errno");

   procedure Perror
     ( Str : in String ) is
      Saved_Errno : Interfaces.C.Int := Errno;
   begin
      Log.Report ( "TCP "
                   & Str & " errno = "
                   & Interfaces.C.Int'Image (Saved_Errno),
                   Log.Error );
   end Perror;

   Ewouldblock : constant Interfaces.C.Int := 70;

   function Create
     ( A_Port : in Natural;
       Max_Queued_Connections : in Natural )
     return Handle is
      -- Create an instance of the interface on the specified port

      A_Handle : Handle :=
       new Io_Medium.Tcp.Instance'( The_Transfers => 0,
                                    The_Ip_Port => Vx_Ip_Binding.Ip_Port ( A_Port ),
                                    The_Max_Queued_Connections =>
                                      Max_Queued_Connections,
                                    The_Listen_Socket => -1,
                                    The_Client_Socket => -1,
                                    Listen_Socket_Valid => False,
                                    Client_Socket_Valid => False,
                                    Send_Buffer_Size    => Default_Tcp_Sndbuf_Size,
                                    Receive_Buffer_Size => Default_Tcp_Rcvbuf_Size);

   begin

      return A_Handle;

   end Create;

   procedure Delete
     ( A_Handle : access Instance ) is
      -- Delete an instance of the interface

   begin

      Vx_Ip_Binding.Close ( A_Handle.The_Listen_Socket );
      Vx_Ip_Binding.Close ( A_Handle.The_Client_Socket );
      A_Handle.The_Listen_Socket := -1;
      A_Handle.The_Client_Socket := -1;
      A_Handle.Listen_Socket_Valid := False;
      A_Handle.Client_Socket_Valid := False;

   end Delete;

   procedure Initialize
     ( A_Handle : access Instance ) is
   -- Initialize the medium, creating any data structures necessary
   begin
      -- Initialize is the same as Reset.
      Reset ( A_Handle );
   end Initialize;

   procedure Create_Listen_Socket
     ( A_Port                 : in  Vx_Ip_Binding.Ip_Port;
       Max_Queued_Connections : in  Natural;
       Ip_Send_Size           : in  Natural;
       Ip_Receive_Size        : in  Natural;
       A_Socket               : out Vx_Ip_Binding.Socket_Type;
       Socket_Valid           : out Boolean
     ) is

      Sockaddr : aliased Vx_Ip_Binding.Sockaddr_Type;
      Option_Value : aliased Interfaces.C.Int;
      Result_1 : Interfaces.C.Int;
      Result_2 : Interfaces.C.Int;

      Bad_Socket : exception;

      Send_Buffer_Size    : aliased Interfaces.C.Int := Interfaces.C.Int(Ip_Send_Size);
      Receive_Buffer_Size : aliased Interfaces.C.Int := Interfaces.C.Int(Ip_Receive_Size);

   begin

      Log.Report ("Creating listen socket...");

      -- Assemble an address to accept any connection on the specified port

      Sockaddr := ( Sin_Len  => Vx_IP_Binding.Sockaddr_Size,
                    Sin_Family => Vx_Ip_Binding.Pf_Inet,
                    Sin_Port => Vx_Ip_Binding.Htons ( A_Port ),
                    Sin_Addr => Vx_Ip_Binding.Inaddr_Any,
                    Sin_Zero_Low => 0,
                    Sin_Zero_High => 0 );

      -- Create a socket

      A_Socket := Vx_Ip_Binding.Socket ( Vx_Ip_Binding.Pf_Inet,
                                         Vx_Ip_Binding.Sock_Stream, 0 );
      if A_Socket < 0 then
         Perror ( "Error on socket" );
         raise Bad_Socket;
      end if;

      -- Set socket options TCP_NODELAY, SO_REUSEADDR
      Option_Value := 1;
      Result_1 := Vx_Ip_Binding.Setsockopt ( A_Socket,
                                             Vx_Ip_Binding.Ipproto_Tcp,
                                             Vx_Ip_Binding.Tcp_Nodelay,
                                             Option_Value'Access,
                                             Option_Value'Size / 8 );

      Result_2 := Vx_Ip_Binding.Setsockopt ( A_Socket,
                                             Vx_Ip_Binding.Sol_Socket,
                                             Vx_Ip_Binding.So_Reuseaddr,
                                             Option_Value'Access,
                                             Option_Value'Size / 8 );


      if Result_1 < 0 or Result_2 < 0 then
         Perror ( "Error on setsockopt (listen)" );
         raise Bad_Socket;
      end if;

      -- Make socket non-blocking
      Result_1 := Vx_Ip_Binding.Ioctl ( A_Socket,
                                        Vx_Ip_Binding.Fionbio,
                                        Option_Value'Access );
      if Result_1 < 0 then
         Perror ( "Error on ioctl (listen)" );
         raise Bad_Socket;
      end if;


      -- Bind socket to address

      Result_1 := Vx_Ip_Binding.Bind ( A_Socket, Sockaddr'Access, Sockaddr'Size / 8 );

      if Result_1 < 0 then
         Perror ( "Error on bind" );
         raise Bad_Socket;
      end if;

      -- Listen
      Result_1 := Vx_Ip_Binding.Listen ( A_Socket,
                                         Max_Queued_Connections );
      if Result_1 < 0 then
         Perror ( "Error on listen" );
         raise Bad_Socket;
      end if;

      Socket_Valid := True;

      Log.Report ("Listen socket created...");

      -- Set the socket send buffer size
      if
        Vx_Ip_Binding.Setsockopt ( A_Socket,
                                   Vx_Ip_Binding.Sol_Socket,
                                   Vx_Ip_Binding.So_Sndbuf,
                                   Send_Buffer_Size'Access,
                                   (Send_Buffer_Size'Size + 7) / 8 ) < 0
      then
         Perror ( "Error on setsockopt send buffer increase" );
         raise Bad_Socket;
      end if;

      -- Set the socket's receive buffer size
      if
        Vx_Ip_Binding.Setsockopt ( A_Socket,
                                   Vx_Ip_Binding.Sol_Socket,
                                   Vx_Ip_Binding.So_Rcvbuf,
                                   Receive_Buffer_Size'Access,
                                   (Receive_Buffer_Size'Size + 7) / 8 ) < 0
      then
         Perror ( "Error on setsockopt receive buffer increase" );
         raise Bad_Socket;
      end if;

   exception

      -- If any of the above fail, close the listen socket and mark it invalid
      when Bad_Socket =>
         if A_Socket >=0 then
            Vx_Ip_Binding.Close ( A_Socket );
         end if;
         Socket_Valid := False;

   end Create_Listen_Socket;

   procedure Accept_Connection
     ( A_Listen_Socket : in Vx_Ip_Binding.Socket_Type;
       A_Client_Socket : out Vx_Ip_Binding.Socket_Type;
       Listen_Socket_Valid : out Boolean;
       Client_Socket_Valid : out Boolean ) is

      Client_Address : aliased Vx_Ip_Binding.Sockaddr_Type;
      Addrlen : aliased Interfaces.C.Int;

      Option_Value : aliased Interfaces.C.Int;

      Result_1 : Interfaces.C.Int;
      Result_2 : Interfaces.C.Int;

   begin

      -- assume listen socket is valid until otherwise learned
      Listen_Socket_Valid := True;

      -- assume client socket is invalid until otherwise learned
      Client_Socket_Valid := False;

      Addrlen := Vx_Ip_Binding.Sockaddr_Type'Size / 8;

      -- Accept (no-wait) connection on socket
      A_Client_Socket := Vx_Ip_Binding.Accept_Connection ( A_Listen_Socket,
                                                           Client_Address'Access,
                                                           Addrlen'Access );

      if A_Client_Socket < 0 then

         -- If any error except EWOULDBLOCK, close the listen socket and
         -- mark it invalid and return.

         case Errno is
            when Ewouldblock =>
               -- no connection attempt yet
               null;
            when others =>
               Perror ( "Error on accept" );
               Vx_Ip_Binding.Close ( A_Listen_Socket );
               Listen_Socket_Valid := False;
         end case;
         Client_Socket_Valid := False;

      else

         -- We have a connection: Set socket options TCP_NODELAY,
         -- SO_REUSEADDR.  If setting option fails, close the accept
         -- socket and mark it invalid.
         -- Set socket options TCP_NODELAY, SO_REUSEADDR
         Option_Value := 1;
         Result_1 := Vx_Ip_Binding.Setsockopt ( A_Client_Socket,
                                                Vx_Ip_Binding.Ipproto_Tcp,
                                                Vx_Ip_Binding.Tcp_Nodelay,
                                                Option_Value'Access,
                                                Option_Value'Size / 8 );

         Result_2 := Vx_Ip_Binding.Setsockopt ( A_Client_Socket,
                                                Vx_Ip_Binding.Sol_Socket,
                                                Vx_Ip_Binding.So_Reuseaddr,
                                                Option_Value'Access,
                                                Option_Value'Size / 8 );


         if Result_1 < 0 or Result_2 < 0 then
         Perror ( "Error on setsockopt (client)" );
            Vx_Ip_Binding.Close ( A_Client_Socket );
            Client_Socket_Valid := False;
         else
            Client_Socket_Valid := True;
         end if;

         -- Make socket non-blocking

         Result_1 := Vx_Ip_Binding.Ioctl ( A_Client_Socket,
                                           Vx_Ip_Binding.Fionbio,
                                           Option_Value'Access );
         if Result_1 < 0 then
            Perror ( "Error on ioctl (client)" );
            Vx_Ip_Binding.Close ( A_Client_Socket );
            Client_Socket_Valid := False;
         end if;

      end if;

   end Accept_Connection;

   procedure Reset
     ( A_Handle : access Instance ) is
      -- Reset the medium, reinitializing as needed
   begin

      -- *** If the listen socket is invalid, create new listen socket ***
      if not A_Handle.Listen_Socket_Valid then

         Create_Listen_Socket
           ( A_Handle.The_Ip_Port,
             A_Handle.The_Max_Queued_Connections,
             A_Handle.Send_Buffer_Size,
             A_Handle.Receive_Buffer_Size,
             A_Handle.The_Listen_Socket,
             A_Handle.Listen_Socket_Valid);

      end if;

      -- If the client socket is invalid (no client), try to accept a
      -- client connection.  (Don't bother if the listen socket is
      -- down.)

      if A_Handle.Listen_Socket_Valid and not A_Handle.Client_Socket_Valid then

         Accept_Connection
           ( A_Handle.The_Listen_Socket,
             A_Handle.The_Client_Socket,
             A_Handle.Listen_Socket_Valid,
             A_Handle.Client_Socket_Valid );

      end if;

   end Reset;

   function Is_Up
     ( A_Handle : access Instance )
      return Boolean is
      -- Report status of the medium
   begin
      return A_Handle.Listen_Socket_Valid and A_Handle.Client_Socket_Valid;
   end Is_Up;

   procedure Read
     ( A_Handle : access Instance;
       A_Buffer_Pointer : in System.Address;
       A_Length : in Natural;
       Bytes_Read : out Natural ) is

      -- Receive I/O buffer via the medium

      Result : Interfaces.C.Int;
      Current_Buffer_Pointer : System.Address := A_Buffer_Pointer;
      Current_Length : Natural;
      Short_Reads : Natural := 0;
      Max_Short_Reads : constant := 2;

   begin

      -- initialize output variable
      Bytes_Read := 0;

      while Bytes_Read < A_Length loop

         -- set pointer and length
         Current_Buffer_Pointer
           := A_Buffer_Pointer + System.Storage_Elements.Storage_Offset ( Bytes_Read );
         Current_Length := A_Length - Bytes_Read;

         -- Call recv to read data into the buffer
         Result := Vx_Ip_Binding.Recv ( A_Handle.The_Client_Socket,
                                        Current_Buffer_Pointer,
                                        Interfaces.C.Int(Current_Length),
                                        0 );

         -- Check for errors -- flag if no data is read, or a
         -- non-EWOULDBLOCK error occurs. If a non-EWOULDBLOCK error log
         -- an error and close the Client_Socket. If EWOULDBLOCK log a
         -- warning and continue.

         if Result < 0 then
            case Errno is
               when Ewouldblock =>
                  null;
               when others =>
                  Perror ( "Error on read" );
                  Vx_Ip_Binding.Close ( A_Handle.The_Client_Socket );
                  A_Handle.Client_Socket_Valid := False;
            end case;
         elsif Result = 0 then
            Vx_Ip_Binding.Close ( A_Handle.The_Client_Socket );
            A_Handle.Client_Socket_Valid := False;
         else
            Bytes_Read := Bytes_Read + Natural ( Result );
            if Bytes_Read < A_Length then
               Short_Reads := Short_Reads + 1;
               if Short_Reads >= Max_Short_Reads then
                  Log.Report ( "Short TCP recv: "
                               & Integer'Image ( Short_Reads )
                               & " attempts, "
                               & Integer'Image ( A_Length )
                               & " attempted, "
                               & Integer'Image ( Bytes_Read )
                               & " received.",
                               Log.Warning );
                  exit;
               end if;
            end if;
         end if;

      end loop;

   end Read;

   procedure Write
     ( A_Handle : access Instance;
       A_Buffer_Pointer : in System.Address;
       A_Length : in Natural;
       Bytes_Written : out Natural ) is

      -- Transmit I/O buffer via the medium

      Result : Interfaces.C.Int;
      Current_Buffer_Pointer : System.Address := A_Buffer_Pointer;
      Current_Length : Natural;
      Short_Writes : Natural := 0;
      Max_Short_Writes : Natural := 2;

   begin

      Bytes_Written := 0;

      while Bytes_Written < A_Length loop

         -- set pointer and length
         Current_Buffer_Pointer
           := A_Buffer_Pointer + System.Storage_Elements.Storage_Offset ( Bytes_Written );
         Current_Length := A_Length - Bytes_Written;

         Result := Vx_Ip_Binding.Send ( A_Handle.The_Client_Socket,
                                        Current_Buffer_Pointer,
                                        Interfaces.C.Int(Current_Length),
                                        0 );

         -- Check for errors -- flag if no data is written, or a
         -- non-EWOULDBLOCK error occurs. If a non-EWOULDBLOCK error log
         -- an error and close the Client_Socket. If EWOULDBLOCK log a
         -- warning and continue.

         if Result < 0 then
            case Errno is
               when Ewouldblock =>
                  null;
               when others =>
                  Perror ( "Error on write" );
                  Vx_Ip_Binding.Close ( A_Handle.The_Client_Socket );
                  A_Handle.Client_Socket_Valid := False;
            end case;
         elsif Result = 0 then
            Vx_Ip_Binding.Close ( A_Handle.The_Client_Socket );
            A_Handle.Client_Socket_Valid := False;
         else
            Bytes_Written := Bytes_Written + Natural ( Result );
            if Bytes_Written < A_Length then
               Short_Writes := Short_Writes + 1;
               if Short_Writes >= Max_Short_Writes then
                  Log.Report ( "Short TCP send: "
                               & Integer'Image ( Short_Writes )
                               & " attempts, "
                               & Integer'Image ( A_Length )
                               & " bytes attempted, "
                               & Integer'Image ( Bytes_Written )
                               & " bytes sent.",
                               Log.Warning );
                  exit;
               end if;
            end if;
         end if;
      end loop;

   end Write;

   procedure Print_Status
     ( A_Handle : access Instance ) is
   -- Print the state of the IO_Medium object
   begin
      Log.Report ( "IO_Medium.TCP Port "
                   & Vx_Ip_Binding.Ip_Port'Image ( A_Handle.The_Ip_Port )
                   & ", Max Queued Connections "
                   & Natural'Image
                   ( A_Handle.The_Max_Queued_Connections ) );
      Log.Report ( "TCP Listen_Socket "
                   & Vx_Ip_Binding.Socket_Type'Image
                   ( A_Handle.The_Listen_Socket )
                   & " "
                   & Boolean'Image ( A_Handle.Listen_Socket_Valid ) );
      Log.Report ( "TCP Client_Socket "
                   & Vx_Ip_Binding.Socket_Type'Image
                   ( A_Handle.The_Client_Socket )
                   & " "
                   & Boolean'Image ( A_Handle.Client_Socket_Valid ) );

   end Print_Status;

   procedure Set_Buffers
     ( A_Handle    : access Instance;
       Output_Size : in     Natural;
       Input_Size  : in     Natural
     ) is
   -- Set the maximum size of the medium's internal send and receive buffers
   begin

      A_Handle.Send_Buffer_Size := Output_Size;
      A_Handle.Receive_Buffer_Size := Input_Size;

   end Set_Buffers;

   procedure Set_Multicast_TTL
     ( A_Handle    : access Instance;
       TTL         : in Natural
     ) is
   -- Set the time to live of the mediums send packets
   begin
        null;
   end Set_Multicast_TTL; 

end Io_Medium.Tcp;
