-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Io_Strings;
with Ada.Strings.Maps.Constants;
with Bc.Containers;
with Bc.Containers.Maps;
with Bc.Containers.Maps.Bounded;
--with Bc.Containers.Maps.Unbounded;
with Global_Heap;
with Ada.Text_Io;
with Address_Io;
with System.Address_To_Access_Conversions;
with System.Storage_Elements;
pragma Elaborate_All ( Bc.Containers,
                       Bc.Containers.Maps,
                       Bc.Containers.Maps.bounded,
                       System.Address_To_Access_Conversions );
package body Io_Map is

   use type IO_Strings.Unbounded_String;
   use System.Storage_Elements;

   -- According to RM 13.5.3, the following test should be accurate
   --
   --   Host_Is_Network_Order : constant Boolean
   --      := ( System.Default_Bit_Order = System.High_Order_First );
   --
   -- but the Green Hills-supplied spec for package System seems to be
   -- set up for a big-endian Sparc processor, not a little-endian
   -- Intel processor, so until this is fixed, we will have to
   -- hard-code the value.

   Host_Is_Network_Order : constant Boolean := False;

   procedure Print
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean );

   Debug_Io_Map : Boolean := False;

   -- A hash function, created by summing the characters in the string.

   function Sum_String ( V : Io_Strings.Unbounded_String )
     return Positive is
      Result : Positive := 1;
   begin
      for Index in 1 .. Io_Strings.Length(V) loop
         Result :=
           Result + Character'Pos(Io_Strings.Element(V,Index));
      end loop;
      return Result;
   end Sum_String;

   Buckets : constant Positive := 257;


   -- Instantiations of Booch components.  The generics build on each
   -- other.  The mapping is from unbounded_strings to io_map_entries.

   -- Maps is an abstract map class; Concrete_Maps specifies the form and
   -- storage management policy.  We are using Unbounded, but Bounded will
   -- also work.

   package Containers is new
     Bc.Containers ( Item => Io_Strings.Unbounded_String,
                     "="  => Io_Strings."=" );

   package Maps is new
     Containers.Maps ( Value => Io_Map_Entry.Instance,
                       "="   => Io_Strings."=" );


--  Declarations for using the bounded form

   Max_Bucket_Size : constant Positive := 20;

   package Concrete_Maps is new
     Maps.Bounded
     ( Hash => Sum_String,
       Buckets => Buckets,
       Size => Max_Bucket_Size );

   subtype Concrete_Map is Concrete_Maps.Bounded_Map;


   -- We use a type rename here to make it simple to change the concrete
   -- form of the Map abstraction--

--  Declarations for using the unbounded form

-- Uncomment the following to use the unbounded map abstraction instead.
--
--   package Concrete_Maps is new
--     Maps.Unbounded
--    ( Hash => Sum_String,
--      Buckets => Buckets,
--      Storage_Manager => Global_Heap.Pool,
--      Storage => Global_Heap.Storage );
--
--   subtype Concrete_Map is Concrete_Maps.Unbounded_Map;

   -- For more efficient traversal during real-time, we can convert
   -- the map to a flat array, since we don't have to look up names
   -- during real-time.  In fact, we don't care about names at all,
   -- just input addresses, output addresses, and data size.

   type Entry_Array is
     array ( Natural range <> ) of Io_Map_Entry.Instance;

   type Entry_Array_Handle is
     access Entry_Array;

   type Instance is new Concrete_Map with record
      Is_Converted_For_Real_Time : Boolean := False;
      Scatter_Gather_Iterator : Containers.Iterator;
      The_Number_Of_Entries : Natural := 0;
      The_Entry_Array : Entry_Array_Handle;
   end record;

   function Create return Handle is
      New_Instance : constant Handle := new Instance;
   begin
      New_Instance.Scatter_Gather_Iterator := New_Iterator (New_Instance.all);
      New_Instance.The_Number_Of_Entries := 0;
      New_Instance.Is_Converted_For_Real_Time := False;
      return New_Instance;
   end Create;

   function Is_Defined
     ( A_Mnemonic : in String;
       A_Map : in Handle )
     return Boolean is

   begin

      return Concrete_Maps.Is_Bound
        ( Concrete_Map (A_Map.all),
          Io_Strings.Translate
          ( Source  => Io_Strings.To_Unbounded_String (A_Mnemonic),
            Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map ) );

   end Is_Defined;

   procedure Find_Entry
     ( A_Mnemonic : in String;
       A_Map : in Handle;
       An_Entry : out Io_Map_Entry.Instance;
       Was_Found : out Boolean ) is

   begin

      An_Entry  :=
        Concrete_Maps.Value_Of
        ( Concrete_Map (A_Map.all),
          Io_Strings.Translate
          ( Source  => Io_Strings.To_Unbounded_String (A_Mnemonic),
            Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map ) );

      Was_Found := True;

   exception

      when Bc.Not_Found =>
         Was_Found := False;

   end Find_Entry;


   procedure Add_Entry
     ( A_Mnemonic : in String;
       A_Map : in Handle;
       An_Entry : out Io_Map_Entry.Instance ) is
   begin

      if A_Map.all.Is_Converted_For_Real_Time then
         -- if it's already been converted don't allow additions
         raise Map_Is_Read_Only;
      else
         Concrete_Maps.Bind
           ( Concrete_Map(A_Map.all),
             Io_Strings.Translate
             ( Source  => Io_Strings.To_Unbounded_String (A_Mnemonic),
               Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map ),
             An_Entry );
         A_Map.The_Number_Of_Entries := A_Map.The_Number_Of_Entries + 1;
      end if;

   end Add_Entry;

   procedure Set_Entry
     ( A_Mnemonic : in String;
       A_Map : in Handle;
       An_Entry : in Io_Map_Entry.Instance;
       Was_Found : out Boolean ) is

   begin

      if A_Map.all.Is_Converted_For_Real_Time then
         -- if it's already been converted don't allow changes
         raise Map_Is_Read_Only;
      else
         Was_Found := True;

         Concrete_Maps.Rebind
           ( Concrete_Map(A_Map.all),
             Io_Strings.Translate
             ( Source  => Io_Strings.To_Unbounded_String (A_Mnemonic),
               Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map ),
             An_Entry );
      end if;

   exception

      when Bc.Not_Found =>
         Was_Found := False;

   end Set_Entry;

   -- For purposes of copying data between the host and the subsystem, we
   -- don't care about the structure.  We only care about the size.
   -- We treat all I/O data as if it were unsigned ints, except for the
   -- 8 byte data, which we treat as Long_Float (which it probably really
   -- is), because there is not 8 byte integer in Green Hills Ada.

   type Unsigned_1_Byte is mod  2**(8*1);
   type Unsigned_2_Bytes is mod 2**(8*2);
   type Unsigned_4_Bytes is mod 2**(8*4);

   package Unsigned_1_Byte_Conversions is new
     System.Address_To_Access_Conversions (Unsigned_1_Byte);

   package Unsigned_2_Byte_Conversions is new
     System.Address_To_Access_Conversions (Unsigned_2_Bytes);

   package Unsigned_4_Byte_Conversions is new
     System.Address_To_Access_Conversions (Unsigned_4_Bytes);

   package Long_Float_Conversions is new
     System.Address_To_Access_Conversions (Long_Float);

   type Unsigned_2_Byte_Array is
     array ( Natural range 0 .. 1 ) of Unsigned_1_Byte;

   type Unsigned_4_Byte_Array is
     array ( Natural range 0 .. 3 ) of Unsigned_1_Byte;

   type Unsigned_8_Byte_Array is
     array ( Natural range 0 .. 7 ) of Unsigned_1_Byte;

   package Unsigned_2_Byte_Array_Conversions is new
     System.Address_To_Access_Conversions (Unsigned_2_Byte_Array);

   package Unsigned_4_Byte_Array_Conversions is new
     System.Address_To_Access_Conversions (Unsigned_4_Byte_Array);

   package Unsigned_8_Byte_Array_Conversions is new
     System.Address_To_Access_Conversions (Unsigned_8_Byte_Array);

   function Flip ( X : in Unsigned_2_Byte_Array )
     return Unsigned_2_Byte_Array is
   begin
      return ( 0 => X(1), 1 => X(0) );
   end Flip;

   function Flip ( X : in Unsigned_4_Byte_Array )
     return Unsigned_4_Byte_Array is
   begin
      return ( 0 => X(3), 1 => X(2), 2 => X(1), 3 => X(0) );
   end Flip;

   function Flip ( X : in Unsigned_8_Byte_Array )
     return Unsigned_8_Byte_Array is
   begin
      return ( 0 => X(7), 1 => X(6), 2 => X(5), 3 => X(4),
               4 => X(3), 5 => X(2), 6 => X(1), 7 => X(0) );
   end Flip;

   pragma Inline (Flip);

   -- The following procedures, Copy_Host_To_Buffer, Copy_Buffer_To_Host,
   -- and Print, are operations on a key-value pair in the map, used to
   -- instantiate Maps.Visit, which will executes a procedure
   -- on each key-value pair in the map.

   procedure Copy_Host_To_Buffer
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean ) is

      -- Copy memory from the host variable to the correct place in the
      -- outgoing I/O buffer.

   begin

      if Debug_Io_Map then
         Print ( I, V, Ok );
      end if;

      -- Only try to copy if an Ada variable has been mapped to the ICD entry

      if V.Is_Mapped then

-- A really clever bit of code that doesn't work because the expression for
-- Current_Size_In_Bits is not static and modulus types must be defined
-- with static expressions.
--
--       Dereference_Block:
--       declare
--          Current_Size_In_Bits : constant := 2**(The_Current_Item.The_Size*8);
--          type Current_Unsigned_Int_Type is mod Current_Size_In_Bits;
--          The_Buffer_Value : Current_Unsigned_Int_Type;
--          for The_Buffer_Value'Address use The_Current_Item.The_Buffer_Address;
--          The_Host_Value : Current_Unsigned_Int_Type;
--          for The_Host_Value'Address use The_Current_Item.The_Host_Address;
--       begin
--          The_Buffer_Value := The_Host_Value;
--       end Dereference_Block;

         case V.The_Size is
            when 0 =>
               null;
            when 1 =>
               Unsigned_1_Byte_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Unsigned_1_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all;
            when 2 =>
               Unsigned_2_Byte_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Unsigned_2_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all;
            when 4 =>
               Unsigned_4_Byte_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Unsigned_4_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all;
            when 8 =>
               Long_Float_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Long_Float_Conversions.
                 To_Pointer(V.The_Host_Address).all;

            when others =>

               -- for larger numbers divisible by 4, treat as array of 4-byte-words
               if V.The_Size mod 4 = 0 then
                  declare
                     Starting_Host_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( V.The_Host_Address );
                     Starting_Buffer_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( V.The_Buffer_Address );

                     Current_Host_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Buffer_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Offset
                       : System.Storage_Elements.Integer_Address;

                  begin
                     for I in 0 .. ( V.The_Size / 4 ) - 1 loop

                        Current_Offset
                          := System.Storage_Elements.Integer_Address ( I * 4 );
                        Current_Host_Address
                          := Starting_Host_Address + Current_Offset;
                        Current_Buffer_Address
                          := Starting_Buffer_Address + Current_Offset;

                        Unsigned_4_Byte_Conversions.
                          To_Pointer
                          ( System.Storage_Elements.To_Address
                            ( Current_Buffer_Address ) ).all
                          :=
                          Unsigned_4_Byte_Conversions.
                          To_Pointer
                          ( System.Storage_Elements.To_Address
                            ( Current_Host_Address ) ).all;

                     end loop;
                  end;

               else -- not divisible by 4

                  -- do nothing for any other case;
                  null;

               end if;

         end case;

      end if;

      Ok := True;

   end Copy_Host_To_Buffer;

   procedure Copy_Buffer_To_Host
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Secondary : Boolean;
       Ok : out Boolean ) is
      -- Copy memory from the incoming I/O buffer to the correct host variable
      Source : System.Address;
   begin

      -- Only try to copy if an Ada variable has been mapped to the ICD entry

      if Debug_Io_Map then
         Print ( I, V, Ok );
      end if;

      if V.Is_Mapped then

         if Secondary then
            Source := V.Secondary_Buffer;
         else
            Source := V.The_Buffer_Address;
         end if;

         case V.The_Size is
            when 0 =>
               null;
            when 1 =>
               Unsigned_1_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Unsigned_1_Byte_Conversions.
                 To_Pointer(Source).all;
            when 2 =>
               Unsigned_2_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Unsigned_2_Byte_Conversions.
                 To_Pointer(Source).all;
            when 4 =>
               Unsigned_4_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Unsigned_4_Byte_Conversions.
                 To_Pointer(Source).all;
            when 8 =>
               Long_Float_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Long_Float_Conversions.
                 To_Pointer(Source).all;

            when others =>

               -- for larger numbers divisible by 4, treat as array of 4-byte-words
               if V.The_Size mod 4 = 0 then
                  declare
                     Starting_Host_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( V.The_Host_Address );
                     Starting_Buffer_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( Source );

                     Current_Host_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Buffer_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Offset
                       : System.Storage_Elements.Integer_Address;

                  begin
                     for I in 0 .. ( V.The_Size / 4 ) - 1 loop

                        Current_Offset
                          := System.Storage_Elements.Integer_Address ( I * 4 );
                        Current_Host_Address
                          := Starting_Host_Address + Current_Offset;
                        Current_Buffer_Address
                          := Starting_Buffer_Address + Current_Offset;

                        Unsigned_4_Byte_Conversions.
                          To_Pointer
                          ( System.Storage_Elements.To_Address
                            ( Current_Host_Address ) ).all
                          :=
                          Unsigned_4_Byte_Conversions.
                          To_Pointer
                          ( System.Storage_Elements.To_Address
                            ( Current_Buffer_Address ) ).all;

                     end loop;
                  end;

               else -- not divisible by 4

                  -- do nothing for any other case;
                  null;

               end if;

         end case;

      end if;

      Ok := True;

   end Copy_Buffer_To_Host;

   procedure Copy_Buffer_To_Host
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean ) is
   begin
      Copy_Buffer_To_Host
        ( I => I,
          V => V,
          Secondary => False,
          Ok => Ok
          );
   end Copy_Buffer_To_Host;

   procedure Copy_Secondary_Buffer_To_Host
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean ) is
   begin
      Copy_Buffer_To_Host
        ( I => I,
          V => V,
          Secondary => True,
          Ok => Ok
          );
   end Copy_Secondary_Buffer_To_Host;


   procedure Copy_And_Flip_Host_To_Buffer
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean ) is

      -- Copy memory from the host variable to the correct place in the
      -- outgoing I/O buffer.

   begin

      if Debug_Io_Map then
         Print ( I, V, Ok );
      end if;

      -- Only try to copy if an Ada variable has been mapped to the ICD entry

      if V.Is_Mapped then

         case V.The_Size is
            when 0 =>
               null;
            when 1 =>
               Unsigned_1_Byte_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Unsigned_1_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all;
            when 2 =>
               Unsigned_2_Byte_Array_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Flip ( Unsigned_2_Byte_Array_Conversions.
                        To_Pointer(V.The_Host_Address).all );
            when 4 =>
               Unsigned_4_Byte_Array_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Flip ( Unsigned_4_Byte_Array_Conversions.
                        To_Pointer(V.The_Host_Address).all );
            when 8 =>
               Unsigned_8_Byte_Array_Conversions.
                 To_Pointer(V.The_Buffer_Address).all
                 :=
                 Flip ( Unsigned_8_Byte_Array_Conversions.
                        To_Pointer(V.The_Host_Address).all );

            when others =>

               -- for larger numbers divisible by 4, treat as array of 4-byte-words
               if V.The_Size mod 4 = 0 then
                  declare
                     Starting_Host_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( V.The_Host_Address );
                     Starting_Buffer_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( V.The_Buffer_Address );

                     Current_Host_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Buffer_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Offset
                       : System.Storage_Elements.Integer_Address;

                  begin

                     for I in 0 .. ( V.The_Size / 4 ) - 1 loop

                        Current_Offset
                          := System.Storage_Elements.Integer_Address ( I * 4 );
                        Current_Host_Address
                          := Starting_Host_Address + Current_Offset;
                        Current_Buffer_Address
                          := Starting_Buffer_Address + Current_Offset;

                        Unsigned_4_Byte_Array_Conversions.
                          To_Pointer
                          ( System.Storage_Elements.To_Address
                            ( Current_Buffer_Address ) ).all
                          :=
                          Flip ( Unsigned_4_Byte_Array_Conversions.
                                 To_Pointer
                                 ( System.Storage_Elements.To_Address
                                   ( Current_Host_Address ) ).all );

                     end loop;
                  end;

               else -- not divisible by 4

                  -- do nothing for any other case;
                  null;

               end if;

         end case;

      end if;

      Ok := True;

   end Copy_And_Flip_Host_To_Buffer;

   procedure Copy_And_Flip_Buffer_To_Host
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Secondary : Boolean;
       Ok : out Boolean ) is
      -- Copy memory from the incoming I/O buffer to the correct host variable

      Source : System.Address;
   begin

      -- Only try to copy if an Ada variable has been mapped to the ICD entry

      if Debug_Io_Map then
         Print ( I, V, Ok );
      end if;

      if V.Is_Mapped then

         if Secondary then
            Source := V.Secondary_Buffer;
         else
            Source := V.The_Buffer_Address;
         end if;

         case V.The_Size is
            when 1 =>
               Unsigned_1_Byte_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Unsigned_1_Byte_Conversions.
                 To_Pointer(Source).all;
            when 2 =>
               Unsigned_2_Byte_Array_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Flip ( Unsigned_2_Byte_Array_Conversions.
                        To_Pointer(Source).all );
            when 4 =>
               Unsigned_4_Byte_Array_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Flip ( Unsigned_4_Byte_Array_Conversions.
                        To_Pointer(Source).all );
            when 8 =>
               Unsigned_8_Byte_Array_Conversions.
                 To_Pointer(V.The_Host_Address).all
                 :=
                 Flip ( Unsigned_8_Byte_Array_Conversions.
                        To_Pointer(Source).all );

            when others =>
               -- for larger numbers divisible by 4, treat as array of 4-byte-words
               if V.The_Size mod 4 = 0 then
                  declare
                     Starting_Host_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( V.The_Host_Address );
                     Starting_Buffer_Address
                       : System.Storage_Elements.Integer_Address
                       := System.Storage_Elements.To_Integer
                         ( Source );
                     Current_Host_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Buffer_Address
                       : System.Storage_Elements.Integer_Address;
                     Current_Offset
                       : System.Storage_Elements.Integer_Address;

                  begin
                     for I in 0 .. ( V.The_Size / 4 ) - 1 loop

                        Current_Offset
                          := System.Storage_Elements.Integer_Address ( I * 4 );
                        Current_Host_Address
                          := Starting_Host_Address + Current_Offset;
                        Current_Buffer_Address
                          := Starting_Buffer_Address + Current_Offset;

                        Unsigned_4_Byte_Array_Conversions.
                          To_Pointer
                          ( System.Storage_Elements.To_Address
                            ( Current_Host_Address ) ).all
                          :=
                          Flip ( Unsigned_4_Byte_Array_Conversions.
                                 To_Pointer
                                 ( System.Storage_Elements.To_Address
                                   ( Current_Buffer_Address ) ).all );

                     end loop;
                  end;

               else -- not divisible by 4

                  -- do nothing for any other case;
                  null;

               end if;

         end case;

      end if;

      Ok := True;

   end Copy_And_Flip_Buffer_To_Host;

   procedure Copy_And_Flip_Buffer_To_Host
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean ) is
   begin
      Copy_And_Flip_Buffer_To_Host (I => I, V => V, Secondary => False, Ok => Ok);
   end Copy_And_Flip_Buffer_To_Host;

   procedure Copy_And_Flip_Secondary_Buffer_To_Host
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean ) is
   begin
      Copy_And_Flip_Buffer_To_Host (I => I, V => V, Secondary => True, Ok => Ok);
   end Copy_And_Flip_Secondary_Buffer_To_Host;

   procedure Print
     ( I : Io_Strings.Unbounded_String;
       V : Io_Map_Entry.Instance;
       Ok : out Boolean ) is

   begin

      Ada.Text_Io.Put
        ( Io_Strings.To_String(I) & " : " &
          Natural'Image (V.The_Size) & " : " &
          Io_Types.Direction'Image(V.The_Direction) & " : " &
          Boolean'Image (V.Is_Mapped) & " : " );
      Address_Io.Put
        ( System.Storage_Elements.To_Integer
          ( V.The_Buffer_Address ),
          Width => 8, Base => 16 );
      Ada.Text_Io.Put ( " : " );
      Address_Io.Put
        ( System.Storage_Elements.To_Integer
          ( V.The_Host_Address ),
          Width => 8, Base => 16 );
      Ada.Text_Io.New_Line;
      Ok := True;

   end Print;


   -- Generic instantiations which visit each entry in the map and
   -- apply the supplied procedure to each.

   procedure Gather_Each_Entry is new
     Maps.Visit_With_Iterator ( Apply => Copy_Host_To_Buffer );
   -- copy the designated amount of data from the host variable to the
   -- I/O buffer location (for outputs)

   procedure Scatter_Each_Entry is new
     Maps.Visit_With_Iterator ( Apply => Copy_Buffer_To_Host );
   -- copy the designated amount of data from the I/O buffer location
   -- to the host variable (for inputs)

   procedure Scatter_Each_Entry_From_Secondary is new
     Maps.Visit_With_Iterator ( Apply => Copy_Secondary_Buffer_To_Host );
   -- copy the designated amount of data from the secondary I/O buffer location
   -- to the host variable (for inputs)

   procedure Gather_And_Flip_Each_Entry is new
     Maps.Visit_With_Iterator ( Apply => Copy_And_Flip_Host_To_Buffer );
   -- copy and change endian-ness of the designated amount of data
   -- from the host variable to the I/O buffer location (for outputs)

   procedure Scatter_And_Flip_Each_Entry is new
     Maps.Visit_With_Iterator ( Apply => Copy_And_Flip_Buffer_To_Host );
   -- copy and change endian-ness the designated amount of data from
   -- the I/O buffer location to the host variable (for inputs)

   procedure Scatter_And_Flip_Each_Entry_From_Secondary is new
     Maps.Visit_With_Iterator ( Apply => Copy_And_Flip_Secondary_Buffer_To_Host );
   -- copy and change endian-ness the designated amount of data from
   -- the secondary I/O buffer location to the host variable (for inputs)

   procedure Print_Each_Entry is new
     Maps.Visit_With_Iterator ( Apply => Print );
   -- print the name, size, status, and addresses of each variable in
   -- the map

   -- These procedures just dereference the Map handle to call the
   -- instantiated generic procedures above

   procedure Gather
     ( A_Map : in Handle;
       Buffer_Is_Network_Order : in Boolean ) is
      Ok : Boolean;
   begin
      if Host_Is_Network_Order = Buffer_Is_Network_Order then
         if A_Map.Is_Converted_For_Real_Time then
            for Index in 1 .. A_Map.The_Number_Of_Entries loop
               Copy_Host_To_Buffer
                 ( I => Io_Strings.Null_Unbounded_String,
                   V => A_Map.The_Entry_Array.all(Index),
                   Ok => Ok );
            end loop;
         else
            Gather_Each_Entry ( A_Map.Scatter_Gather_Iterator );
         end if;
      else
         if A_Map.Is_Converted_For_Real_Time then
            for Index in 1 .. A_Map.The_Number_Of_Entries loop
               Copy_And_Flip_Host_To_Buffer
                 ( I => Io_Strings.Null_Unbounded_String,
                   V => A_Map.The_Entry_Array.all(Index),
                   Ok => Ok );
            end loop;
         else
            Gather_And_Flip_Each_Entry ( A_Map.Scatter_Gather_Iterator );
         end if;
      end if;
   end Gather;

   procedure Scatter
     ( A_Map : in Handle;
       Buffer_Is_Network_Order : in Boolean;
       Use_Secondary_Buffer    : in Boolean) is
      Ok : Boolean;
   begin
      if Host_Is_Network_Order = Buffer_Is_Network_Order then
         if A_Map.Is_Converted_For_Real_Time then
            for Index in 1 .. A_Map.The_Number_Of_Entries loop
               Copy_Buffer_To_Host
                 ( I => Io_Strings.Null_Unbounded_String,
                   V => A_Map.The_Entry_Array.all(Index),
                   Secondary => Use_Secondary_Buffer,
                   Ok => Ok );
            end loop;
         else
            if Use_Secondary_Buffer then
               Scatter_Each_Entry_From_Secondary ( A_Map.Scatter_Gather_Iterator );
            else
               Scatter_Each_Entry ( A_Map.Scatter_Gather_Iterator );
            end if;
         end if;
      else
         if A_Map.Is_Converted_For_Real_Time then
            for Index in 1 .. A_Map.The_Number_Of_Entries loop
               Copy_And_Flip_Buffer_To_Host
                 ( I => Io_Strings.Null_Unbounded_String,
                   V => A_Map.The_Entry_Array.all(Index),
                   Secondary => Use_Secondary_Buffer,
                   Ok => Ok );
            end loop;
         else
            if Use_Secondary_Buffer then
               Scatter_And_Flip_Each_Entry_From_Secondary ( A_Map.Scatter_Gather_Iterator );
            else
               Scatter_And_Flip_Each_Entry ( A_Map.Scatter_Gather_Iterator );
            end if;
         end if;
      end if;
   end Scatter;

   procedure Print
     ( A_Map : in Handle ) is
   begin
      Print_Each_Entry ( A_Map.Scatter_Gather_Iterator );
   end Print;

   procedure Convert_For_Real_Time
     ( A_Map : in Handle ) is
   -- Convert internal implementation of map from structure which is
   -- efficient for initialization to a structure which is efficient
   -- for real-time.  After this method is called, the map becomes
   -- read-only, and calling the Add_Entry and Set_Entry methods will
   -- raise an exception.

      Index : Natural := 0;

   begin

      if A_Map.Is_Converted_For_Real_Time then
         -- if it's already been converted don't do it again
         raise Map_Is_Read_Only;
      else

         -- set the flag to prevent further additions or changes
         A_Map.Is_Converted_For_Real_Time := True;

         if ( A_Map.The_Number_Of_Entries > 0 ) then
            -- ignore for zero length maps

            -- allocate the entry_array
            A_Map.The_Entry_Array := new Entry_Array ( 1 .. A_Map.The_Number_Of_Entries );

            -- iterate through the map and copy data from the map to the entry array
            Containers.Reset ( A_Map.Scatter_Gather_Iterator );
            Index := 0;

            while not Containers.Is_Done ( A_Map.Scatter_Gather_Iterator ) loop
               Index := Index + 1;
               A_Map.The_Entry_Array.all(Index)
                 := Maps.Current_Value ( A_Map.Scatter_Gather_Iterator );
               Containers.Next ( A_Map.Scatter_Gather_Iterator );
            end loop;

         end if;


      end if;


   end Convert_For_Real_Time;

end Io_Map;
