-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with System, Io_Types, Io_Buffer, Io_Medium, Io_Map_Entry, Io_Map;
with Ada.Strings.Unbounded;
with Ada.Streams;
package Io_Interface is

   -- Implementation of interface, binding ICD file, I/O medium, I/O
   -- dictionaries, and I/O buffers

   -- We're maintaining separate buffers and maps for each direction of I/O
   -- to speed up and simplify transfers in real-time

   type Io_Buffer_Array is
      array (Io_Types.Direction) of Io_Buffer.Handle;

   type Io_Dictionary_Array is
      array (Io_Types.Direction) of Io_Map.Handle;

   type Natural_Direction_Array is
      array (Io_Types.Direction) of Natural;

   type Instance is
      record
         Is_Valid : Boolean := False;
         The_Name         : Ada.Strings.Unbounded.Unbounded_String;  -- interface name
         The_Icd_Filename : Ada.Strings.Unbounded.Unbounded_String;  -- ICD filename
         Buffer_Is_Network_Order : Boolean; -- does other end of interface expect network order
         The_Buffer_Sizes : Natural_Direction_Array;
         The_Medium       : Io_Medium.Handle;    -- medium for interface
         The_Maps         : IO_Dictionary_Array; -- maps from host to buffer
         The_Buffers      : IO_Buffer_Array;     -- input & output buffers
         The_Bit_Bucket   : Io_Buffer.Handle;    -- for discarding inputs
		 Transfer_Count   : Natural_Direction_Array; -- count of transfers in/out
		 Error_Count      : Natural_Direction_Array; -- count of errors in/out
		 The_Version      : Natural;  -- the ICD version identifier
      end record;

   function Is_Up
     ( An_Instance : in Instance )
     return Boolean;
   -- Return status of specified interface

   function Create
     ( A_Name : in String;
       An_Icd_Filename : in String;
       A_Medium : in Io_Medium.Handle;
       Buffer_Is_Network_Order : in Boolean )
     return Instance;
   -- Read an ICD file and create an I/O interface

   procedure Map
     ( An_Instance : in out Instance;
       An_Io_Mnemonic : in String;
       A_Direction : in Io_Types.Direction;
       An_Address : in System.Address;
       A_Size : in Natural;
       Replay_Passthrough : in Boolean);
   -- Map an Ada address to an ICD mnemonic

   procedure Write_Outputs
     ( An_Instance : in out Instance );
   -- Gather outputs from host classes and transmit via medium

   procedure Read_Inputs
     ( An_Instance    : in out Instance;
       Replay_Discard : in Boolean := False
     );
   -- Receive inputs via medium and distribute to host classes
   -- or else restore inputs from snapshot

   procedure Save_Snapshot
     ( An_Instance : in Instance;
       To_Stream   : access Ada.Streams.Root_Stream_Type'Class );

   procedure Restore_Snapshot
     ( An_Instance : in out Instance;
       From_Stream : access Ada.Streams.Root_Stream_Type'Class );

   function Snapshot_Size
     ( An_Instance : in Instance )
     return Natural;

   procedure Print_Status
     ( An_Instance : in Instance );
   -- Print status of interface

   function Mnemonic_Buffer_Address
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return System.Address;

   function Mnemonic_Host_Address
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return System.Address;

   function Mnemonic_Is_Mapped
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return Boolean;

   function Mnemonic_Size
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return Natural;

   procedure Increment_Count
     ( A_Counter     : in out Natural_Direction_Array;
       A_Direction   : in     Io_Types.Direction );
   -- Increment counter 

   function Retrieve_Count
     ( A_Counter     : in Natural_Direction_Array;
       A_Direction   : in Io_Types.Direction )
     return Natural;

end Io_Interface;

