-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Io_Icd_File;
with System.Storage_Elements;
with Log;

with Ada.Exceptions;

package body Io_Interface is

   use type Io_Types.Direction;


   --| Some off-the-shelf component will be used to implement
   --| the data dictionary.  Possible sources are the Booch Components
   --| for Ada 95, or the University of Scranton Data Structures library.
   --| Time efficiency in traversing the data dictionary during real-time
   --| will be the most important criterion in selecting an
   --| implementation.

   Gather_Outputs : Boolean := True;
   pragma Export ( C, Gather_Outputs, "gather_outputs" );

   Scatter_Inputs : Boolean := True;
   pragma Export ( C, Scatter_Inputs, "scatter_inputs" );

   Send_Outputs : Boolean := True;
   pragma Export ( C, Send_Outputs, "send_outputs" );

   Recv_Inputs : Boolean := True;
   pragma Export ( C, Recv_Inputs, "recv_inputs" );

   Clear_Before_Read : Boolean := False;
   pragma Export ( C, Clear_Before_Read, "clear_before_read" );

   Clear_Before_Write : Boolean := True;
   pragma Export ( C, Clear_Before_Write, "clear_before_write" );

   Disable_After_Short_Reads : Boolean := False;
   pragma Export ( C, Disable_After_Short_Reads, "disable_shorts" );

   Short_Read : exception;

   function Is_Up
     ( An_Instance : Instance )
     return Boolean is

      -- Return status of specified interface

   begin
      -- dispatch to status function of the medium class
      return Io_Medium.Is_Up (An_Instance.The_Medium);
   end Is_Up;

   function Create
     ( A_Name : in String;
       An_Icd_Filename : in String;
       A_Medium : in Io_Medium.Handle;
       Buffer_Is_Network_Order : in Boolean )
     return Instance is

   -- Read an ICD file and create a map

      The_Instance : Instance;

      The_Icd_File : Io_Icd_File.Instance;

      The_Number_Of_Entries : Natural;
      The_Buffer_Sizes : Io_Icd_File.Natural_Direction_Array;
      The_Buffer_Offsets : Io_Icd_File.Natural_Direction_Array;

      -- Offsets in the ICD file are given in bytes from the beginning
      -- of a hypothetical buffer which includes a control section, an
      -- input section, and an output section.  (This is a legacy of
      -- the bidirectional reflective memory interface -- Bit3
      -- formerly used by FSI to interface the host and subsystem
      -- computers.) The offset and size of each section is given in
      -- the header of the ICD file and returned by procedure
      -- Read_Header.  The input or output section offset must be used
      -- to calculate the offset of each entry with respect to the
      -- input or output buffer.

      The_Memory_Size : Natural;
	  
	  The_Version : Natural;

      The_New_Element : Io_Map_Entry.Instance;

      At_End_Of_File : Boolean := False;
      The_Mnemonic : Ada.Strings.Unbounded.Unbounded_String;
      The_Direction : Io_Types.Direction;
      The_Offset : Natural;
      The_Size : Natural;

      Interface_Name : constant String := A_Name;
   begin

      --| copy name, filename, and medium parameters to the new interface

      The_Instance.The_Name :=
        Ada.Strings.Unbounded.To_Unbounded_String ( A_Name );

      The_Instance.The_Icd_Filename
        := Ada.Strings.Unbounded.To_Unbounded_String ( An_Icd_Filename );

      -- Network byte order of a data is a function of the interface,
      -- not the medium.  The medium only knows that it has a buffer
      -- of data to deliver, but doesn't know about its internal
      -- structure.  The interface knows about the device on the other
      -- end of the interface and the interface's maps define the
      -- structure of the buffers being transmitted.  Network order
      -- mainly affects the process of copying between host variables
      -- and the buffers in the gather and scatter process.

      The_Instance.Buffer_Is_Network_Order := Buffer_Is_Network_Order;

      The_Instance.The_Medium := A_Medium;

      --| open the ICD file

      Io_Icd_File.Open ( An_Icd_Filename, The_Icd_File );

      -- need some error handling here if the file isn't found

      --| read the header

      Io_Icd_File.Read_Header
        ( An_Instance           => The_Icd_File,
          The_Number_Of_Entries => The_Number_Of_Entries,
          The_Sizes             => The_Buffer_Sizes,
          The_Offsets           => The_Buffer_Offsets,
          The_Memory_Size       => The_Memory_Size,
	      The_Version           => The_Version );

      Log.Report ( "ICD " & An_Icd_Filename & " entries "
                   & Integer'Image ( The_Number_Of_Entries ) );

       Log.Report ( "ICD " & An_Icd_Filename & " MEMORY_SIZE " 
                   & Integer'Image ( The_Memory_Size ) );

       Log.Report ( "ICD " & An_Icd_Filename & " VERSION " 
                   & Integer'Image ( The_Version ) );
				   
	   The_Instance.The_Version := The_Version;

      --| Create buffers and maps
      for The_Direction in Io_Types.Direction loop

         Log.Report ( "ICD " & An_Icd_Filename & " "
                      & Io_Types.Direction'Image ( The_Direction )
                      & "_SECTION size "
                      & Integer'Image ( The_Buffer_Sizes (The_Direction) )
                      & ",  offset "
                      & Integer'Image ( The_Buffer_Offsets (The_Direction) ) );

         The_Instance.The_Buffer_Sizes ( The_Direction )
           := The_Buffer_Sizes (The_Direction);

         if The_Instance.The_Buffer_Sizes (The_Direction) > 0 then

            The_Instance.The_Buffers(The_Direction)
              := Io_Buffer.Create( The_Buffer_Sizes(The_Direction) );

            The_Instance.The_Maps(The_Direction)
              := Io_Map.Create;


         else

            Log.Report ( An_Icd_Filename
                         & " has no "
                         & Io_Types.Direction'Image ( The_Direction )
                         & "_SECTION.  Buffer and map will not be created.",
                         Log.Warning );

         end if;

         --| Initialize the Transfer_Count and Error_Count for this interface
         The_Instance.Transfer_Count( The_Direction ) := 0;
         The_Instance.Error_Count( The_Direction )    := 0;

      end loop;

      -- we need a bit bucket the same size as the input buffer,
      -- to have a place to dump input data from the interface
      -- when we are restoring a snapshot.
      if The_Instance.The_Buffer_Sizes (Io_Types.Input) > 0 then
         The_Instance.The_Bit_Bucket
           := Io_Buffer.Create( The_Buffer_Sizes(Io_Types.Input) );
      end if;

      Io_Medium.Set_Buffers
        ( The_Instance.The_Medium,
          The_Instance.The_Buffer_Sizes (Io_Types.Output) * 2,
          The_Instance.The_Buffer_Sizes (Io_Types.Input) * 4
          );

      --| read each line in the file until EOF
      --| while (not end of file) loop

      loop

         begin
            --| read the line from disk, parse the line and return
            --| a structure containing its information

            Io_Icd_File.Read_Entry (The_Icd_File, The_Mnemonic, The_Direction,
                                    The_Size, The_Offset, At_End_Of_File);
            if At_End_Of_File then
               exit;
            end if;

            --| If the offset is not within the range of the buffer
            --| for the specified direction, give a warning and
            --| ignore the entry.

            if The_Offset < The_Buffer_Offsets(The_Direction) or
              The_Offset >= ( The_Buffer_Offsets (The_Direction)
                              + The_Buffer_Sizes (The_Direction) ) then

            -- if The_Instance.The_Buffer_Sizes ( The_Direction ) = 0 then

                  Log.Report ( "Mnemonic "
                               & Ada.Strings.Unbounded.To_String(The_Mnemonic)
                               & " in "
                               & An_Icd_Filename
                               & " has an offset out of bounds for the "
                               & Io_Types.Direction'Image ( The_Direction )
                               & "_SECTION",
                               Log.Warning );

            --| see if the element is a duplicate; if so, log an error
            --|   and proceed to the next line in the file

            elsif not Io_Map.Is_Defined
                     ( Ada.Strings.Unbounded.To_String(The_Mnemonic),
                       The_Instance.The_Maps(The_Direction) ) then

               --| if the line is valid, copy the appropriate fields to the
               --|   data dictionary element

               --| The offset given in the ICD file is in bytes from
               --| the beginning of a hypothetical buffer that includes
               --| both inputs and outputs.  (This is the form in which
               --| it is supplied by the flight deck I/O group.)  The
               --| offset we need to pass to Address_of_Offset is the
               --| offset from the beginning of the buffer for the
               --| appropriate direction, so a subtraction has to be
               --| done.

               if The_Direction = Io_Types.Input then

                  The_New_Element :=
                    ( The_Buffer_Address =>
                        Io_Buffer.Address_Of_Offset
                      ( The_Instance.The_Buffers(The_Direction),
                        The_Offset - The_Buffer_Offsets(The_Direction)),
                      Secondary_Buffer   =>
                        Io_Buffer.Address_Of_Offset
                      ( The_Instance.The_Bit_Bucket,
                        The_Offset - The_Buffer_Offsets(The_Direction)),
                      The_Size => The_Size,
                      The_Direction => The_Direction,
                      -- the following values will be initialized when the
                      -- label is mapped to an Ada variable
                      The_Host_Address => System.Null_address,
                      Is_Mapped => False );
               else

                  The_New_Element :=
                    ( The_Buffer_Address =>
                        Io_Buffer.Address_Of_Offset
                      ( The_Instance.The_Buffers(The_Direction),
                        The_Offset - The_Buffer_Offsets(The_Direction)),
                      Secondary_Buffer   => System.Null_Address,
                      The_Size => The_Size,
                      The_Direction => The_Direction,
                      -- the following values will be initialized when the
                      -- label is mapped to an Ada variable
                      The_Host_Address => System.Null_address,
                      Is_Mapped => False );

               end if;

               --| store the element in the data dictionary

               Io_Map.Add_Entry
                 ( Ada.Strings.Unbounded.To_String(The_Mnemonic),
                   The_Instance.The_Maps(The_Direction),
                   The_New_Element );

            end if;

         exception
            when Io_Icd_File.Bad_Format | Io_Icd_File.Control_Entry =>
               -- just go to the next record
               null;
            when others =>
               -- propagate any other error
               raise;
         end;
      end loop;

      Io_Icd_File.Close ( The_Icd_File );

      The_Instance.Is_Valid := True;

      --| medium should already have been created
      return The_Instance;

   exception

      when Io_Icd_File.Bad_Header =>

         Io_Icd_File.Close ( The_Icd_File );

         Log.Report ( "Bad header in ICD file " & An_Icd_Filename
                      & " for interface " & A_Name,
                      Log.Warning );
         The_Instance.Is_Valid := False;
         return The_Instance;

      when Io_Icd_File.Cannot_Open =>

         Log.Report ( "Cannot open ICD file " & An_Icd_Filename
                      & " for interface " & A_Name,
                      Log.Warning );
         The_Instance.Is_Valid := False;
         return The_Instance;

      when Error: others =>

         Log.Report ( "Error in Io_Interface.Create for interface " & Interface_Name,
                      Log.Warning );
         Log.Report ( "Exception was " & Ada.Exceptions.Exception_Information (Error));
         The_Instance.Is_Valid := False;
         return The_Instance;

   end Create;

   procedure Map
     ( An_Instance : in out Instance;
       An_Io_Mnemonic : in String;
       A_Direction : in Io_Types.Direction;
       An_Address : in System.Address;
       A_Size : in Natural;
       Replay_Passthrough : in Boolean) is
   -- Map an Ada address to an ICD mnemonic

      The_Mnemonic_Item : Io_Map_Entry.Instance;
      The_Mnemonic_Item_Was_Found : Boolean;

   begin

      --| search the data dictionary of An_Instance for the mnemonic
      Io_Map.Find_Entry ( An_Io_Mnemonic,
                          An_Instance.The_Maps(A_Direction),
                          The_Mnemonic_Item,
                          The_Mnemonic_Item_Was_Found );

      --| log error message if mnemonic is not there or already mapped
      --| or if the size is wrong
      if not The_Mnemonic_Item_Was_Found then

         --| log error message that mnemonic not found
         Log.Report
           ( "Mnemonic " &
             An_Io_Mnemonic &
             " was not found in the " &
             Io_Types.Direction'Image ( A_Direction ) &
             " map of the " &
             Ada.Strings.Unbounded.To_String (An_Instance.The_Name) &
             " interface.",
             Log.Warning );

      elsif The_Mnemonic_Item.Is_Mapped then

         --| log error message that mnemonic has already been mapped to a
         --| host address
         Log.Report
           ( "Mnemonic " &
             An_Io_Mnemonic &
             " has already been mapped to an address.",
             Log.Warning );

      elsif The_Mnemonic_Item.The_Size /= A_Size then

         --| compare requested size to data dictionary size
         --| log error if size mismatch
         Log.Report
           ( "Mnemonic " &
             An_Io_Mnemonic &
             " has a different size than that of the mapped variable.  " &
             "Variable has not been mapped.",
             Log.Warning );

      else

         --| if no error, copy An_Address into the data dictionary entry
         The_Mnemonic_Item.The_Host_Address := An_Address;
         The_Mnemonic_Item.Is_Mapped := True;

         -- Only replay passthrough variables actually read from the
         -- secondary buffer (where the real IO gets put).
         if not Replay_Passthrough then
            The_Mnemonic_Item.Secondary_Buffer := The_Mnemonic_Item.The_Buffer_Address;
         end if;

         --| update the data dictionary entry
         Io_Map.Set_Entry ( An_Io_Mnemonic,
                            An_Instance.The_Maps(A_Direction),
                            The_Mnemonic_Item,
                            The_Mnemonic_Item_Was_Found );
      end if;

   end Map;

   procedure Write_Outputs
     ( An_Instance : in out Instance ) is
      -- Gather outputs from host classes and transmit via medium

      Bytes_Written : Natural;

   begin

      -- skip all this if there isn't a buffer for this direction
      if An_Instance.The_Buffer_Sizes ( Io_Types.Output ) > 0 then

         if Clear_Before_Write then
            -- skip the following in debug mode, because we want to be
            -- able to set values in the buffer directly and not have
            -- them cleared out or overriden by software

            --| clear the output buffer

            Io_Buffer.Clear ( An_Instance.The_Buffers ( Io_Types.Output ) );

         end if;

         if Gather_Outputs then

            --| traverse the data dictionary and copy from the host
            --| variable to the output buffer

            Io_Map.Gather ( An_Instance.The_Maps ( Io_Types.Output ),
                            An_Instance.Buffer_Is_Network_Order );

         end if;

         if Send_Outputs then
            --| write the latest data buffer to the IO interface

            Io_Medium.Write
              ( An_Instance.The_Medium,
                Io_Buffer.Address ( An_Instance.The_Buffers ( Io_Types.Output ) ),
                Io_Buffer.Size ( An_Instance.The_Buffers ( Io_Types.Output ) ),
                Bytes_Written );

            if Bytes_Written <
              An_Instance.The_Buffer_Sizes ( Io_Types.Output ) then

              Increment_Count
                ( A_Counter   => An_Instance.Error_Count,
                  A_Direction => IO_Types.Output);

               Log.Report
                 ( "Short write on " &
                   Ada.Strings.Unbounded.To_String (An_Instance.The_Name) &
                   " interface. Expected " &
                   Integer'Image
                   ( An_Instance.The_Buffer_Sizes ( Io_Types.Output ) ) &
                   ", sent " &
                   Integer'Image ( Bytes_Written ),
                   Log.Warning );
            else

              Increment_Count
                ( A_Counter   => An_Instance.Transfer_Count,
                  A_Direction => IO_Types.Output);

            end if;

         end if;

      end if;

   end Write_Outputs;

   procedure Read_Inputs
     ( An_Instance    : in out Instance;
       Replay_Discard : in Boolean := False
     ) is
   -- Receive inputs via medium and distribute to host classes

      Bytes_Read : Natural;

      Into_Buffer : Io_Buffer.Handle := An_Instance.The_Buffers(Io_Types.Input);
   begin

      -- skip all this if there isn't a buffer for this direction
      if An_Instance.The_Buffer_Sizes ( Io_Types.Input ) > 0 then

         if Clear_Before_Read then

            -- skip the following to allow us to test the scatter
            -- mechanism

            --| clear the input buffer

            Io_Buffer.Clear ( An_Instance.The_Buffers ( Io_Types.Input ) );

         end if;

         if Recv_Inputs then

            -- If we're restoring from snapshot then read the input
            -- packet from the interface into the bit bucket, thus
            -- discarding it, and then restore to the real input
            -- buffer from the snapshot.
            if Replay_Discard then
               Into_Buffer := An_Instance.The_Bit_Bucket;
            end if;

            Io_Medium.Read
              ( An_Instance.The_Medium,
                Io_Buffer.Address ( Into_Buffer ),
                Io_Buffer.Size ( Into_Buffer ),
                Bytes_Read );

            if Bytes_Read
              < An_Instance.The_Buffer_Sizes ( Io_Types.Input ) then
              
               Increment_Count
                ( A_Counter   => An_Instance.Error_Count,
                  A_Direction => IO_Types.Output);

                Log.Report
                 ( "Short read on " &
                   Ada.Strings.Unbounded.To_String (An_Instance.The_Name) &
                   " interface. Expected " &
                   Integer'Image
                   ( An_Instance.The_Buffer_Sizes ( Io_Types.Input ) ) &
                   ", received " &
                   Integer'Image ( Bytes_Read ), 
                   Log.Warning );
               if Disable_After_Short_Reads then
                  raise Short_Read;
               end if;
            else
              Increment_Count
                ( A_Counter   => An_Instance.Transfer_Count,
                  A_Direction => IO_Types.Output);
            end if;
         end if;

         if Scatter_Inputs then

            -- traverse the data dictionary and copy from the buffer to the host
            -- variable

            Io_Map.Scatter ( An_Instance.The_Maps ( Io_Types.Input ),
                             An_Instance.Buffer_Is_Network_Order,
                             Use_Secondary_Buffer => Replay_Discard);

         end if;

      end if;

   end Read_Inputs;

   procedure Print_Status
     ( An_Instance : in Instance ) is
   -- Print status of interface
   begin
      Log.Report ( "Io_Interface "
                   & Ada.Strings.Unbounded.To_String ( An_Instance.The_Name )
                   & ", ICD="
                   & Ada.Strings.Unbounded.To_String
                   ( An_Instance.The_Icd_Filename ) );
      Io_Medium.Print_Status ( An_Instance.The_Medium );
      for Dir in Io_Types.Direction loop
         Log.Report
           ( "The_Buffers("
             & Io_Types.Direction'Image (Dir)
             & ") => Address "
             & System.Storage_Elements.Integer_Address'Image
             ( System.Storage_Elements.To_Integer ( Io_Buffer.Address
                                                    ( An_Instance.The_Buffers (Dir) ) ) )
             & ", Size "
             & Natural'Image ( Io_Buffer.Size
                               ( An_Instance.The_Buffers (Dir) ) ) );
      end loop;

   end Print_Status;


   function Mnemonic_Buffer_Address
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return System.Address is

      The_Mnemonic_Item : Io_Map_Entry.Instance;
      The_Mnemonic_Item_Was_Found : Boolean;

   begin

      --| search the data dictionary of An_Instance for the mnemonic
      Io_Map.Find_Entry ( An_Io_Mnemonic,
                          An_Instance.The_Maps(A_Direction),
                          The_Mnemonic_Item,
                          The_Mnemonic_Item_Was_Found );

      if not The_Mnemonic_Item_Was_Found then

         --| log error message that mnemonic not found
         Log.Report
           ( "Mnemonic " &
             An_Io_Mnemonic &
             " was not found in the " &
             Io_Types.Direction'Image ( A_Direction ) &
             " map of the " &
             Ada.Strings.Unbounded.To_String (An_Instance.The_Name) &
             " interface.",
             Log.Warning );
         return System.Null_Address;

      else

         return The_Mnemonic_Item.The_Buffer_Address;

      end if;

   end Mnemonic_Buffer_Address;

   function Mnemonic_Host_Address
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return System.Address is

      The_Mnemonic_Item : Io_Map_Entry.Instance;
      The_Mnemonic_Item_Was_Found : Boolean;

   begin

      --| search the data dictionary of An_Instance for the mnemonic
      Io_Map.Find_Entry ( An_Io_Mnemonic,
                          An_Instance.The_Maps(A_Direction),
                          The_Mnemonic_Item,
                          The_Mnemonic_Item_Was_Found );

      if not The_Mnemonic_Item_Was_Found then

         --| log error message that mnemonic not found
         Log.Report
           ( "Mnemonic " &
             An_Io_Mnemonic &
             " was not found in the " &
             Io_Types.Direction'Image ( A_Direction ) &
             " map of the " &
             Ada.Strings.Unbounded.To_String (An_Instance.The_Name) &
             " interface.",
             Log.Warning );
         return System.Null_Address;

      else

         return The_Mnemonic_Item.The_Host_Address;

      end if;

   end Mnemonic_Host_Address;

   function Mnemonic_Is_Mapped
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return Boolean is

      The_Mnemonic_Item : Io_Map_Entry.Instance;
      The_Mnemonic_Item_Was_Found : Boolean;

   begin

      --| search the data dictionary of An_Instance for the mnemonic
      Io_Map.Find_Entry ( An_Io_Mnemonic,
                          An_Instance.The_Maps(A_Direction),
                          The_Mnemonic_Item,
                          The_Mnemonic_Item_Was_Found );

      if not The_Mnemonic_Item_Was_Found then

         --| log error message that mnemonic not found
         Log.Report
           ( "Mnemonic " &
             An_Io_Mnemonic &
             " was not found in the " &
             Io_Types.Direction'Image ( A_Direction ) &
             " map of the " &
             Ada.Strings.Unbounded.To_String (An_Instance.The_Name) &
             " interface.",
             Log.Warning );
         return False;

      else

         return The_Mnemonic_Item.Is_Mapped;

      end if;

   end Mnemonic_Is_Mapped;


   function Mnemonic_Size
     ( An_Instance : in Instance;
       A_Direction : in Io_Types.Direction;
       An_Io_Mnemonic : in String )
     return Natural is

      The_Mnemonic_Item : Io_Map_Entry.Instance;
      The_Mnemonic_Item_Was_Found : Boolean;

   begin

      --| search the data dictionary of An_Instance for the mnemonic
      Io_Map.Find_Entry ( An_Io_Mnemonic,
                          An_Instance.The_Maps(A_Direction),
                          The_Mnemonic_Item,
                          The_Mnemonic_Item_Was_Found );

      if not The_Mnemonic_Item_Was_Found then

         --| log error message that mnemonic not found
         Log.Report
           ( "Mnemonic " &
             An_Io_Mnemonic &
             " was not found in the " &
             Io_Types.Direction'Image ( A_Direction ) &
             " map of the " &
             Ada.Strings.Unbounded.To_String (An_Instance.The_Name) &
             " interface.",
             Log.Warning );
         return 0;

      else

         return The_Mnemonic_Item.The_Size;

      end if;

   end Mnemonic_Size;


   procedure Save_Snapshot
     ( An_Instance : in Instance;
       To_Stream   : access Ada.Streams.Root_Stream_Type'Class ) is
   begin
      -- don't save anything if the buffer size is 0
      if An_Instance.The_Buffer_Sizes ( Io_Types.Input ) > 0 then
         Io_Buffer.Save ( An_Instance.The_Buffers(Io_Types.Input),
                          To_Stream );
      end if;
   end Save_Snapshot;

   procedure Restore_Snapshot
     ( An_Instance : in out Instance;
       From_Stream : access Ada.Streams.Root_Stream_Type'Class ) is
   begin
      Io_Buffer.Restore ( An_Instance.The_Buffers(Io_Types.Input),
                          From_Stream );
   end Restore_Snapshot;

   function Snapshot_Size
     ( An_Instance : in Instance )
     return Natural is
   begin
      if An_Instance.The_Buffer_Sizes ( Io_Types.Input ) > 0 then
         -- a snapshot consists of the input buffer, so return its size
         return Io_Buffer.Snapshot_Size
           ( An_Instance.The_Buffers ( Io_Types.Input ) );
      else
         -- we don't save a snapshot for zero-length buffers
         return 0;
      end if;
   end Snapshot_Size;

   procedure Increment_Count
     ( A_Counter     : in out Natural_Direction_Array;
       A_Direction   : in     Io_Types.Direction ) is
   begin
      -- Increment counter 
      A_Counter ( A_Direction ) := A_Counter ( A_Direction ) + 1;
   end Increment_Count;

   function Retrieve_Count
     ( A_Counter     : in Natural_Direction_Array;
       A_Direction   : in Io_Types.Direction )
     return Natural is
   begin
     return ( A_Counter ( A_Direction ));
   end Retrieve_Count;

end Io_Interface;
