-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Log;
with Interfaces.C, Interfaces.C.Strings;
package body Io_Icd_File is

   use type Interfaces.C.Int, Interfaces.C.Size_T;
   use Io_Types;  -- make enumeration values visible

   -- errno
   Errno : Interfaces.C.Int;
   pragma Import (C, Errno, "errno");

   Input_Line_Size : constant := 256;

   type Stdio_File is array (1..8) of Integer;
   -- this is just something big enough to hold the FILE structure,
   -- but all we do is use a pointer to it, so we don't care about its
   -- internal structure

   -- All calls to New_String and any other dynamic allocation is done
   -- here at the package body level and not in the subprograms, so as
   -- to allocate memory only once, rather than with each call to
   -- Read_Entry

   Entries_Keyword : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "ENTRIES" );

   Entries_Scan_Format : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "ENTRIES %d" );

   Variables_Keyword : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "VARIABLES" );

   Memory_Size_Keyword : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "MEMORY_SIZE" );

   Memory_Size_Scan_Format : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "MEMORY_SIZE %d" );

   Version_Keyword : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "VERSION" );

   Version_Scan_Format : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "VERSION %d" );

   type Direction_Chars_Ptr_Array is
     array ( Io_Types.Direction ) of Interfaces.C.Strings.Chars_Ptr;

   type Direction_Int_Array is
     array ( Io_Types.Direction ) of aliased Interfaces.C.Int;

   Section_Keywords : Direction_Chars_Ptr_Array
     := ( Input  => Interfaces.C.Strings.New_String ( "INPUT_SECTION" ),
          Output => Interfaces.C.Strings.New_String ( "OUTPUT_SECTION" ) );

   Section_Scan_Formats : Direction_Chars_Ptr_Array
     := ( Input  => Interfaces.C.Strings.New_String ( "INPUT_SECTION %d %d" ),
          Output => Interfaces.C.Strings.New_String ( "OUTPUT_SECTION %d %d" ) );

   Section_Keyword_Lengths : Direction_Int_Array
     := ( Input => 13,
          Output => 14 );

   Io_Label_Scan_Alt_Format : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String
     ( "%d %s ""%*[^ ] ""%[^""]"" %s %*d %*d %*s %d %d" );
	 -- test: read 'Description' up to the first white space
   Io_Label_Scan_Format : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String
     ( "%d %s ""%*[^""]"" ""%[^""]"" %s %*d %*d %*s %d %d" );

   Input_Keyword : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String("INPUT");

   Output_Keyword : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String("OUTPUT");

   Control_Keyword : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String("CONTROL");

   Read_Only_Mode_String : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.New_String ( "r" );

   Input_Line : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.To_Chars_Ptr
     (new Interfaces.C.Char_Array ( 0 .. Input_Line_Size - 1 ));

   Mnemonic : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.To_Chars_Ptr
     (new Interfaces.C.Char_Array ( 0 .. Input_Line_Size - 1 ));

   Label_Type : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.To_Chars_Ptr
     (new Interfaces.C.Char_Array ( 0 .. Input_Line_Size - 1 ));

   Direction : Interfaces.C.Strings.Chars_Ptr
     := Interfaces.C.Strings.To_Chars_Ptr
     (new Interfaces.C.Char_Array ( 0 .. Input_Line_Size - 1 ));


   function Fopen
     ( Pathname : in Interfaces.C.Strings.Chars_Ptr;
       Mode : in Interfaces.C.Strings.Chars_Ptr ) return Stdio_File_Ptr;
   pragma Import (C, Fopen, "fopen");

   function Fclose
     ( Stream : in Stdio_File_Ptr )
     return Interfaces.C.int;
   pragma Import (C, Fclose, "fclose");

   function Fgets
     ( S : in Interfaces.C.Strings.Chars_Ptr;
       N : in Interfaces.C.Int;
       Stream : in Stdio_File_Ptr )
     return Interfaces.C.Int;
   pragma Import (C, Fgets, "fgets");

   function Strncmp
     ( S1 : in Interfaces.C.Strings.Chars_Ptr;
       S2 : in Interfaces.C.Strings.Chars_Ptr;
       N  : in Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Strncmp, "strncmp");

   function Sscanf_1_Int
     ( S : in Interfaces.C.Strings.Chars_Ptr;
       Format : in Interfaces.C.Strings.Chars_Ptr;
       I : access Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Sscanf_1_Int, "sscanf");

   function Sscanf_2_Int
     ( S : in Interfaces.C.Strings.Chars_Ptr;
       Format : in Interfaces.C.Strings.Chars_Ptr;
       I1 : access Interfaces.C.Int;
       I2 : access Interfaces.C.Int )
     return Interfaces.C.Int;
   pragma Import (C, Sscanf_2_Int, "sscanf");

   function Sscanf_Io_Label
     ( S : in Interfaces.C.Strings.Chars_Ptr;
       Format : in Interfaces.C.Strings.Chars_Ptr;
       Index : access Interfaces.C.Int;
       Label : in Interfaces.C.Strings.Chars_Ptr;
       Label_Type : in Interfaces.C.Strings.Chars_Ptr;
       Direction : in Interfaces.C.Strings.Chars_Ptr;
       Size : access Interfaces.C.Int;
       Offset : access Interfaces.C.int ) return Interfaces.C.Int;
   pragma Import (C, Sscanf_Io_Label, "sscanf");

   procedure Open
     ( A_Filename  : in String;
       An_Instance : out Instance ) is
      Saved_Errno : Interfaces.C.Int;
   begin
      An_Instance.File_Ptr := Fopen ( Pathname =>
                                        Interfaces.C.Strings.New_String ( A_Filename ),
                                      Mode =>
                                        Read_Only_Mode_String );
      An_Instance.Filename := Ada.Strings.Unbounded.To_Unbounded_String ( A_Filename );
      Saved_Errno := Errno;
      if ( An_Instance.File_Ptr = null ) then
         Log.Report ( "Error opening ICD file " & A_Filename & "; errno = "
                      & Interfaces.C.Int'Image (Saved_Errno),
                      Log.Error );
         raise Cannot_Open;
      end if;
   end Open;

   procedure Close
     ( An_Instance : in Instance ) is
      Result : Interfaces.C.Int;
   begin
      Result := Fclose ( An_Instance.File_Ptr );
   end Close;

   procedure Read_Header
     ( An_Instance : in Instance;
       The_Number_Of_Entries : out Natural;
       The_Sizes : out Natural_Direction_Array;
       The_Offsets : out Natural_Direction_Array;
       The_Memory_Size : out Natural;	   
	   The_Version : out Natural ) is

      Entries : aliased Interfaces.C.Int := 0;
      Memory_Size : aliased Interfaces.C.Int := 0;
	  Version : aliased Interfaces.C.Int := 0;
      Sizes : Direction_Int_Array := ( 0, 0 );
      Offsets : Direction_Int_Array := ( 0, 0 );
      Result : Interfaces.C.Int;

      Entries_Found : Boolean := False;
      Memory_Size_Found : Boolean := False;
	  Version_Found : Boolean := False;
      Input_Section_Found : Boolean := False;
      Output_Section_Found : Boolean := False;

   begin

      -- If a required line is not encountered, the default value
      -- returned is zero.  NB: Green Hills compiler does NOT
      -- initialize objects to zero!

      while (Fgets (Input_Line, Input_Line_Size, An_Instance.File_Ptr) /= 0) loop

         if (Strncmp (Input_Line, Entries_Keyword, 7) = 0) then

            Result := Sscanf_1_Int (Input_Line, Entries_Scan_Format, Entries'Access);
            Entries_Found := True;

         elsif (Strncmp (Input_Line, Memory_Size_Keyword, 11) = 0) then

            Result := Sscanf_1_Int (Input_Line, Memory_Size_Scan_Format, Memory_Size'Access);
            Memory_Size_Found := True;

         elsif (Strncmp (Input_Line, Version_Keyword, 7) = 0) then

            Result := Sscanf_1_Int (Input_Line, Version_Scan_Format, Version'Access);
            Version_Found := True;

         elsif (Strncmp (Input_Line, Section_Keywords(Input),
                         Section_Keyword_Lengths(Input)) = 0) then

            Result := Sscanf_2_Int (Input_Line, Section_Scan_Formats(Input),
                                    Offsets(Input)'Access, Sizes(Input)'Access);
            Input_Section_Found := True;

         elsif (Strncmp (Input_Line, Section_Keywords(Output),
                         Section_Keyword_Lengths(Output)) = 0) then

            Result := Sscanf_2_Int (Input_Line, Section_Scan_Formats(Output),
                                    Offsets(Output)'Access, Sizes(Output)'Access);
            Output_Section_Found := True;

         elsif (Strncmp (Input_Line, Variables_Keyword, 9) = 0) then

            exit;

         end if;

      end loop;

      if Entries_Found and Memory_Size_Found and Version_Found
        and Input_Section_Found and Output_Section_Found then

         The_Number_Of_Entries := Natural(Entries);
         The_Memory_Size := Natural(Memory_Size);
		 The_Version := Natural(Version);
         for D in Io_Types.Direction loop
            The_Offsets (D) := Natural ( Offsets (D) );
            The_Sizes (D) := Natural ( Sizes (D) );
         end loop;

      else

         if not Entries_Found then
            Log.Report ( "ENTRIES header line not found in "
                         & Ada.Strings.Unbounded.To_String ( An_Instance.Filename ),
                         Log.Error );
         end if;

         if not Memory_Size_Found then
            Log.Report ( "MEMORY_SIZE header line not found in "
                         & Ada.Strings.Unbounded.To_String ( An_Instance.Filename ),
                         Log.Error );
         end if;

         if not Version_Found then
            Log.Report ( "VERSION header line not found in "
                         & Ada.Strings.Unbounded.To_String ( An_Instance.Filename ),
                         Log.Error );
         end if;
		 
         if not Input_Section_Found then
            Log.Report ( "INPUT_SECTION header line not found in "
                         & Ada.Strings.Unbounded.To_String ( An_Instance.Filename ),
                         Log.Error );
         end if;

         if not Output_Section_Found then
            Log.Report ( "OUTPUT_SECTION header line not found in "
                         & Ada.Strings.Unbounded.To_String ( An_Instance.Filename ),
                         Log.Error );
         end if;

         raise Bad_Header;

      end if;

   end Read_Header;

   procedure Read_Entry
     ( An_Instance : in Instance;
       The_Mnemonic : out Ada.Strings.Unbounded.Unbounded_String;
       The_Direction : out Io_Types.Direction;
       The_Size : out Natural;
       The_Offset : out Natural;
       At_End_Of_File : out Boolean ) is

      Index : aliased Interfaces.C.Int;
      Size : aliased Interfaces.C.Int;
      Offset : aliased Interfaces.C.Int;
      Result : Interfaces.C.Int;

   begin

      if Fgets (Input_Line, Input_Line_Size, An_Instance.File_Ptr) /= 0 then

         Result := Sscanf_Io_Label ( Input_Line,
                                     Io_Label_Scan_Format,
                                     Index'Access,
                                     Mnemonic,
                                     Label_Type,
                                     Direction,
                                     Size'Access,
                                     Offset'Access );

		 if (Result /= 6) then
            Result := Sscanf_Io_Label ( Input_Line,
                                        Io_Label_Scan_Alt_Format,
                                        Index'Access,
                                        Mnemonic,
                                        Label_Type,
                                        Direction,
                                        Size'Access,
                                        Offset'Access );
	     end if;
         The_Mnemonic :=
           Ada.Strings.Unbounded.To_Unbounded_String
           ( Interfaces.C.Strings.Value(Mnemonic) );

         if (Strncmp (Direction, Input_Keyword, 5) = 0) then
            The_Direction := Io_Types.Input;
         elsif (Strncmp (Direction, Output_Keyword, 6) = 0) then
            The_Direction := Io_Types.Output;
         elsif (Strncmp (Direction, Control_Keyword, 7) = 0) then
            raise Control_Entry;
         elsif (Strncmp (Direction, Control_Keyword, 7) = 0) then
            raise Control_Entry;
         else
            -- need to raise an exception if invalid
            Log.Report ("Bad format => " & Interfaces.C.Strings.Value(Input_Line),
                        Log.Warning);
            raise Bad_Format;
         end if;

         The_Size := Natural(Size);
         The_Offset := Natural(Offset);

         At_End_Of_File := False;

      else

         -- need to raise an exception if invalid format
         At_End_Of_File := True;

      end if;

   end Read_Entry;

end Io_Icd_File;
