-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Interfaces;

with Ada.Streams;
with System.Storage_Elements;

use type Ada.Streams.Stream_Element_Offset;
use type System.Storage_Elements.Storage_Offset;

package body Io_Buffer is

   -- Class of amorphous I/O buffer, used to transmit and receive data
   -- between host simulation software and simulation subsystems

   Bytes_Per_Element   : constant := (Ada.Streams.Stream_Element'Size + 7) / 8;

   type Instance is new Ada.Streams.Stream_Element_Array;

   function Create
     ( A_Size : in Positive )
      return Handle is
      -- Create a new I/O buffer of the specified size in bytes
   begin
      return new Instance'(1 .. (Ada.Streams.Stream_Element_Offset(A_Size) + Bytes_Per_Element - 1) / Bytes_Per_Element => 0);
   end Create;

   function Size
     ( A_Handle : in Handle )
      return Natural is
   -- Return the size in bytes
   begin
      return A_Handle.all'Length * Bytes_Per_Element;
   end Size;

   function Address
     ( A_Handle : in Handle )
     return System.Address is
      -- return the address of the start of the buffer
   begin
      return A_Handle.all'Address;
   end Address;

   function Address_Of_Offset
     ( A_Handle : in Handle;
       An_Offset : in Natural )
     return System.Address is
   -- Return the address of an offset in bytes from the beginning of the buffer
   begin
      return A_Handle.all'Address + System.Storage_Elements.Storage_Offset(An_Offset);
   end Address_Of_Offset;

   procedure Clear
     ( A_Handle : in Handle ) is
   -- Zero the buffer
   begin
      A_Handle.all := ( others => 0 );
   end Clear;

   -- Write and read buffers to and from a stream (for use with
   -- record/playback). We must use 'Input and 'Output instead of
   -- 'Read and 'Write because Instance is an indefinite subtype
   -- (unconstrained array).

   procedure Save
     (A_Handle : in Handle;
      To_Stream : access Ada.Streams.Root_Stream_Type'Class) is
   begin
      Ada.Streams.Stream_Element_Offset'Write ( To_Stream, A_Handle.all'Length );
      Ada.Streams.Write
        (Stream => To_Stream.all,
         Item   => Ada.Streams.Stream_Element_Array(A_Handle.all)
         );
   end Save;

   procedure Restore
     (A_Handle : in Handle;
      From_Stream : access Ada.Streams.Root_Stream_Type'Class) is
      Size : Ada.Streams.Stream_Element_Offset;
   begin
      Ada.Streams.Stream_Element_Offset'Read ( From_Stream, Size );
      if Size /= A_Handle.all'Length then
         raise Snapshot_Size_Mismatch;
      end if;
      Ada.Streams.Read
        (Stream => From_Stream.all,
         Item   => Ada.Streams.Stream_Element_Array(A_Handle.all),
         Last   => Size
         );
      if Size /= A_Handle.all'Length then
         raise Snapshot_Size_Mismatch;
      end if;
   end Restore;

   Natural_Bytes : Natural := Natural'Size / System.Storage_Unit;

   function Snapshot_Size
     (A_Handle : in Handle)
     return Natural is
      -- actual number of bytes output when a snapshot of the buffer is written to a stream
   begin
      return (A_Handle.all'Length * Bytes_Per_Element) + Natural_Bytes;
   end Snapshot_Size;

end Io_Buffer;
