-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with Interfaces.C;
pragma Elaborate_All ( Ada.Unchecked_Conversion );
package body Arinc_429_Utils is

   use type Arinc_429_Types.Octal_Label;
   use type Interfaces.C.Unsigned;

   type Bit_Array is array (Natural range 0 .. 31) of Boolean;
   -- bit operations type

   pragma Pack (Bit_Array);

   function To_Bit_Array is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Message_Type,
                                Target => Bit_Array );

   function To_Message_Type is new
     Ada.Unchecked_Conversion ( Source => Bit_Array,
                                Target => Arinc_429_Types.Message_Type );

   function To_Bit_Array is new
     Ada.Unchecked_Conversion ( Source => Interfaces.C.Unsigned,
                                Target => Bit_Array );

   function To_Unsigned is new
     Ada.Unchecked_Conversion ( Source => Bit_Array,
                                Target => Interfaces.C.Unsigned );

   Two_Bit_Reverse : array (Arinc_429_Types.Octal_Label range 0 .. 3)
     of Arinc_429_Types.Octal_Label
     := ( 2#00# => 2#00#,
          2#01# => 2#10#,
          2#10# => 2#01#,
          2#11# => 2#11# );

   Three_Bit_Reverse : array (Arinc_429_Types.Octal_Label range 0 .. 7)
     of Arinc_429_Types.Octal_Label
     := ( 2#000# => 2#000#,
          2#001# => 2#100#,
          2#010# => 2#010#,
          2#011# => 2#110#,
          2#100# => 2#001#,
          2#101# => 2#101#,
          2#110# => 2#011#,
          2#111# => 2#111# );

   Parity_Bit_Mask : Bit_Array := To_Bit_Array ( 16#7fffffff# );
   -- 16#7FFFFFFF# to mask out high bit

   -- Take octal label and reverse it

   function To_Reversed_Label ( A_Label : in Arinc_429_Types.Octal_Label )
     return Arinc_429_Types.Reversed_Octal_Label is

      A_Label_High : Arinc_429_Types.Octal_Label;
      A_Label_Mid  : Arinc_429_Types.Octal_Label;
      A_Label_Low  : Arinc_429_Types.Octal_Label;

   begin

      A_Label_High := (A_Label / 8#100#);
      A_Label_Mid  := (A_Label / 8#010#) mod 8#010#;
      A_Label_Low  := (A_Label mod 8#010#);

      -- Reversed high bits 6, 7 become low bits 1, 0
      -- Reversed bits 3, 4, 5 become bits 4, 3, 2
      -- Reversed bits 0, 1, 2 become high bits 7, 6, 5

      -- HHMMMLLL   =>  LLLMMMHH

      return
        Arinc_429_Types.Reversed_Octal_Label
        (Two_Bit_Reverse(A_Label_High)
         + Three_Bit_Reverse(A_Label_Mid) * 2**2
         + Three_Bit_Reverse(A_Label_Low) * 2**5);

   end To_Reversed_Label;


   function Odd_Parity_Bit ( A_Message : in Arinc_429_Types.Message_Type )
     return Boolean is

      Message_Bit_Array : Bit_Array;
      Count : Natural;

   begin

      -- Convert message to bit array
      Message_Bit_Array := To_Bit_Array ( A_Message );

      -- mask out MSB (odd parity bit)
      Message_Bit_Array := Message_Bit_Array and Parity_Bit_Mask;

      -- Count the number of one bits in the word
      -- This is a lot easier in C, where you can easily treat an integer as a bit array
      -- for (count=0; m; count++) { m &= (m - 1) };

      Count := 0;
      while To_Unsigned (Message_Bit_Array) /= 0 loop
         Message_Bit_Array :=
           Message_Bit_Array and
           To_Bit_Array ( To_Unsigned (Message_Bit_Array) - 1 );
         Count := Count + 1;
      end loop;

      -- Return true if word had even number of bits, meaning we need to set
      -- the odd parity bit.  Return false if word is already odd parity, so
      -- we can leave the parity bit clear.
      return ( Count mod 2 ) = 0;

   end Odd_Parity_Bit;



end Arinc_429_Utils;
