-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with Arinc_429_Utils;
with Interfaces.C;
with Ada.Exceptions;
with Log;
pragma Elaborate_All ( Ada.Exceptions, Ada.Unchecked_Conversion, Log );
package body Arinc_429_Discrete is

   use type Interfaces.C.Unsigned;
   use type Interfaces.C.Int;
   use type Arinc_429_Types.Value_Field_Type;

   subtype Discretes_Bit_Array is
     Arinc_429_Types.Bit_Array ( 1 .. Discretes_Type'Size );

   -- This declaration is used instead of Discretes_Type'Size to prevent bogous
   -- compiler warnings in dead code. Its value is not changed, but it cannot
   -- be "constant", or the compiler will report warnings.
   Discretes_Type_Size : Integer := Discretes_Type'Size;

--    type No_Sdi_Value_Field_Type is
--      mod 2 ** ( Arinc_429_Types.Value_Field_Length
--                 + Arinc_429_Types.Sdi_Field_Length );

   The_Reversed_Label : Arinc_429_Types.Reversed_Octal_Label
     := Arinc_429_Utils.To_Reversed_Label (Label);

   -- Unchecked conversions between enumeration or private types and integer types

   function To_No_Sdi_Message_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Message_Type,
                                Target => Arinc_429_Types.No_Sdi_Message_Type );

   function To_Message_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.No_Sdi_Message_Type,
                                Target => Arinc_429_Types.Message_Type );

   function To_Ssm_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Ssm_Type,
                                Target => Arinc_429_Types.Ssm_Field_Type );

   function To_Ssm_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Ssm_Field_Type,
                                Target => Ssm_Type );

   function To_Sdi_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Sdi_Type,
                                Target => Arinc_429_Types.Sdi_Field_Type );

   function To_Sdi_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Sdi_Field_Type,
                                Target => Sdi_Type );

   -- convert between discrete type and value field type

--     function To_Discretes_Mod_Type is new
--       Ada.Unchecked_Conversion ( Source => Discretes_Type,
--                              Target => Discretes_Mod_Type );

--     function To_Discretes_Type is new
--       Ada.Unchecked_Conversion ( Source => Discretes_Mod_Type,
--                              Target => Discretes_Type );

--    function To_No_Sdi_Value_Field_Type is new
--      Ada.Unchecked_Conversion ( Source => Discretes_Type,
--                             Target => No_Sdi_Value_Field_Type );

--    function To_Discretes_Type is new
--      Ada.Unchecked_Conversion ( Source => No_Sdi_Value_Field_Type,
--                             Target => Discretes_Type );

   -- Convert to and from bit arrays

   function To_Bit_Array is new Ada.Unchecked_Conversion ( Discretes_Type,
                                                           Discretes_Bit_Array );

   function To_Discretes_Type is new Ada.Unchecked_Conversion ( Discretes_Bit_Array,
                                                                Discretes_Type );




   function Pack ( A_Discretes_Value : in Discretes_Type;
                   A_Sdi   : in Sdi_Type;
                   A_Ssm   : in Ssm_Type )
     return Arinc_429_Types.Message_Type is

      Result : Arinc_429_Types.Message_Type;
      No_Sdi_Result : Arinc_429_Types.No_Sdi_Message_Type;
      A_Discretes_Bit_Array : Discretes_Bit_Array;

   begin

      A_Discretes_Bit_Array := To_Bit_Array ( A_Discretes_Value );

      -- add in any discretes to low end of data field:
      -- 1. the appropriate bits are already cleared
      -- 2. convert the discrete record type to an integer of
      --    the appropriate size
      -- 3. "or" or add in this integer to the data field

      if No_Sdi then

         -- use special "no SDI" message type to insert value

         -- add in the integer equivalent of the discrete type

         No_Sdi_Result.The_Value := ( others => False );
         No_Sdi_Result.The_Value ( 1 .. Discretes_Type'Size )
           := A_Discretes_Bit_Array ( 1 .. Discretes_Type'Size );

         -- cast to normal message type for further processing
         Result := To_Message_Type ( No_Sdi_Result );

      else

         -- We're using the SDI field, so we can insert the value
         -- directly in the standard message type

         -- We have first to convert the discretes private type
         -- to the equivalently sized modulus type, then we can
         -- cast to the value field type.

         Result.The_Value := ( others => False );
         Result.The_Value ( 1 .. Discretes_Type_Size )
           := A_Discretes_Bit_Array ( 1 .. Discretes_Type'Size );

         Result.The_Sdi := To_Sdi_Field_Type ( A_Sdi );

      end if;

      -- copy in the reversed octal 429 label and sign-status matrix

      Result.The_Label := The_Reversed_Label;
      Result.The_Ssm := To_Ssm_Field_Type ( A_Ssm );


      -- calculate parity bit for odd parity
      Result.The_Parity_Bit := False;
      Result.The_Parity_Bit := Arinc_429_Utils.Odd_Parity_Bit ( Result );

      return Result;

   end Pack;

   procedure Unpack ( A_Message : in Arinc_429_Types.Message_Type;
                      A_Discretes_Value : out Discretes_Type;
                      A_Sdi : out Sdi_Type;
                      A_Ssm : out Ssm_Type ) is

      No_Sdi_Message : Arinc_429_Types.No_Sdi_Message_Type;
      A_Discretes_Bit_Array : Discretes_Bit_Array;


   begin

      A_Ssm := To_Ssm_Type ( A_Message.The_Ssm );
      A_Sdi := To_Sdi_Type ( A_Message.The_Sdi );

      -- grab the discretes value
      if No_Sdi then

         -- convert to special message type w/o SDI field
         No_Sdi_Message := To_No_Sdi_Message_Type ( A_Message );

         -- Select out the bits actually used as discretes, then cast
         -- to the modulus type before converting to the private
         -- discretes type.

         A_Discretes_Bit_Array :=
           No_Sdi_Message.The_Value ( 1 .. Discretes_Type'Size );
         A_Discretes_Value :=
           To_Discretes_Type ( A_Discretes_Bit_Array );

      else

         -- Select out the bits actually used as discretes, then cast
         -- to the modulus type before converting to the private
         -- discretes type.

         A_Discretes_Bit_Array :=
           A_Message.The_Value ( 1 .. Discretes_Type_Size );

         A_Discretes_Value :=
           To_Discretes_Type ( A_Discretes_Bit_Array );

      end if;


   end Unpack;

begin

   -- Check that all the generic formal parameters are within allowed ranges

   if Sdi_Type'Size > Arinc_429_Types.Sdi_Field_Length then
      raise Sdi_Type_Too_Big;
   end if;

   if Ssm_Type'Size > Arinc_429_Types.Ssm_Field_Length then
      raise Ssm_Type_Too_Big;
   end if;

--     if Discretes_Type'Size /= Discretes_Mod_Type'Size then
--        raise Discretes_Type_Size_Mismatch;
--     end if;

   case No_Sdi is
      when False =>
         -- SDI bits are used for the SDI
         if Discretes_Type'Size > Arinc_429_Types.Value_Field_Length then
            raise Discretes_Type_Too_Big;
         end if;
      when True =>
         -- SDI bits are used for extra discrete bits
         if Discretes_Type'Size > Arinc_429_Types.No_Sdi_Value_Field_Length then
            raise Discretes_Type_Too_Big;
         end if;
   end case;

   -- At startup, calculate the reversed octal label for this message
   The_Reversed_Label := Arinc_429_Utils.To_Reversed_Label (Label);

--    -- Calculate the modulus for extracting the discretes value,
--    -- which sits in the low bits of the value field
--    The_Discretes_Value_Modulus := 2 ** Discretes_Type'Size;




exception

   when Error : Sdi_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BNR, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Sdi_Type'Size "
                   & Integer'Image ( Sdi_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Arinc_429_Types.Sdi_Field_Length ),
                   Severity => Log.Error );

   when Error : Ssm_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BNR, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Ssm_Type'Size "
                   & Integer'Image ( Ssm_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Arinc_429_Types.Ssm_Field_Length ),
                   Severity => Log.Error );

   when Error : Discretes_Type_Too_Big =>

      case No_Sdi is
         when False =>
            Log.Report ( Event =>
                           Ada.Exceptions.Exception_Information ( Error )
                         & " in generic instantiation of ARINC_429_BCD, "
                         & " label(decimal) => "
                         & Arinc_429_Types.Octal_Label'Image ( Label )
                         & ", Discretes_Type'Size "
                         & Integer'Image ( Discretes_Type'Size )
                         & " > maximum of "
                         & Integer'Image ( Arinc_429_Types.Value_Field_Length ),
                         Severity => Log.Error );
         when True =>
            Log.Report ( Event =>
                           Ada.Exceptions.Exception_Information ( Error )
                         & " in generic instantiation of ARINC_429_BCD, "
                         & " label(decimal) => "
                         & Arinc_429_Types.Octal_Label'Image ( Label )
                         & ", Discretes_Type'Size "
                         & Integer'Image ( Discretes_Type'Size )
                         & " > maximum of "
                         & Integer'Image ( Arinc_429_Types.No_Sdi_Value_Field_Length ),
                         Severity => Log.Error );
      end case;

   when others =>
      raise;


end Arinc_429_Discrete;
