-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Unchecked_Conversion;
with Arinc_429_Utils;
with Interfaces.C;
with Ada.Exceptions;
with Log;
pragma Elaborate_All ( Ada.Exceptions, Ada.Unchecked_Conversion, Log );
package body Arinc_429_Bnr is

   use type Interfaces.C.Unsigned;
   use type Interfaces.C.Int;
   use type Arinc_429_Types.Value_Field_Type;

   Max_Signed_Value_Bits : constant := Arinc_429_Types.Value_Field_Length - 1;
   -- Subtract 1 for the sign bit

   Max_Significant_Digits : constant := Max_Signed_Value_Bits + Arinc_429_Types.Sdi_Field_Length;

   subtype Signed_Value_Field_Type is
     Integer range -2 ** Max_Signed_Value_Bits .. 2 ** Max_Signed_Value_Bits - 1;

   The_Reversed_Label : Arinc_429_Types.Reversed_Octal_Label;

   The_Multiplication_Factor : Value_Type;

   The_Offset : Interfaces.C.Int;

   -- The number of bits in a SDI message value that are used. This number should not
   -- be changed, but making it a constant will cause the compiler to complain about
   -- possible constraint errors in code that will not ever be called.
   Value_Bits : Positive := Significant_Digits + 1;

   subtype Discretes_Bit_Array is
     Arinc_429_Types.Bit_Array ( 1 .. Discretes_Type'Size );

   -- Unchecked conversions between enumeration or private types and integer types

   function To_Bit_Array is new Ada.Unchecked_Conversion ( Discretes_Type,
                                                           Discretes_Bit_Array );

   function To_Discretes_Type is new Ada.Unchecked_Conversion ( Discretes_Bit_Array,
                                                                Discretes_Type );

   function To_Ssm_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Ssm_Type,
                                Target => Arinc_429_Types.Ssm_Field_Type );

   function To_Ssm_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Ssm_Field_Type,
                                Target => Ssm_Type );

   function To_Sdi_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Sdi_Type,
                                Target => Arinc_429_Types.Sdi_Field_Type );

   function To_Sdi_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Sdi_Field_Type,
                                Target => Sdi_Type );


   function Pack ( A_Value : in Value_Type;
                   A_Sdi   : in Sdi_Type;
                   A_Ssm   : in Ssm_Type;
                   A_Discretes_Value : in Discretes_Type )
     return Arinc_429_Types.Message_Type is

      Result : Arinc_429_Types.Message_Type;
      No_Sdi_Result : Arinc_429_Types.No_Sdi_Message_Type;
      Value_Int : Interfaces.C.Int;
      Value_Bit_Array : Arinc_429_Types.Unsigned_Bit_Array;
      A_Discretes_Bit_Array : Discretes_Bit_Array;

   begin

      -- convert and scale value to unsigned integer
      Value_Int
        := Interfaces.C.Int ( A_Value * The_Multiplication_Factor );

      Value_Bit_Array := Arinc_429_Utils.To_Unsigned_Bit_Array ( Value_Int );

      -- account for BNR data with more than 18 significant digits
      case Significant_Digits is
         when 1 .. Max_Signed_Value_Bits =>
            -- normal case, we pad to the right and we use SDI provided
            Result.The_Value := Arinc_429_Types.Zero_Value_Field;
              Result.The_Value ( Arinc_429_Types.Value_Field_Length - Value_Bits + 1
                               .. Arinc_429_Types.Value_Field_Length )
              := Value_Bit_Array ( 1 .. Value_Bits );

            -- discretes only for messages that have SDI
            if Discretes_Type'Size > 0 then

               A_Discretes_Bit_Array := To_Bit_Array ( A_Discretes_Value );

               Result.The_Value ( 1 .. Discretes_Type'Size )
                 := A_Discretes_Bit_Array ( 1 .. Discretes_Type'Size );

            end if;

            Result.The_Sdi
              := To_Sdi_Field_Type ( A_Sdi );

         when Max_Significant_Digits =>
            -- some lat and long messages use SDI bits for data
            No_Sdi_Result.The_Value := Arinc_429_Types.Zero_No_Sdi_Value_Field;
            No_Sdi_Result.The_Value ( Arinc_429_Types.No_Sdi_Value_Field_Length - Value_Bits + 1
                                      .. Arinc_429_Types.No_Sdi_Value_Field_Length )
              := Value_Bit_Array ( 1 .. Value_Bits );
            Result := Arinc_429_Utils.To_Message ( No_Sdi_Result );
         when others =>
            raise Constraint_Error;
      end case;


      -- copy in the reversed octal 429 label and the sign-status matrix

      Result.The_Label := The_Reversed_Label;
      Result.The_Ssm := To_Ssm_Field_Type ( A_Ssm );


      -- calculate parity bit for odd parity
      Result.The_Parity_Bit := False;
      Result.The_Parity_Bit := Arinc_429_Utils.Odd_Parity_Bit ( Result );

      return Result;

   end Pack;

   procedure Unpack ( A_Message : in Arinc_429_Types.Message_Type;
                      A_Value : out Value_Type;
                      A_Sdi : out Sdi_Type;
                      A_Ssm : out Ssm_Type;
                      A_Discretes_Value : out Discretes_Type) is

      No_Sdi_Message : Arinc_429_Types.No_Sdi_Message_Type;
      Value_Int : Interfaces.C.Int;
      Value_Bit_Array : Arinc_429_Types.Unsigned_Bit_Array;
      A_Discretes_Bit_Array : Discretes_Bit_Array;

   begin

      A_Ssm := To_Ssm_Type ( A_Message.The_Ssm );
      A_Sdi := To_Sdi_Type ( A_Message.The_Sdi );

      -- Initialize the 32-bit array with the sign bit of the input value.
      -- This way when we copy bits from the shorter bit array into the
      -- 32-bit array, the resulting 32-bit array will have the correct
      -- sign.  E.g.  A 12-bit value of -103 => 0xf99, but 0x00000f99 =>
      -- 3993, so initialize with all 1s and then copy the 12 bits to get
      -- 0xffffff99 => -103

      if A_Message.The_Value ( Arinc_429_Types.Value_Field_Length ) then
         Value_Bit_Array := Arinc_429_Types.Ones_Unsigned_Bit_Array;
      else
         Value_Bit_Array := Arinc_429_Types.Zero_Unsigned_Bit_Array;
      end if;

      case Significant_Digits  is

         when 1 .. Max_Signed_Value_Bits =>

            Value_Bit_Array ( 1 .. Value_Bits )
              := A_Message.The_Value ( Arinc_429_Types.Value_Field_Length - Value_Bits + 1
                                       .. Arinc_429_Types.Value_Field_Length );
            Value_Int := Arinc_429_Utils.To_Int ( Value_Bit_Array );

            -- grab the discretes value
            if Discretes_Type'Size > 0 then
               A_Discretes_Bit_Array ( 1 .. Discretes_Type'Size )
                 := A_Message.The_Value ( 1 .. Discretes_Type'Size );
               A_Discretes_Value
                 := To_Discretes_Type ( A_Discretes_Bit_Array );
            end if;

         when Max_Significant_Digits =>
            -- convert to type without SDI
            No_Sdi_Message := Arinc_429_Utils.To_No_Sdi_Message ( A_Message );

            Value_Bit_Array ( 1 .. Value_Bits )
              := No_Sdi_Message.The_Value ( Arinc_429_Types.No_Sdi_Value_Field_Length
                                            - Value_Bits + 1
                                            .. Arinc_429_Types.No_Sdi_Value_Field_Length );

            Value_Int := Arinc_429_Utils.To_Int ( Value_Bit_Array );

         when others =>
            raise Constraint_Error;
      end case;

      A_Value := Value_Type ( Value_Int );
      A_Value := A_Value / The_Multiplication_Factor;
--        := Value_Type ( Value_Int ) / The_Multiplication_Factor;

   end Unpack;

begin

   -- Check that all the generic formal parameters are within allowed ranges

   if Sdi_Type'Size > Arinc_429_Types.Sdi_Field_Length then
      raise Sdi_Type_Too_Big;
   end if;

   if Ssm_Type'Size > Arinc_429_Types.Ssm_Field_Length then
      raise Ssm_Type_Too_Big;
   end if;

   if Significant_Digits > Max_Significant_Digits then
      raise Too_Many_Significant_Digits;
   end if;

   case Significant_Digits is
      -- if we're not using the SDI bits, we can use the pad area for discretes
      when 1 .. Max_Signed_Value_Bits =>
         if Discretes_Type'Size > Max_Signed_Value_Bits - Significant_Digits then
            raise Discretes_Type_Too_Big;
         end if;
      -- if we're using the SDI bits, we can't have any discretes
      when others =>
         if Discretes_Type'Size > 0 then
            raise Discretes_Type_Too_Big;
         end if;
   end case;

   -- At startup, calculate the reversed octal label for this message
   The_Reversed_Label := Arinc_429_Utils.To_Reversed_Label (Label);

   -- Calculate the multiplication factor
   The_Multiplication_Factor
     := Value_Type ( 2 ** Significant_Digits )
     / Value_Type ( Scale );

exception

   when Error : Sdi_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BNR, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Sdi_Type'Size "
                   & Integer'Image ( Sdi_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Arinc_429_Types.Sdi_Field_Length ),
                   Severity => Log.Error );

   when Error : Ssm_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BNR, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Ssm_Type'Size "
                   & Integer'Image ( Ssm_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Arinc_429_Types.Ssm_Field_Length ),
                   Severity => Log.Error );

   when Error : Too_Many_Significant_Digits =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BNR, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Significant_Digits "
                   & Integer'Image ( Significant_Digits )
                   & " > maximum of "
                   & Integer'Image ( Max_Significant_Digits ),
                   Severity => Log.Error );

   when Error : Discretes_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BCD, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Discretes_Type'Size "
                   & Integer'Image ( Discretes_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Max_Signed_Value_Bits
                                     - Significant_Digits ),
                   Severity => Log.Error );

   when others =>
      raise;


end Arinc_429_Bnr;
