-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Exceptions;
with Ada.Unchecked_Conversion;
with Arinc_429_Utils;
with Log;
pragma Elaborate_All ( Ada.Exceptions, Ada.Unchecked_Conversion, Log );
package body Arinc_429_Bcd is

   use type Arinc_429_Types.Ssm_Field_Type;
   use type Arinc_429_Types.Value_Field_Type;

   Max_Normal_Significant_Digits : constant := 5;
   -- uses ARINC word bits 11 - 29 (where LSB is bit 1)
   -- high digit is three bits, low digits are 4 bits each

   Lat_Long_Significant_Digits : constant := 6;
   -- uses ARINC word bits 9 - 29 (where LSB is bit 1)
   -- high digit of degrees is one bit, low digits of degrees
   -- are 4 bits each; minutes and tenths digits are all 4 bits.

   Least_Significant_Digit_Index : Natural := 0;
   -- The five digits able to be represented are numbered from 0 .. 4
   -- starting with the least significant.  This value is the index of the
   -- least significant digit to be used.

   Lat_Long_Multiplier : constant := 1_000_000;
   -- Lat long values are input as this multiple of the actual angle
   -- For example, a longitude of -98.7389 degrees would be input
   -- as -98738900.  (If the compiler supports fixed decimals, that would
   -- be a better approach.)

   Lat_Long_Rounding_Offset : constant := Lat_Long_Multiplier / 2;
   -- We want to round to the nearest million; integer division truncates
   -- so we'll add this number first.

   Bits_Per_Bcd_Digit : constant := 4;

   type Digit_1_Bit_Type is new Value_Type range 0 .. 1;
   -- subtype for 1 bit long BCD digit field -- used for high
   -- bit of degrees for lat/long value

   type Digit_2_Bit_Type is new Value_Type range 0 .. 3;
   -- subtype for 2 bit long BCD digit field -- used for high bits
   -- of tenth of a minute for lat/long value

   type Digit_3_Bit_Type is new Value_Type range 0 .. 7;
   -- subtype for 3 bit long BCD digit field

   type Digit_Type is new Value_Type range 0 .. 9;
   -- subtype for decimal digit

   type Digit_Array_Type is array (Natural range <>) of Digit_Type;
   pragma Pack (Digit_Array_Type);
   for Digit_Array_Type'Component_Size use 4;

   type Bcd_Value_Type is
      record
         Low_Digits : Digit_Array_Type ( 0 .. 3);
         High_Digit : Digit_3_Bit_Type;
      end record;
--   pragma Pack (Bcd_Value_Type);
   for Bcd_Value_Type use
      record
        Low_Digits     at 0 range 0 .. 15;
        High_Digit     at 0 range 16 .. 18;
      end record;

   for Bcd_Value_Type'Size use 19;

   type Lat_Long_Value_Type is
      record
         Min_Digits     : Digit_Array_Type( 0 .. 2);
         Deg_Digits     : Digit_Array_Type( 0 .. 1);
         High_Deg_Digit : Digit_1_Bit_Type;
      end record;
--   pragma Pack (Lat_Long_Value_Type);
   for Lat_Long_Value_Type use
      record
         Min_Digits     at 0 range  0 .. 11;
         Deg_Digits     at 0 range 12 .. 19;
         High_Deg_Digit at 0 range 20 .. 20;
      end record;
   for Lat_Long_Value_Type'Size use 21;

   subtype Discretes_Bit_Array is
     Arinc_429_Types.Bit_Array ( 1 .. Discretes_Type'Size );

   -- Unchecked conversions between enumeration or private types and integer types

   function To_Bit_Array is new Ada.Unchecked_Conversion ( Discretes_Type,
                                                           Discretes_Bit_Array );

   function To_Discretes_Type is new Ada.Unchecked_Conversion ( Discretes_Bit_Array,
                                                                Discretes_Type );

   subtype Signed_Value_Field_Type is Integer range -2 ** 18 .. 2 ** 18 - 1;

   The_Reversed_Label : Arinc_429_Types.Reversed_Octal_Label := 0;

   The_Pad_Factor : Value_Type := 1;

   function To_Ssm_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Ssm_Type,
                                Target => Arinc_429_Types.Ssm_Field_Type );

   function To_Ssm_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Ssm_Field_Type,
                                Target => Ssm_Type );

   function To_Sdi_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Sdi_Type,
                                Target => Arinc_429_Types.Sdi_Field_Type );

   function To_Sdi_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Sdi_Field_Type,
                                Target => Sdi_Type );

   -- convert
   function To_Value_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Bcd_Value_Type,
                                Target => Arinc_429_Types.Value_Field_Type );

   function To_Bcd_Value_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.Value_Field_Type,
                                Target => Bcd_Value_Type );

   function To_No_Sdi_Value_Field_Type is new
     Ada.Unchecked_Conversion ( Source => Lat_Long_Value_Type,
                                Target => Arinc_429_Types.No_Sdi_Value_Field_Type );

   function To_Lat_Long_Value_Type is new
     Ada.Unchecked_Conversion ( Source => Arinc_429_Types.No_Sdi_Value_Field_Type,
                                Target => Lat_Long_Value_Type );


   function Pack ( A_Value : in Value_Type;
                   A_Sdi   : in Sdi_Type;
                   A_Ssm   : in Ssm_Type;
                   A_Discretes_Value : in Discretes_Type )
     return Arinc_429_Types.Message_Type is

      Result : Arinc_429_Types.Message_Type;
      Value_Temp : Value_Type;
      Value_Int : Value_Type;
      Value_Is_Negative : Boolean := False;
      Bcd_Value_Field  : Bcd_Value_Type
        := ( Low_Digits => ( others => 0 ),
             High_Digit => 0 );

      A_Discretes_Bit_Array : Discretes_Bit_Array;

   begin

      -- zero out the value field
      Result.The_Value := Arinc_429_Types.Zero_Value_Field;

      -- store value in temp variable so we can write to it
      Value_Temp := A_Value;

      Value_Is_Negative := ( Value_Temp < 0 );
      if Value_Is_Negative then
         -- we want the absolute value
         Value_Temp := -Value_Temp;
      end if;

      case Significant_Digits is
         when 1 .. Max_Normal_Significant_Digits =>

            -- convert and scale value to unsigned integer, and factor out sign
            Value_Int
              := The_Pad_Factor * Value_Temp;
            -- convert to individual digits
            declare
               Value_Int_Temp : Value_Type := Value_Int;
            begin
               for I in 0 .. 3 loop
                  Bcd_Value_Field.Low_Digits (I) := Digit_Type(Value_Int_Temp mod 10);
                  Value_Int_Temp := Value_Int_Temp / 10;
               end loop;
               -- what's left after last division had better be less than 8
               Bcd_Value_Field.High_Digit := Digit_3_Bit_Type(Value_Int_Temp);
            end;


            Result.The_Value
              := To_Value_Field_Type ( Bcd_Value_Field );
            Result.The_Sdi
              := To_Sdi_Field_Type ( A_Sdi );

            -- add in any discretes to low end of data field:
            -- 1. the appropriate bits are already cleared
            -- 2. convert the discrete record type to an integer of
            --    the appropriate size
            -- 3. "or" or add in this integer to the data field

            if Discretes_Type'Size > 0 then

               A_Discretes_Bit_Array := To_Bit_Array ( A_Discretes_Value );

               Result.The_Value ( 1 .. Discretes_Type'Size )
                 := A_Discretes_Bit_Array ( 1 .. Discretes_Type'Size );

            end if;

         when Lat_Long_Significant_Digits =>

            -- special case for latitude/longitude
            -- input as an integer in millionths of degrees
            declare

               -- NOTE FOR CONVERSION --
               -- LAT LONG INPUT NEEDS TO BE AS A SIGNED INT IN FORMAT DDDMMM
               -- OR SOME OTHER FORMAT TO BE DISCUSSED WITH JIM REYNOLDS

               -- we want to truncate not round

               Degrees_Fraction : Value_Type
                 := Value_Temp mod Lat_Long_Multiplier;

               Degrees_Int : Value_Type
                 := ( Value_Temp - Degrees_Fraction ) / Lat_Long_Multiplier;

               Tenth_Minutes : Value_Type
                 := ( Degrees_Fraction * 600 + Lat_Long_Rounding_Offset ) / Lat_Long_Multiplier;
               Minutes_Int : Value_Type
                 := Tenth_Minutes / 10;
               Minutes_Tenths_Int : Value_Type
                 := Tenth_Minutes mod 10;
               Lat_Long_Value_Field : Lat_Long_Value_Type;

               Lat_Long_Result : Arinc_429_Types.No_Sdi_Message_Type;

            begin

               Lat_Long_Value_Field.Min_Digits ( 0 )
                 := Digit_Type(Minutes_Tenths_Int);
               Lat_Long_Value_Field.Min_Digits ( 1 )
                 := Digit_Type(Minutes_Int mod 10);
               Lat_Long_Value_Field.Min_Digits ( 2 )
                 := Digit_Type(Minutes_Int / 10);
               Lat_Long_Value_Field.Deg_Digits ( 0 )
                 := Digit_Type(Degrees_Int mod 10);
               Lat_Long_Value_Field.Deg_Digits ( 1 )
                 := Digit_Type((Degrees_Int / 10) mod 10);
               Lat_Long_Value_Field.High_Deg_Digit
                 := Digit_1_Bit_Type(Degrees_Int / 100);

               Lat_Long_Result.The_Value
                 := To_No_Sdi_Value_Field_Type ( Lat_Long_Value_Field );

               -- convert back to standard format with 19-bit value field
               Result := Arinc_429_Utils.To_Message ( Lat_Long_Result );

            end;


         when others =>
            raise Constraint_Error;
      end case;

      Result.The_Ssm := To_Ssm_Field_Type ( A_Ssm );

      -- If the SSM was set by user to no computed data or functional
      -- test, we don't want to touch it, but if it was normal (0) and
      -- a negative value was supplied, we need to set both SSM bits
      -- to indicate negative (minus, down, south, west, from, below).

      if ( Result.The_Ssm = 0 ) and Value_Is_Negative then
         Result.The_Ssm := 2#11#;
      end if;

      -- copy in the reversed octal 429 label
      Result.The_Label := The_Reversed_Label;

      -- calculate parity bit for odd parity
      Result.The_Parity_Bit := False;
      Result.The_Parity_Bit := Arinc_429_Utils.Odd_Parity_Bit ( Result );

      return Result;

   end Pack;

   procedure Unpack ( A_Message : in Arinc_429_Types.Message_Type;
                      A_Value : out Value_Type;
                      A_Sdi : out Sdi_Type;
                      A_Ssm : out Ssm_Type;
                      A_Discretes_Value : out Discretes_Type ) is

      Value_Temp : Value_Type;
      Value_Int : Value_Type;
      Bcd_Value_Field  : Bcd_Value_Type;
      A_Discretes_Bit_Array : Discretes_Bit_Array;


   begin

      A_Sdi := To_Sdi_Type ( A_Message.The_Sdi );
      A_Ssm := To_Ssm_Type ( A_Message.The_Ssm );

      case Significant_Digits is
         when 1 .. Max_Normal_Significant_Digits =>

            Bcd_Value_Field := To_Bcd_Value_Type ( A_Message.The_Value );

            -- convert to individual digits
            declare
               Value_Int_Temp : Value_Type;
            begin
               -- shift in the high digit
               Value_Int_Temp := Value_Type(Bcd_Value_Field.High_Digit);

               -- Look at the remaining digits from high to low, but
               -- ignore the pad digits -- they may contain discretes
               -- and may be non-zero.
               for I in
                 reverse
                 Least_Significant_Digit_Index .. 3 loop
                  Value_Int_Temp :=
                    Value_Int_Temp * 10 + Value_Type(Bcd_Value_Field.Low_Digits (I));
               end loop;
               Value_Int := Value_Int_Temp;
            end;

--             Value_Temp :=
--               Value_Type (Value_Int / The_Pad_Factor)
--               * The_Multiplication_Factor;

            Value_Temp
              := Value_Int;

      -- grab the discretes value
            if Discretes_Type'Size > 0 then
               A_Discretes_Bit_Array ( 1 .. Discretes_Type'Size )
                 := A_Message.The_Value ( 1 .. Discretes_Type'Size );
               A_Discretes_Value
                 := To_Discretes_Type ( A_Discretes_Bit_Array );
            end if;

         when Lat_Long_Significant_Digits =>

            declare
               Lat_Long_Message : Arinc_429_Types.No_Sdi_Message_Type;
               Lat_Long_Value_Field : Lat_Long_Value_Type;
               Tenth_Minutes : Value_Type;
               Degrees_Int : Value_Type;
            begin

               -- convert message to type without an SDI (SDI bits used to
               -- extend value field)

               Lat_Long_Message
                 := Arinc_429_Utils.To_No_Sdi_Message ( A_Message );

               -- convert value field to BCD digits

               Lat_Long_Value_Field
                 := To_Lat_Long_Value_Type ( Lat_Long_Message.The_Value );
               Degrees_Int
                 := Value_Type(Lat_Long_Value_Field.High_Deg_Digit) * 100 +
                 Value_Type(Lat_Long_Value_Field.Deg_Digits ( 1 )) * 10 +
                 Value_Type(Lat_Long_Value_Field.Deg_Digits ( 0 ));
               Tenth_Minutes
                 := Value_Type(Lat_Long_Value_Field.Min_Digits ( 2 )) * 100 +
                 Value_Type(Lat_Long_Value_Field.Min_Digits ( 1 )) * 10 +
                 Value_Type(Lat_Long_Value_Field.Min_Digits ( 0 ));
               Value_Temp
                 := Value_Type ( Degrees_Int ) * Lat_Long_Multiplier +
                 Value_Type ( Tenth_Minutes ) * Lat_Long_Multiplier / 600;
            end;

         when others =>
            raise Constraint_Error;
      end case;

      -- account for negative SSM
      if A_Message.The_Ssm = 2#11# then
        Value_Temp := - Value_Temp;
      end if;

      A_Value := Value_Temp;

   end Unpack;

begin

   -- Check that all the generic formal parameters are within allowed ranges

   if Sdi_Type'Size > Arinc_429_Types.Sdi_Field_Length then
      raise Sdi_Type_Too_Big;
   end if;

   if Ssm_Type'Size > Arinc_429_Types.Ssm_Field_Length then
      raise Ssm_Type_Too_Big;
   end if;

   if Significant_Digits > Lat_Long_Significant_Digits then
      raise Too_Many_Significant_Digits;
   end if;

   case Significant_Digits is
      -- if we're not using the SDI bits, we can use the pad area for discretes
      when 1 .. Max_Normal_Significant_Digits =>
         Least_Significant_Digit_Index
           := Max_Normal_Significant_Digits - Significant_Digits;
         if Discretes_Type'Size >
           Least_Significant_Digit_Index * Bits_Per_Bcd_Digit then
            raise Discretes_Type_Too_Big;
         end if;
      -- if we're using the SDI bits, we can't have any discretes
      when others =>
         Least_Significant_Digit_Index := 0;
         if Discretes_Type'Size > 0 then
            raise Discretes_Type_Too_Big;
         end if;
   end case;

   -- At startup, calculate the reversed octal label for this message
   The_Reversed_Label := Arinc_429_Utils.To_Reversed_Label (Label);

   -- Calculate the pad factor to shift the number into the correct
   -- position in the array of BCD digits
   if Significant_Digits <= Max_Normal_Significant_Digits then
      The_Pad_Factor
        := 10 ** ( Least_Significant_Digit_Index );
   end if;

exception

   when Error : Sdi_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BCD, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Sdi_Type'Size "
                   & Integer'Image ( Sdi_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Arinc_429_Types.Sdi_Field_Length ),
                   Severity => Log.Error );

   when Error : Ssm_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BCD, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Ssm_Type'Size "
                   & Integer'Image ( Ssm_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Arinc_429_Types.Ssm_Field_Length ),
                   Severity => Log.Error );

   when Error : Too_Many_Significant_Digits =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BCD, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Significant_Digits "
                   & Integer'Image ( Significant_Digits )
                   & " > maximum of "
                   & Integer'Image ( Lat_Long_Significant_Digits ),
                   Severity => Log.Error );

   when Error : Discretes_Type_Too_Big =>

      Log.Report ( Event =>
                     Ada.Exceptions.Exception_Information ( Error )
                   & " in generic instantiation of ARINC_429_BCD, "
                   & " label(decimal) => "
                   & Arinc_429_Types.Octal_Label'Image ( Label )
                   & ", Discretes_Type'Size "
                   & Integer'Image ( Discretes_Type'Size )
                   & " > maximum of "
                   & Integer'Image ( Least_Significant_Digit_Index
                                     * Bits_Per_Bcd_Digit ),
                   Severity => Log.Error );

   when others =>
      raise;


end Arinc_429_Bcd;
