-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Streams;
with Buffer_Stream;
with Buffer_Stream.Save;
with Buffer_Stream.Restore;

-------------------------------------------------------------------------------
-- This package implements an abstract definition of a scheduled item. Any
-- subsystem that wishes to be scheduled on this system must implement their
-- own version of Instance, register it with the scheduler registrar, and add
-- the appropriate definition to sim.cfg.
-------------------------------------------------------------------------------
package Scheduler is

   type Instance is abstract tagged private;

   type Handle is access all Instance'Class;

   ----------------------------------------------------------------------------
   -- Initialize defines non-real-time initializations run each time the
   -- simulation is started
   ----------------------------------------------------------------------------
   procedure Initialize
     ( An_Instance : in out Instance );

   ----------------------------------------------------------------------------
   -- For reinitialization after an exception
   ----------------------------------------------------------------------------
   procedure Re_Initialize
     ( An_Instance : in out Instance );

   ----------------------------------------------------------------------------
   -- Update defines normal realtime behavior of the subsystem
   ----------------------------------------------------------------------------
   procedure Update
     ( An_Instance          : in out Instance;
       Integration_Constant : in     Float );

   ----------------------------------------------------------------------------
   -- Update_In_Freeze defines behavior of subsystem during Total Freeze
   ----------------------------------------------------------------------------
   procedure Update_In_Freeze
     ( An_Instance : in out Instance );


   -------------------------------------------------------------------------------
   -- Routine to save a snapshot of the instance to the given stream.
   --
   -- Note that the predefined attributes 'Write and 'Output can be used on any
   -- object to store its value in the given stream.
   --
   -- All models should override this routine with an implementation that saves
   -- off any data they would need to perform a complete restore of their model,
   -- and proceed to run from there without error.
   -- If this routine saves off any data that is not physically contained within
   -- the instance itself (eg: in a package body or an object pointed to by a
   -- pointer in the instance), then the function Snapshot_Size will need to be
   -- overridden as well.
   -- Note that performing a 'Write or 'Output on a pointer will save the value
   -- of the pointer rather than the values it points to. That is almost certainly
   -- *not* what you want.
   -------------------------------------------------------------------------------
   procedure Save
     ( An_Instance : in out Instance;
       To_Stream   : access Ada.Streams.Root_Stream_Type'Class
     );

   -------------------------------------------------------------------------------
   -- Routine to retrieve a snapshot of the instance from the given stream.
   --
   -- Note that the predefined attributes 'Read and 'Input can be used on any
   -- object to retrieve its value in the given stream. This routine should be
   -- written so that data is read from the stream in the same order it was placed
   -- into the stream.
   --
   -------------------------------------------------------------------------------
   procedure Restore
     ( An_Instance : in out Instance;
       From_Stream : access Ada.Streams.Root_Stream_Type'Class
     );


   -------------------------------------------------------------------------------
   -- Return a pointer to the save stream associated with the given scheduler.
   -------------------------------------------------------------------------------
   function Save_Stream (An_Instance : in Instance'Class) return
     Buffer_Stream.Save.Handle;

   -------------------------------------------------------------------------------
   -- Return a pointer to the restore stream associated with the given scheduler.
   -------------------------------------------------------------------------------
   function Restore_Stream (An_Instance : in Instance'Class) return
     Buffer_Stream.Restore.Handle;

   -------------------------------------------------------------------------------
   -- Returns the maximum amount of *bytes* that a snapshot takes up. This is
   -- used by Create_Streams to help size some interneral buffers.
   --
   -- By default, this will return the size of Instance in bytes (not counting the
   -- Save_Stream and Restore_Stream fields below). But you can override this to
   -- return a different value. You should do this if you wrote a custom Save
   -- routine which writes out data that is not contained directly in your
   -- instance (ie: in an object pointed to by your instance, or in a package body
   -- somewhere).
   -------------------------------------------------------------------------------
   function Snapshot_Size (An_Instance : Instance) return Natural;

private

   --
   -- Save and Restore stream constructor and accessor methods.
   -- Note: These routines are not intended for use by the subsystems
   --

   -------------------------------------------------------------------------------
   -- Create the save and restore streams for the given scheduler, using the given
   -- amount of bytes per instance stored.
   -------------------------------------------------------------------------------
   procedure Create_Streams (An_Instance : in Handle);

   type Instance is tagged record
      Save_Stream    : Buffer_Stream.Save.Handle;
      Restore_Stream : Buffer_Stream.Restore.Handle;
   end record;

end Scheduler;
