-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Saved_Data_Header;
with Save_Stream_Notifier;
with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Log;

-------------------------------------------------------------------------------
-- This package implements an abstract definition of a scheduled item.
-------------------------------------------------------------------------------
package body Scheduler is

   -- Null implementations of some of the primitives. If the users want these
   -- routines to actually do something, they will have to override them.

   procedure Initialize
     ( An_Instance : in out Instance ) is
   begin
      null;
   end Initialize;

   procedure Re_Initialize
     ( An_Instance : in out Instance ) is
   begin
      null;
   end Re_Initialize;

   procedure Update
     ( An_Instance          : in out Instance;
       Integration_Constant : in     Float ) is
   begin
      null;
   end Update;

   procedure Update_In_Freeze
     ( An_Instance : in out Instance ) is
   begin
      null;
   end Update_In_Freeze;

   -------------------------------------------------------------------------------
   -- Routine to save a snapshot of the instance to the given stream.
   -------------------------------------------------------------------------------
   procedure Save
     ( An_Instance : in out Instance;
       To_Stream   : access Ada.Streams.Root_Stream_Type'Class
     ) is
   begin
      null;
   end Save;

   -------------------------------------------------------------------------------
   -- Routine to retrieve a snapshot of the instance from the given stream.
   -------------------------------------------------------------------------------
   procedure Restore
     ( An_Instance : in out Instance;
       From_Stream : access Ada.Streams.Root_Stream_Type'Class
     ) is
   begin
      null;
exception
      when Error : others =>
         Log.Report (Event => "Exception in Scheduler.Restore: " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);
   end Restore;

   -------------------------------------------------------------------------------
   -- Returns the maximum amount of *bytes* that a snapshot takes up. This is
   -- used by Create_Streams to help size some interneral buffers.
   --
   -- By default, this will return the size of Instance in bytes (not counting the
   -- Save_Stream and Restore_Stream fields below). But you can override this to
   -- return a different value. You should do this if you wrote a custom Save
   -- routine which writes out data that is not contained directly in your
   -- instance (ie: in an object pointed to by your instance, or in a package body
   -- somewhere).
   -------------------------------------------------------------------------------
   function Snapshot_Size (An_Instance : Instance) return Natural is
   begin
      return ((An_Instance'Size -
               (An_Instance.Save_Stream'Size + An_Instance.Restore_Stream'Size)) +
              7) / 8;
   end Snapshot_Size;

   -------------------------------------------------------------------------------
   -- Create the save and restore streams for the given scheduler, using the given
   -- amount of bytes per instance stored.
   -------------------------------------------------------------------------------
   procedure Create_Streams (An_Instance : in Handle) is
      Save_Size : constant Natural := Snapshot_Size(An_Instance.all);

      Forget_About_It : Save_Stream_Notifier.Handle;
   begin
      An_Instance.Save_Stream    := new Buffer_Stream.Save.Instance;
      An_Instance.Restore_Stream := new Buffer_Stream.Restore.Instance;

      Buffer_Stream.Save.Create
        (Stream   => An_Instance.Save_Stream.all,
         Max_Size => Save_Size + (3 * Saved_Data_Header.Bytes)
         );

      Buffer_Stream.Restore.Create
        (Stream   => An_Instance.Restore_Stream.all,
         Max_Size => (Save_Size + (2 * Saved_Data_Header.Bytes)) * 2 +
         Saved_Data_Header.Bytes
         );

      -- Create the notifier for the save stream
      Forget_About_It := new Save_Stream_Notifier.Instance(An_Instance);
   end Create_Streams;

   -------------------------------------------------------------------------------
   -- Return a pointer to the save stream associated with the given scheduler.
   -------------------------------------------------------------------------------
   function Save_Stream (An_Instance : in Instance'Class) return
     Buffer_Stream.Save.Handle is
   begin
      return An_Instance.Save_Stream;
   end Save_Stream;

   -------------------------------------------------------------------------------
   -- Return a pointer to the restore stream associated with the given scheduler.
   -------------------------------------------------------------------------------
   function Restore_Stream (An_Instance : in Instance'Class) return
     Buffer_Stream.Restore.Handle is
   begin
      return An_Instance.Restore_Stream;
   end Restore_Stream;

end Scheduler;
