-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Buffer_Stream.Simulation;
with Mission_Time;
with Scheduler.Band;
with Scheduler.Registrar;
with JPATS_Reposition;

with Angle_Types;
with JPATS_DCLS;
with JPATS_Simulated_Aircraft;
with JPATS_Flight_Instruments;
with JPATS_Radio_DB_IF;
with Log;

pragma Elaborate_All (Buffer_Stream.Simulation);

-------------------------------------------------------------------------------
-- This package provides funtionality for creating and maintaining snapshots of
-- the entire system.
-------------------------------------------------------------------------------
package body Scheduler.Snapshot is

   type Snapshot_Record is record
      Descriptor : Snapshot_Descriptor;
      Stream     : Buffer_Stream.Simulation.Handle;
   end record;

   type Snapshot_Array is array (Snapshot_Index) of Snapshot_Record;

   Snapshot_List : Snapshot_Array;
   First_Free    : Snapshot_Index := Snapshot_Index'First;

   ----------------------------------------------------------------------------
   -- Internal helper routines
   --

   ----------------------------------------------------------------------------
   -- Externally visible routines
   --

   ----------------------------------------------------------------------------
   -- Initialize this facility by allocating each of the snapshot buffers with
   -- enough space to store the given amount of stream elements.
   ----------------------------------------------------------------------------
   procedure Initialize (Snapshot_Size : in Natural) is
   begin
      -- Allocate and create all of the snapshots.
      for Index in Snapshot_Index loop
         Snapshot_List(Index).Stream := new Buffer_Stream.Simulation.Instance;
         Buffer_Stream.Simulation.Create
           (Stream   => Snapshot_List(Index).Stream.all,
            Max_Size => Snapshot_Size
            );
      end loop;
   end Initialize;

   ----------------------------------------------------------------------------
   -- Create a new snapshot of the simulation. This is a CPU-intensive
   -- operation, which should not be called directly by a real-time model.
   --
   -- Calling it from a background task should ensure the following:
   --   o It won't interfere with real-time processing.
   --   o It should be able to run w/o being interrupted. I figure this should
   --     be the case, because all realtime tasks get to run to completion
   --     first. If we can't easily snapshot everything before the next
   --     real-time task starts off, then we won't have enough time to
   --     perform save/restores either.
   ----------------------------------------------------------------------------
   procedure Create is
      New_Index : Snapshot_Index := First_Free;
   begin
      -- If there aren't any free entries, free up one
      if Snapshot_List(New_Index).Descriptor.Valid = True then
         Delete (Snapshot_List'First);
      end if;

      -- Increment the First_Free index
      if First_Free < Snapshot_Index'Last then
         First_Free := First_Free + 1;
      end if;

      -- Take the actual snapshot
      Buffer_Stream.Simulation.Open_Writes (Snapshot_List(New_Index).Stream.all);
      Band.Snapshot_All (Snapshot_List(New_Index).Stream);
      Buffer_Stream.Simulation.Close_Writes (Snapshot_List(New_Index).Stream.all);

      -- Fill in the snapshot descriptor
      -- (Fill this in with data as per the IOS's spec)
      Snapshot_List(New_Index).Descriptor :=
        (Mission_Time => Mission_Time.Get,
         Latitude     => JPATS_Simulated_Aircraft.Get_North,
         Longitude    => JPATS_Simulated_Aircraft.Get_East,
         Altitude     => -JPATS_Simulated_Aircraft.Get_Aircraft_Height_Above_Local_Terrain,
         Heading      => Angle_Types.Radians_To_Degrees (JPATS_Simulated_Aircraft.Get_Hdg_Angle)
         - JPATS_Radio_DB_IF.Magnetic_Variation,
         Airspeed     => JPATS_Flight_Instruments.Primary_IAS,
         Valid        => True
         );

   end Create;

   ----------------------------------------------------------------------------
   -- Restore the simulation from the given snapshot. This is a time-consuming
   -- operation, so it needs to be performed from a non-realtime thread.
   ----------------------------------------------------------------------------
   procedure Restore (Index : in Snapshot_Index) is
   begin
      Locked := true;
      Set_Index(Index);
      -- Warn DCLS about possible discontinuities
      JPATS_DCLS.Set_Filter;
      JPATS_Simulated_Aircraft.Set_System_Freeze;
      Log.Report("Scheduler.Snapshot.restore: Snapshot Index coming in is " & Snapshot_Index'Image(Index));
      if not Snapshot_List(Get_Index).Descriptor.Valid then
         Log.Report (Event    => "Attempt to restore invalid snapshot (ignored).",
                     Severity => Log.Warning
                     );
         return;
      end if;
      Buffer_Stream.Simulation.Open_Reads (Snapshot_List(Get_Index).Stream.all);
      while Buffer_Stream.Simulation.Data_Available (Snapshot_List(Get_Index).Stream) loop
         Scheduler.Restore
           (An_Instance => Registrar.Retrieve_Handle(String'Input(Snapshot_List(Get_Index).Stream)).all,
            From_Stream => Snapshot_List(Get_Index).Stream
            );
      end loop;
      Buffer_Stream.Simulation.Close_Reads (Snapshot_List(Get_Index).Stream.all);
      Log.Report("Controls not in agreement");
      JPATS_Reposition.Controls_Not_In_Agreement;
      Locked := False;
      Log.Report("Restore Complete");
   exception
      when Error : others =>
         Log.Report (Event => "Exception in Snapshot.Restore: " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);

         -- *** Should we set output ( Host --> DCLS) stick/rudder forces to zero in dcls_container.io_interface here
         --     to try and stop stick slams? I don't know enough about it to try it safely ******

         Log.Report("Scheduler.Snapshot.restore: Snapshot Index after exception is " & Snapshot_Index'Image(Index));
         Log.Report("Scheduler.Snapshot.restore: Snapshot Index global after exception is " & Snapshot_Index'Image(Get_Index));
         Buffer_Stream.Simulation.Close_Reads (Snapshot_List(Get_Index).Stream.all);
         JPATS_Reposition.Abort_Snapshot;

   end Restore;

   ----------------------------------------------------------------------------
   -- Delete the given snapshot.
   ----------------------------------------------------------------------------
   procedure Delete (Index : in Snapshot_Index) is
      Free_Index  : Snapshot_Index := Index;

      -- Save off the stream of the old snapshot
      Swap_Handle : Buffer_Stream.Simulation.Handle := Snapshot_List (Index).Stream;
   begin

      if not Snapshot_List(Index).Descriptor.Valid then
         Log.Report (Event    => "Attempt to delete invalid snapshot (ignored).",
                     Severity => Log.Warning
                     );
         return;
      end if;

      Buffer_Stream.Simulation.Flush (Snapshot_List(Free_Index).Stream.all);
      Snapshot_List(Free_Index).Descriptor.Valid := False;

      if Index < Snapshot_Index'Last then

         -- Move all the rest of the snapshots up one position
         for Source in Index + 1..Snapshot_Index'Last loop
            exit when not Snapshot_List(Source).Descriptor.Valid;

            Snapshot_List(Free_Index) := Snapshot_List(Source);
            Free_Index := Source;
         end loop;

         -- Put the stream of the deleted entry at the position of the last vacated entry.
         Snapshot_List (Free_Index).Stream           := Swap_Handle;
         Snapshot_List (Free_Index).Descriptor.Valid := False;
      end if;

      -- Decrement the First_Free index
      if First_Free > Snapshot_Index'First and Free_Index < Snapshot_Index'Last then
         First_Free := First_Free - 1;
      end if;

   end Delete;

   ----------------------------------------------------------------------------
   -- Get the descriptor for the snapshot with the given index.
   ----------------------------------------------------------------------------
   function Get_Descriptor (Index : in Snapshot_Index) return Snapshot_Descriptor is
   begin
      return Snapshot_List (Index).Descriptor;
   end Get_Descriptor;

   function Snapshot_Index_Locked return Boolean is
   begin
      return Locked;
   end Snapshot_Index_Locked;

   function Get_Index return Snapshot_Index is
   begin
      return Index;
   end Get_Index;

   procedure Set_Index(The_Index : in Snapshot_Index) is
   begin
      Index := The_Index;
   end Set_Index;


end Scheduler.Snapshot;
