-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;
with Ada.Tags;
with Ada.Text_IO;

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Log;

-------------------------------------------------------------------------------
-- This package allows subsystem schedulers to register themselves with the
-- system for scheduling.
-------------------------------------------------------------------------------
package body Scheduler.Registrar is

   -- Maximum length of a file name.
   Max_Name_Length : constant Positive := 100;

   ----------------------------------------------------------------------------
   -- The registration list declarations

   type Controller_Class_List_Node_Type;

   type Controller_Class_List_Node_Ptr_Type is access Controller_Class_List_Node_Type;

   type Controller_Class_List_Node_Type is record
      Name          : String( 1 .. Max_Name_Length );
      Name_Length   : Natural := 0;
      Module_Handle : Scheduler.Handle;
      Next          : Controller_Class_List_Node_Ptr_Type;
   end record;

   -- List of registered controller classes.
   Controller_Class_List      : Controller_Class_List_Node_Ptr_Type;
   Controller_Class_List_Last : Controller_Class_List_Node_Ptr_Type;

   ----------------------------------------------------------------------------
   -- Public subprograms

   ----------------------------------------------------------------------------
   -- Register the given subsystem scheduler for scheduling. The name will be
   -- taken automaticly from the scheduler's tag. This is a bit cheesy, as
   -- the tag is implementation defined. It also prevents multiple schedulers
   -- of the same type. But it does save a parameter, and removes one source
   -- of errors in registration mismatches.
   ----------------------------------------------------------------------------
   procedure Register( The_Handle  : in Scheduler.Handle ) is

      End_Marker        : constant String  := ".instance";
      Tag_String        : constant String  := Ada.Tags.External_Tag(The_Handle'Tag);

      Index_Not_Found   : constant := 0;
      Index_Of_Instance : constant Natural := Ada.Strings.Fixed.Index
                                                (Source  => Tag_String,
                                                 Pattern => End_Marker );
   begin

      if Index_Of_Instance /= Index_Not_Found then

         -- Spit out the registered name. This is useful for a progress report during
         -- elaboration, so we have kept it in.
         Ada.Text_IO.Put_Line( Tag_String (Tag_String'First..Index_Of_Instance - 1) & " registered.");

         -- Create a new node on the end of the class list
         if Controller_Class_List = null then

            Controller_Class_List      := new Controller_Class_List_Node_Type;
            Controller_Class_List_Last := Controller_Class_List;
         else

            Controller_Class_List_Last.Next := new Controller_Class_List_Node_Type;
            Controller_Class_List_Last      := Controller_Class_List_Last.Next;
         end if;

         -- Put the registration in the new node
         Controller_Class_List_Last.Name_Length := Index_Of_Instance - 1;
         Controller_Class_List_Last.Name (1..Controller_Class_List_Last.Name_Length)
           := Ada.Strings.Fixed.Translate
           (Tag_String (1..Index_Of_Instance - 1), Ada.Strings.Maps.Constants.Lower_Case_Map);

         Controller_Class_List_Last.Module_Handle := The_Handle;
         Controller_Class_List_Last.Next          := null;

      else
         -- Logging may not be here yet, so we have to just spit out messages to the
         -- screen.
         Ada.Text_IO.Put_Line ("ERROR: Unable to register object """ &
                               Ada.Tags.External_Tag( The_Handle'Tag ) & """.");
      end if;

   exception
      when others =>
         Ada.Text_IO.Put_Line ("ERROR: Unable to register object """ &
                               Ada.Tags.External_Tag( The_Handle'Tag ) &
                               """ due to exception.");
      raise;

   end Register;

   ----------------------------------------------------------------------------
   -- Retrieve the handle for the subsystem scheduler with the given name.
   ----------------------------------------------------------------------------
   function Retrieve_Handle ( The_Name : String) return Scheduler.Handle is

      -- Points to the current entry on the linked list.
      Node           : Controller_Class_List_Node_Ptr_Type := Controller_Class_List;
      Lowercase_Name : constant String :=
        Ada.Strings.Fixed.Translate
        (The_Name,
         Ada.Strings.Maps.Constants.Lower_Case_Map
         );

   begin

      while Node /= null loop

         if Node.Name(1..Node.Name_Length) = Lowercase_Name then
            return Node.Module_Handle;
         end if;

         Node := Node.Next;
      end loop;

      return null;

   exception
      when Error : others =>
         Log.Report (Event => "Exception in Registrar.Retrieve_Handle: " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);
         return null;

   end Retrieve_Handle;


   ----------------------------------------------------------------------------
   -- Retrieve the name for the given subsystem scheduler. A zero length string
   -- will be returned if the search fails.
   ----------------------------------------------------------------------------
   function Retrieve_Name ( Handle : in Scheduler.Handle) return String is

      use type Scheduler.Handle;

      -- Points to the current entry on the linked list.
      Node : Controller_Class_List_Node_Ptr_Type := Controller_Class_List;

   begin
      while Node /= null loop

         if Node.Module_Handle = Handle then
            return Node.Name (1..Node.Name_Length);
         end if;

         Node := Node.Next;
      end loop;

      return "";
   end Retrieve_Name;

end Scheduler.Registrar;
