-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Real_Time;
with Buffer_Stream.Restore;
with Buffer_Stream.Save;
with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Log;

with Scheduler.Configuration;

-- Allow builtin operators for the following types
use type Ada.Real_Time.Time;
use type Ada.Real_Time.Time_Span;

pragma Elaborate_All (Scheduler.Configuration);
pragma Elaborate_All (Ada.Real_Time);

-------------------------------------------------------------------------------
-- This package provides control over the record/playback functionality.
-------------------------------------------------------------------------------
package body Scheduler.Record_Playback is

   -- Indicates the amount of cycles between snapshots.
   Cycles_Per_Snap : constant Natural :=
     Natural(Ada.Real_Time.To_Duration(Configuration.Snapshot_Period)) *
     Configuration.Cycles_Per_Frame;

   Cycles_Till_Snap : Natural := Cycles_Per_Snap;

   ----------------------------------------------------------------------------
   -- Check to see if its time to snapshot all the models yet.
   -- The time is based on the number of previous calls to this routine, and
   -- the value of cycle (which must be 0 for snapshots to occur).
   ----------------------------------------------------------------------------
   function Model_Snapshot_Time (Cycle : Natural) return Boolean is
   begin

      if Cycles_Till_Snap > 0 then
         Cycles_Till_Snap := Cycles_Till_Snap - 1;
         return False;
      elsif Cycle = 0 then
         Cycles_Till_Snap := Cycles_Per_Snap - 1;
         return True;
      else
         return False;
      end if;

   end Model_Snapshot_Time;

   -------------------------------------------------------------------------------
   -- Take a snapshot of the state of the given module
   -------------------------------------------------------------------------------
   procedure Snapshot (Module : in out Instance'Class) is

      Save_Stream_Handle : constant Buffer_Stream.Save.Handle :=
        Scheduler.Save_Stream (Module) ;

      Open_Succeeded : Boolean;

   begin

      Buffer_Stream.Save.Open_Writes
        ( Stream  => Save_Stream_Handle.all,
          Success => Open_Succeeded);

      -- Make sure the stream isn't in use
      if Open_Succeeded then

         Scheduler.Save
           (An_Instance => Module,
            To_Stream   => Save_Stream_Handle
            );

         Buffer_Stream.Save.Close_Writes (Save_Stream_Handle.all);

      end if;

   end Snapshot;

   -------------------------------------------------------------------------------
   -- Attempt to restore the state of the given module, if need be.
   -------------------------------------------------------------------------------
   procedure Restore (Module : in out Instance'Class) is

      Restore_Stream_Handle : constant Buffer_Stream.Restore.Handle :=
        Scheduler.Restore_Stream (Module);

      Open_Succeeded : Boolean;

   begin

      -- If there is replay data, see if it needs to be restored
      if Buffer_Stream.Restore.Data_Available (Restore_Stream_Handle) then

         Buffer_Stream.Restore.Open_Reads
           (Stream  => Restore_Stream_Handle.all,
            Success => Open_Succeeded
            );

         if Open_Succeeded then

            Scheduler.Restore
              (An_Instance => Module,
               From_Stream => Restore_Stream_Handle
               );

            Buffer_Stream.Restore.Close_Reads (Restore_Stream_Handle.all);

         end if;
      end if;
exception
      when Error : others =>
         Log.Report (Event => "Exception in Scheduler.Record_Playback.Restore: " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);
   end Restore;

   ----------------------------------------------------------------------------
   -- Reset the snapshot counter.
   ----------------------------------------------------------------------------
   procedure Reset is
   begin
      Cycles_Till_Snap := Cycles_Per_Snap;
      Log.Report ("Snapshot counter reset to" & Integer'Image (Cycles_Till_Snap));
   end Reset;

end Scheduler.Record_Playback;
