-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Characters.Latin_1;
with Buffer_Stream.Simulation;

-------------------------------------------------------------------------------
-- A scheduler module refers to a scheduler and also contains some extra
-- information that the executive scheduler uses. Each subsystem scheduler will
-- be given a module of a type based on the configuration information.
-------------------------------------------------------------------------------
private package Scheduler.Module is

   type Instance is private;
   type List is private;

   Unregistered_Module : exception;

   -------------------------------------------------------------------------------
   -- Create a new module with the given name and activity state.
   -------------------------------------------------------------------------------
   function Create
     (Name   : String;
      Active : Boolean
     ) return Instance;

   -------------------------------------------------------------------------------
   -- Add the given module to the given module list.
   -------------------------------------------------------------------------------
   procedure Enqueue
     (Module_List : in out List;
      New_Module  : in     Instance
     );

   -------------------------------------------------------------------------------
   -- Request a snapshot for all real-time modules in the given module list. This
   -- should cause Record_Playback.Snapshot to be called for each module before
   -- its next update.
   -------------------------------------------------------------------------------
   procedure Set_Snapshot_Requests (Module_List : in List);

   -------------------------------------------------------------------------------
   -- Update all real-time modules in the given module list.
   -------------------------------------------------------------------------------
   procedure Update_All  (Module_List  : in List;
                          Delta_Time   : in Float;
                          Total_Cycles : in Natural;
                          Enabled      : in Boolean);

   -------------------------------------------------------------------------------
   -- Initialize all the real-time modules in the given module list.
   -------------------------------------------------------------------------------
   procedure Initialize_All (Module_List : in List);

   -------------------------------------------------------------------------------
   -- Empty out all the replay buffers for all the schedulers in the given module
   -- list.
   -------------------------------------------------------------------------------
   procedure Flush_All_Replay_Buffers (Module_List : in List);

   -------------------------------------------------------------------------------
   -- Snapshot all the models in the given model list to the given stream.
   -------------------------------------------------------------------------------
   procedure Snapshot_All
     (Module_List  : in List;
      Destination  : in Buffer_Stream.Simulation.Handle
     );

   -------------------------------------------------------------------------------
   -- Return the sum of all the snapshot sizes for all schedulers in the band.
   -- Also added in is the size needed to store the module's name.
   -------------------------------------------------------------------------------
   function Snapshot_Size_Sum (Module_List : in List) return Natural;

private
   type Node_Type;
   type Node_Type_Ptr is access all Node_Type;

   type Instance is new Node_Type_Ptr;
   type List is record
      Head : Node_Type_Ptr;
      Tail : Node_Type_Ptr;
   end record;

end Scheduler.Module;

