-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;

with Log;
with Save_Restore;
with Subsystem_Scheduler;

with Scheduler.Configuration;
with Scheduler.IO;
with Scheduler.IOS_Communications;
with Scheduler.Record_Playback;
with Scheduler.Registrar;
with High_Resolution_Timer;

-------------------------------------------------------------------------------
-- A scheduler module refers to a scheduler and also contains some extra
-- information that the executive scheduler uses. Each subsystem scheduler will
-- be given a module of a type based on the configuration information.
-------------------------------------------------------------------------------
package body Scheduler.Module is

   type String_Ptr is access String;

   -- The name of a module that is *always* enabled, regardless of what the
   -- band thinks.
   Always_Enabled_Module_Name : constant String := "Jpats_HostComms_Scheduler";

   -- Define a type to contain linked lists of modules
   type Node_Type is record

      Module_Handle         : Scheduler.Handle;

      -- Displayable module data
      Name                  : String_Ptr;
      Active                : aliased Boolean;
      Execution_Time        : aliased Float := 0.0;

      -- Internal bookeeping data
      Exception_Deactivated : Boolean := False;
      Snap_Requested        : Boolean := False;
      Always_Enabled        : Boolean := False;

      -- Pointer to the next node in the linked list.
      Next      : Node_Type_Ptr;
   end record;

   -------------------------------------------------------------------------------
   -- Create a new module with the given name and activity state.
   -------------------------------------------------------------------------------
   function Create
     (Name   : String;
      Active : Boolean
     ) return Instance is
      New_Module : Instance := new Node_Type;
   begin

      New_Module.Name := new String'(Name);

      -- Find the registered module with the given name.
      New_Module.Module_Handle := Registrar.Retrieve_Handle (New_Module.Name.all);
      if New_Module.Module_Handle = null then
         Log.Report
           (Event    => "No module named " & New_Module.Name.all & " has been registered.",
            Severity => Log.Fatal
            );
         raise Unregistered_Module;
      end if;

      New_Module.Active := Active;
      if Name = Always_Enabled_Module_Name then
         New_Module.Always_Enabled := True;
      end if;

      IOS_Communications.Register_Module_Data
        (Real_Time_Module_Name => New_Module.Name.all,
         Active_Flag_Ptr       => New_Module.Active'access,
         Execution_Time_Ptr    => New_Module.Execution_Time'access
         );

      return New_Module;
   end Create;

   -------------------------------------------------------------------------------
   -- Add the given module to the given module list.
   -------------------------------------------------------------------------------
   procedure Enqueue
     (Module_List : in out List;
      New_Module  : in     Instance
     ) is
   begin
      -- Add a new module list node to the module list
      if Module_List.Head = null then
         Module_List.Tail := Node_Type_Ptr(New_Module);
         Module_List.Head := Module_List.Tail;
      else
         Module_List.Tail.Next := Node_Type_Ptr(New_Module);
         Module_List.Tail      := Module_List.Tail.Next;
      end if;
   end Enqueue;


   -------------------------------------------------------------------------------
   -- Perform an update on the given model, using the given delta time.
   -- If the Snapshot flag is set and one is taken, it will be reset.
   -------------------------------------------------------------------------------
   procedure Update (Module_Data  : in out Node_Type;
                     Delta_Time   : in     Float;
                     Total_Cycles : in     Natural
                    ) is

      Module : Scheduler.Instance'Class renames Module_Data.Module_Handle.all;
      Timer  : High_Resolution_Timer.Instance;
   begin

      High_Resolution_Timer.Start (Timer);
      if Module in IO.Instance'Class then

         if Save_Restore.Replay_In_Progress then
            -- If we are doing a replay, restore the inputs (unless the replay is
            -- paused), and call Update_In_Freeze.

            if not IOS_Communications.Host_Update_In_Freeze_Active then
               Record_Playback.Restore (Module);
            end if;
            Subsystem_Scheduler.Update_In_Freeze (Module);
         else
            -- Not replay. Perform and update and snapshot the inputs.
            Subsystem_Scheduler.Update( An_Instance          => Module,
                                        Integration_Constant => Delta_Time );
            Record_Playback.Snapshot (Module);
         end if;

      else

         -- Perform record/playback processing
         if Module_Data.Snap_Requested then
            Record_Playback.Snapshot (Module);
            Module_Data.Snap_Requested := False;
         end if;
         Record_Playback.Restore (Module);

         -- Perform the update
         if IOS_Communications.Host_Update_In_Freeze_Active then
            Subsystem_Scheduler.Update_In_Freeze (Module);
         else
            Subsystem_Scheduler.Update (An_Instance          => Module,
                                        Integration_Constant => Delta_Time );
         end if;
      end if;

      -- Update some of the model's statistics
      High_Resolution_Timer.Stop (Timer);

      Module_Data.Execution_Time := High_Resolution_Timer.Milliseconds (Timer);

   exception

      when Error : others =>

         Module_Data.Active := False;

         Module_Data.Exception_Deactivated := True;

         Log.Report (Event    => Ada.Exceptions.Exception_Information (Error)
                     & Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF
                     & "Module_Name deactivated in Update is "
                     & Module_Data.Name.all,
                     Severity => Log.Error);
   end Update;

   -------------------------------------------------------------------------------
   -- Request a snapshot for all real-time modules in the given module list. This
   -- should cause Record_Playback.Snapshot to be called for each module before
   -- its next update.
   -------------------------------------------------------------------------------
   procedure Set_Snapshot_Requests (Module_List : in List) is
      Next_Module : Node_Type_Ptr := Module_List.Head;
   begin

      -- Set all the Snap_Requested flags on all active real-time modules
      while Next_Module /= null loop
         Next_Module.Snap_Requested := True;
         Next_Module := Next_Module.Next;
      end loop;

   end Set_Snapshot_Requests;

   -------------------------------------------------------------------------------
   -- Update all real-time modules in the given module list.
   -------------------------------------------------------------------------------
   procedure Update_All  (Module_List  : in List;
                          Delta_Time   : in Float;
                          Total_Cycles : in Natural;
                          Enabled      : in Boolean
                         ) is

      -- Used to iterate through the module list
      Module_Node : Node_Type_Ptr := Module_List.Head;

   begin

      while Module_Node /= null loop

         if Module_Node.Active and (Enabled or Module_Node.Always_Enabled) then

            -- Restart the model if requested to.
            if Module_Node.Exception_Deactivated then

               Module_Node.Exception_Deactivated := False;

               Subsystem_Scheduler.Re_Initialize
                 (Module_Node.Module_Handle.all);

            end if;

            -- Perform an update
            Update (Module_Data  => Module_Node.all,
                    Delta_Time   => Delta_Time,
                    Total_Cycles => Total_Cycles
                    );

         else
            Module_Node.Execution_Time := 0.0;
         end if;

         Module_Node := Module_Node.Next;

      end loop;

   exception
      when Error : others =>
         Log.Report (Event    => Ada.Exceptions.Exception_Information(Error) &
                     "Module_Name is " & Module_Node.Name.all,
                     Severity => Log.Error);
      raise;

   end Update_All;


   ----------------------------------------------------------------------------
   -- Initialize all the real-time modules.
   ----------------------------------------------------------------------------
   procedure Initialize_All (Module_List : in List) is
      Next_Module : Node_Type_Ptr := Module_List.Head;
   begin

      while Next_Module /= null loop

         Log.Report ("Initializing : " & Next_Module.Name.all);

         Initialize (Next_Module.Module_Handle.all);
         Create_Streams (Next_Module.Module_Handle);

         Next_Module := Next_Module.Next;
      end loop;

   exception
      when others =>
         Log.Report (Event    => "Scheduler.Module.Initialize_All",
                     Severity => Log.Error);
      raise;

   end Initialize_All;

   -------------------------------------------------------------------------------
   -- Empty out all the replay buffers for all the schedulers in the given module
   -- list.
   -------------------------------------------------------------------------------
   procedure Flush_All_Replay_Buffers (Module_List : in List) is
      -- Used to iterate through the module list
      Module_Node : Node_Type_Ptr := Module_List.Head;

      Restore_Stream : Buffer_Stream.Restore.Handle;
   begin

      while Module_Node /= null loop

         Restore_Stream := Scheduler.Restore_Stream (Module_Node.Module_Handle.all);

         Buffer_Stream.Restore.Open_Writes (Restore_Stream.all);
         Buffer_Stream.Restore.Flush (Restore_Stream.all);
         Buffer_Stream.Restore.Close_Writes (Restore_Stream.all);

         Module_Node := Module_Node.Next;
      end loop;

   end Flush_All_Replay_Buffers;

   -------------------------------------------------------------------------------
   -- Snapshot all the models in the given model list to the given stream.
   -------------------------------------------------------------------------------
   procedure Snapshot_All
     (Module_List  : in List;
      Destination  : in Buffer_Stream.Simulation.Handle
     ) is
      Module_Node : Node_Type_Ptr := Module_List.Head;
   begin
      while Module_Node /= null loop

         String'Output (Destination, Scheduler.Registrar.Retrieve_Name
                        (Module_Node.Module_Handle)
                        );

         Scheduler.Save
           (An_Instance => Module_Node.Module_Handle.all,
            To_Stream   => Destination
            );

         Module_Node := Module_Node.Next;
      end loop;
   end Snapshot_All;

   -------------------------------------------------------------------------------
   -- Return the sum of all the snapshot sizes for all schedulers in the band.
   -------------------------------------------------------------------------------
   function Snapshot_Size_Sum (Module_List : in List) return Natural is

      Module_Node : Node_Type_Ptr := Module_List.Head;

      Sum : Natural := 0;
   begin
      while Module_Node /= null loop

         -- Add in the model's snapshot size, the size of the module's name,
         -- and a length field.
         Sum :=
           Sum + Scheduler.Snapshot_Size (Module_Node.Module_Handle.all) +
           Scheduler.Registrar.Retrieve_Name (Module_Node.Module_Handle)'Length +
           (Integer'Size + 7 / 8);

         Module_Node := Module_Node.Next;
      end loop;

      return Sum;
   end Snapshot_Size_Sum;

end Scheduler.Module;
