-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Real_Time;
with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;

with JPATS_IOS_Interface;
with Log;
with Stethoscope;

with Scheduler.Band;

use type Ada.Real_Time.Time;

-------------------------------------------------------------------------------
-- This package provides for communicating scheduler information to the IOS.
-- This is almost entirely for debuging purposes.
-------------------------------------------------------------------------------
package body Scheduler.Ios_Communications is

   ----------------------------------------------------------------------------
   -- Adjustable constants

   -- The number of times per second that stethoscope will collect data
   Stethoscope_Collection_Frequency : constant := 15.0;

   ----------------------------------------------------------------------------
   -- Currently unused IOS variables
   --

   -- True when the real-time sim has been commanded to start.
   Start_Simulation : Boolean := False;

   ----------------------------------------------------------------------------
   -- Internal data
   --

   Nano_Seconds_Per_Second     : constant := 1_000_000_000.0;

   -- The amount of time between stethoscope collections
   Steth_Comm_Interval         : constant Ada.Real_Time.Time_Span :=
     Ada.Real_Time.Nanoseconds
     (Integer ( (1.0 / Stethoscope_Collection_Frequency) * Nano_Seconds_Per_Second ) );

   -- IDs for all the real_time_modules that the IOS knows about. The identifier needs
   -- to match the name of the scheduler object, with any '.' characters changed to '_'s.
   -- This will also be the name used in the IOS interface csv file.
   type Module_Types is
      (JPATS_Aircraft_Body_Scheduler,
       JPATS_Atmosphere_Free_Air_Scheduler,
       JPATS_Atmosphere_Microburst_Scheduler,
       JPATS_Atmosphere_Winds_Scheduler,
       JPATS_Auto_Test_Scheduler,
       JPATS_Avionics_Scheduler,
       JPATS_Avionics_RMU_Scheduler,
       JPATS_Dcls_Scheduler,
       JPATS_Electrical_Scheduler,
       JPATS_Environmental_Scheduler,
       JPATS_Flight_Instruments_Scheduler,
       JPATS_Flight_Instruments_AOA_System_Scheduler,
       JPATS_Formation_Manager_Scheduler,
       JPATS_Formation_Profile_Playback_Scheduler,
       JPATS_Formation_Profile_Recorder_Scheduler,
       JPATS_Fuel_Scheduler,
       JPATS_HostComms_Scheduler,
       JPATS_Hydraulics_Scheduler,
       JPATS_Landing_Gear_Scheduler,
       JPATS_Lights_Scheduler,
       JPATS_Nav_Instruments_Scheduler,
       JPATS_Powerplant_Scheduler,
       JPATS_Propeller_Scheduler,
       JPATS_Radio_Scheduler,
       JPATS_Radio_DB_IF_Scheduler,
       JPATS_Reposition_Scheduler,
       JPATS_Secondary_Flight_Controls_Scheduler,
       JPATS_Secondary_Flight_Controls_Rudder_Trim_Scheduler,
       JPATS_Simulated_Aircraft_Scheduler,
       JPATS_Visual_Airport_Scheduler,
       JPATS_Visual_Buffer_Fast_Data_Scheduler,
       JPATS_Visual_Buffer_Slow_data_Scheduler,
       JPATS_Visual_Models_Scheduler,
       JPATS_Visual_Weather_Scheduler,
       JPATS_Warning_Scheduler,
       JPATS_Engine_Data_Manager_Scheduler,
       JPATS_IO_Scheduler,
       JPATS_Simulated_Aircraft_Mass_Property_Scheduler,
       JPATS_Simphonics_Scheduler,
       JPATS_Scu_Scheduler
       );


   type Bool_Ptr is access all Boolean;
   type String_Ptr is access all String;

   type Register_Active_Var_Proc_Ptr is access procedure;

   type Module_Record is record
      Set_Variable_Routine_Ptr : JPATS_IOS_Interface.Set_Variable_Proc_Ptr_Type;
      Variable_Ptr             : Bool_Ptr;
   end record;

   procedure Set_Aircraft_Body_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Atmosphere_Free_Air_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Atmosphere_Microburst_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Atmosphere_Winds_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Auto_Test_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Avionics_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Avionics_RMU_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Electrical_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Environmental_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Flight_Instruments_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Flight_Instruments_AOA_System_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Formation_Manager_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Formation_Profile_Playback_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Formation_Profile_Recorder_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Fuel_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Dcls_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Hydraulics_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_HostComms_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_IO_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Landing_Gear_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Lights_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Nav_Instruments_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Powerplant_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Propeller_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Radio_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Radio_DB_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Reposition_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Scndry_Flght_Cntrls_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Scndry_Flght_Cntrls_Rudder_Trim_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Simulated_Aircraft_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Vsl_Arprt_Fast_Data_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Visual_Airport_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Vsl_Arprt_Slow_Data_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Visual_Models_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Visual_Weather_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Engine_Data_Manager_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Warning_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Mass_Property_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Simphonics_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );
   procedure Set_Scu_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value );


   type Module_Record_Array is array (Module_Types) of Module_Record;

   Module_Data : Module_Record_Array  :=
     (JPATS_Aircraft_Body_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Aircraft_Body_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Atmosphere_Free_Air_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Atmosphere_Free_Air_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Atmosphere_Microburst_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Atmosphere_Microburst_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Atmosphere_Winds_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Atmosphere_Winds_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Auto_Test_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Auto_Test_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Avionics_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Avionics_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Avionics_RMU_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Avionics_RMU_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Dcls_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Dcls_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Electrical_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Electrical_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Environmental_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Environmental_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Flight_Instruments_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Flight_Instruments_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Flight_Instruments_AOA_System_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Flight_Instruments_AOA_System_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Formation_Manager_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Formation_Manager_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Formation_Profile_Playback_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Formation_Profile_Playback_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Formation_Profile_Recorder_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Formation_Profile_Recorder_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Fuel_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Fuel_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_HostComms_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_HostComms_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Hydraulics_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Hydraulics_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Landing_Gear_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Landing_Gear_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Lights_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Lights_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Nav_Instruments_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Nav_Instruments_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Powerplant_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Powerplant_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Propeller_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Propeller_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Radio_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Radio_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Radio_DB_IF_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Radio_DB_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Reposition_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Reposition_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Secondary_Flight_Controls_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Scndry_Flght_Cntrls_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Secondary_Flight_Controls_Rudder_Trim_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Scndry_Flght_Cntrls_Rudder_Trim_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Simulated_Aircraft_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Simulated_Aircraft_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Visual_Airport_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Visual_Airport_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Visual_Buffer_Fast_Data_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Vsl_Arprt_Fast_Data_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Visual_Buffer_Slow_data_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Vsl_Arprt_Slow_Data_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Visual_Models_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Visual_Models_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Visual_Weather_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Visual_Weather_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Warning_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Warning_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Engine_Data_Manager_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Engine_Data_Manager_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_IO_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_IO_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Simulated_Aircraft_Mass_Property_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Mass_Property_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Simphonics_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Simphonics_Scheduler_Active'access,
         Variable_Ptr             => null
         ),

      JPATS_Scu_Scheduler =>
        (Set_Variable_Routine_Ptr => Set_Scu_Scheduler_Active'access,
         Variable_Ptr             => null
         )

     );



   type IOS_Thread_Types is
       (IO_Thread_Type,
        T60_Hz_Thread_Type,
        T30_Hz_Thread_Type,
        T15_Hz_Thread_Type,
        T5_Hz_Thread_Type,
        T2_Hz_Thread_Type
        );



   type IOS_Thread_Record is record
      Thread_Name        : String_Ptr;
      Set_Active_Routine : JPATS_IOS_Interface.Set_Variable_Proc_Ptr_Type;
      Active_Flag_Ptr    : Bool_Ptr;
   end record;

   procedure Set_IO_Thread_Active    (pData : JPATS_IOS_Interface.Set_Value);
   procedure Set_60_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value);
   procedure Set_30_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value);
   procedure Set_15_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value);
   procedure Set_5_Hz_Thread_Active  (pData : JPATS_IOS_Interface.Set_Value);
   procedure Set_2_Hz_Thread_Active  (pData : JPATS_IOS_Interface.Set_Value);

   Thread_Active_Suffix  : constant String := "_Thread_Active";
   Execution_Time_Suffix : constant String := "_Thread_Execution_Time";

   type IOS_Thread_Record_Array is array ( IOS_Thread_Types ) of IOS_Thread_Record;

   IOS_Thread_Records : IOS_Thread_Record_Array  :=
     (IO_Thread_Type     =>
        (Thread_Name        => new String'("IO"),
         Set_Active_Routine => Set_IO_Thread_Active'access,
         Active_Flag_Ptr    => null
         ),
      T60_Hz_Thread_Type =>
        (Thread_Name        => new String'("60Hz0"),
         Set_Active_Routine => Set_60_Hz_Thread_Active'Access,
         Active_Flag_Ptr    => null
         ),
      T30_Hz_Thread_Type =>
        (Thread_Name        => new String'("30Hz0"),
         Set_Active_Routine => Set_30_Hz_Thread_Active'access,
         Active_Flag_Ptr    => null
         ),
      T15_Hz_Thread_Type =>
        (Thread_Name        => new String'("15Hz0"),
         Set_Active_Routine => Set_15_Hz_Thread_Active'access,
         Active_Flag_Ptr    => null
         ),
      T5_Hz_Thread_Type  =>
        (Thread_Name        => new String'("5Hz0"),
         Set_Active_Routine => Set_5_Hz_Thread_Active'access,
         Active_Flag_Ptr    => null
         ),
      T2_Hz_Thread_Type  =>
        (Thread_Name        => new String'("2Hz0"),
         Set_Active_Routine => Set_2_Hz_Thread_Active'access,
         Active_Flag_Ptr    => null
         )
      );

   ----------------------------------------------------------------------------
   -- IOS Set methods
   --

   procedure Set_Aircraft_Body_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Aircraft_Body_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Aircraft_Body_Scheduler_Active;

   procedure Set_Atmosphere_Free_Air_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Atmosphere_Free_Air_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Atmosphere_Free_Air_Scheduler_Active;

   procedure Set_Atmosphere_Microburst_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Atmosphere_Microburst_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Atmosphere_Microburst_Scheduler_Active;

   procedure Set_Atmosphere_Winds_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Atmosphere_Winds_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Atmosphere_Winds_Scheduler_Active;

   procedure Set_Auto_Test_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Auto_Test_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Auto_Test_Scheduler_Active;

   procedure Set_Avionics_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Avionics_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Avionics_Scheduler_Active;

   procedure Set_Avionics_RMU_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Avionics_RMU_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Avionics_RMU_Scheduler_Active;

   procedure Set_Dcls_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Dcls_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Dcls_Scheduler_Active;

   procedure Set_Electrical_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Electrical_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Electrical_Scheduler_Active;

   procedure Set_Environmental_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Environmental_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Environmental_Scheduler_Active;

   procedure Set_Flight_Instruments_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Flight_Instruments_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Flight_Instruments_Scheduler_Active;

   procedure Set_Flight_Instruments_AOA_System_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Flight_Instruments_AOA_System_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Flight_Instruments_AOA_System_Scheduler_Active;

   procedure Set_Formation_Manager_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Formation_Manager_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Formation_Manager_Scheduler_Active;

   procedure Set_Formation_Profile_Playback_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Formation_Profile_Playback_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Formation_Profile_Playback_Scheduler_Active;

   procedure Set_Formation_Profile_Recorder_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Formation_Profile_Recorder_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Formation_Profile_Recorder_Scheduler_Active;

   procedure Set_Fuel_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Fuel_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Fuel_Scheduler_Active;

   procedure Set_HostComms_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_HostComms_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_HostComms_Scheduler_Active;

   procedure Set_Hydraulics_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Hydraulics_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Hydraulics_Scheduler_Active;

   procedure Set_Landing_Gear_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Landing_Gear_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Landing_Gear_Scheduler_Active;

   procedure Set_Lights_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Lights_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Lights_Scheduler_Active;

   procedure Set_Nav_Instruments_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Nav_Instruments_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Nav_Instruments_Scheduler_Active;

   procedure Set_Powerplant_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Powerplant_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Powerplant_Scheduler_Active;

   procedure Set_Propeller_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Propeller_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Propeller_Scheduler_Active;

   procedure Set_Radio_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Radio_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Radio_Scheduler_Active;

   procedure Set_Radio_DB_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Radio_DB_IF_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Radio_DB_Scheduler_Active;

   procedure Set_Reposition_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Reposition_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Reposition_Scheduler_Active;

   procedure Set_Scndry_Flght_Cntrls_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Secondary_Flight_Controls_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Scndry_Flght_Cntrls_Scheduler_Active;

   procedure Set_Scndry_Flght_Cntrls_Rudder_Trim_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Secondary_Flight_Controls_Rudder_Trim_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Scndry_Flght_Cntrls_Rudder_Trim_Scheduler_Active;

   procedure Set_Simulated_Aircraft_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Simulated_Aircraft_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Simulated_Aircraft_Scheduler_Active;

   procedure Set_Visual_Airport_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Visual_Airport_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Visual_Airport_Scheduler_Active;

   procedure Set_Vsl_Arprt_Fast_Data_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Visual_Buffer_Fast_Data_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Vsl_Arprt_Fast_Data_Scheduler_Active;

   procedure Set_Vsl_Arprt_Slow_Data_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Visual_Buffer_Slow_data_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Vsl_Arprt_Slow_Data_Scheduler_Active;

   procedure Set_Visual_Models_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Visual_Models_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Visual_Models_Scheduler_Active;

   procedure Set_Visual_Weather_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Visual_Weather_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Visual_Weather_Scheduler_Active;

   procedure Set_Warning_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Warning_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Warning_Scheduler_Active;

   procedure Set_Engine_Data_Manager_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Engine_Data_Manager_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Engine_Data_Manager_Scheduler_Active;

   procedure Set_IO_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_IO_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_IO_Scheduler_Active;

   procedure Set_Mass_Property_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Simulated_Aircraft_Mass_Property_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Mass_Property_Scheduler_Active;

   procedure Set_Simphonics_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Simphonics_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Simphonics_Scheduler_Active;

   procedure Set_Scu_Scheduler_Active( pData : JPATS_IOS_Interface.Set_Value ) is

   begin

      Module_Data(JPATS_Scu_Scheduler).Variable_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool( pData );

   end Set_Scu_Scheduler_Active;




   -- Frequency band (thread) requests.

   procedure Set_IO_Thread_Active (pData : JPATS_IOS_Interface.Set_Value) is
   begin
      IOS_Thread_Records (IO_Thread_Type).Active_Flag_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_IO_Thread_Active;

   procedure Set_60_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value) is
   begin
      Ios_Thread_Records (T60_Hz_Thread_Type).Active_Flag_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_60_Hz_Thread_Active;

   procedure Set_30_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value) is
   begin
      Ios_Thread_Records (T30_Hz_Thread_Type).Active_Flag_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_30_Hz_Thread_Active;

   procedure Set_15_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value) is
   begin
      Ios_Thread_Records (T15_Hz_Thread_Type).Active_Flag_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_15_Hz_Thread_Active;

   procedure Set_5_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value) is
   begin
      Ios_Thread_Records (T5_Hz_Thread_Type).Active_Flag_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_5_Hz_Thread_Active;

   procedure Set_2_Hz_Thread_Active (pData : JPATS_IOS_Interface.Set_Value) is
   begin
      Ios_Thread_Records (T2_Hz_Thread_Type).Active_Flag_Ptr.all
         := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_2_Hz_Thread_Active;

   -- General scheduler requests

   procedure Set_Start_Simulation( pData : JPATS_IOS_Interface.Set_Value ) is
   begin
      Start_Simulation := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_Start_Simulation;

   procedure Set_Host_Update_In_Freeze_Active( pData : JPATS_IOS_Interface.Set_Value ) is
   begin
      Host_Update_In_Freeze_Active := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_Host_Update_In_Freeze_Active;

   procedure Set_Reboot_Request ( pData : JPATS_IOS_Interface.Set_Value ) is
   begin
      Reboot_Request := JPATS_IOS_Interface.Convert_To_Bool(pData);
   end Set_Reboot_Request;

   ----------------------------------------------------------------------------
   -- Internal helper routines.
   --


   ----------------------------------------------------------------------------
   -- Internal tasks
   --

   ----------------------------------------------------------------------------
   -- The stethoscope communications thread
   ----------------------------------------------------------------------------
   task Stethoscope_Communications is
      entry Start;
      entry Stop;
   end Stethoscope_Communications;

   task body Stethoscope_Communications is
      Next_Time : Ada.Real_Time.Time;
   begin

      accept Start;

      Band.Set_Task_Priority (1);

      Next_Time := Ada.Real_Time.Clock + Steth_Comm_Interval;

      loop

         begin
            Stethoscope.Collect_Signals;
         exception
            when Error : others =>
               Log.Report (Event    => Ada.Exceptions.Exception_Information(Error) &
                           Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
                           "Stethoscope Collect Signals Error",
                           Severity => Log.Error);
         end;

         delay until Next_Time;

         Next_Time := Next_Time + Steth_Comm_Interval;

      end loop;

      accept Stop;

   end Stethoscope_Communications;

   procedure Register_Scheduler_Variables
     (Module_Id          : in     Module_Types;
      Active_Flag_Ptr    : access Boolean;
      Execution_Time_Ptr : access Float
     ) is

      Active_Name : constant String := Module_Types'Image(Module_Id);

      -- Execution time is the same with _Execution_Time instead of _Scheduler
      Exec_Time_Name : constant String :=
        Active_Name (1..
                     Ada.Strings.Fixed.Index
                     (Source  => Active_Name,
                      Pattern => "_SCHEDULER"
                      )
                     ) &
        "Execution_Time";
   begin

      Log.Report ("Registering " & Active_Name & " and " & Exec_Time_Name);
      JPATS_IOS_Interface.Register
         ( Name        => Active_Name,
           Variable    => Active_Flag_Ptr.all'Address,
           Set_Routine => Module_Data (Module_ID).Set_Variable_Routine_Ptr
           );

      JPATS_IOS_Interface.Register
         ( Name        => Exec_Time_Name,
           Variable    => Execution_Time_Ptr.all'Address
           );

   end Register_Scheduler_Variables;

   procedure Register_Thread_Variables
     (Thread_Type        : in     IOS_Thread_Types;
      Execution_Time_Ptr : access Float
     ) is

   begin

      JPATS_IOS_Interface.Register
         (Name        => IOS_Thread_Records (Thread_Type).Thread_Name.all & Thread_Active_Suffix,
          Variable    => IOS_Thread_Records (Thread_Type).Active_Flag_Ptr.all'Address,
          Set_Routine => Ios_Thread_Records (Thread_Type).Set_Active_Routine
          );

      JPATS_IOS_Interface.Register
         (Name     => IOS_Thread_Records (Thread_Type).Thread_Name.all & Execution_Time_Suffix,
          Variable => Execution_Time_Ptr.all'Address
          );

   end Register_Thread_Variables;

   ----------------------------------------------------------------------------
   -- Set up scheduler debugging.
   ----------------------------------------------------------------------------
   procedure Initialize is
   begin
      Stethoscope_Communications.Start;

      -- Register all the IOS/Stethoscope variables.
      JPATS_IOS_Interface.Register (Name        => "Start_Simulation",
                                    Variable    => Start_Simulation'address,
                                    Set_Routine => Set_Start_Simulation'access
                                    );

      JPATS_IOS_Interface.Register (Name        => "Cycle_Count",
                                    Variable    => Total_Number_Of_Cycles'address
                                    );

      JPATS_IOS_Interface.Register (Name        => "Host_Update_In_Freeze_Active",
                                    Variable    => Host_Update_In_Freeze_Active'address,
                                    Set_Routine => Set_Host_Update_In_Freeze_Active'access
                                    );
      JPATS_IOS_Interface.Register (Name        => "Reboot_Request",
                                    Variable    => Reboot_Request'address,
                                    Set_Routine => Set_Reboot_Request'access
                                    );

   end Initialize;

   ----------------------------------------------------------------------------
   -- Register the IOS variables for the given thread.
   ----------------------------------------------------------------------------
   procedure Register_Band_Data
     (Band_Name          : in     String;
      Active_Flag_Ptr    : access Boolean;
      Execution_Time_Ptr : access Float
     ) is
   begin

      for IOS_Thread_Type in IOS_Thread_Types loop
         if
           Ada.Strings.Fixed.Translate
           (Ios_Thread_Records (IOS_Thread_Type).Thread_Name.all,
            Ada.Strings.Maps.Constants.Lower_Case_Map
            ) =
           Ada.Strings.Fixed.Translate (Band_Name, Ada.Strings.Maps.Constants.Lower_Case_Map)
         then

            IOS_Thread_Records (IOS_Thread_Type).Active_Flag_Ptr := Bool_Ptr(Active_Flag_Ptr);
            Register_Thread_Variables
              (Thread_Type        => IOS_Thread_Type,
               Execution_Time_Ptr => Execution_Time_Ptr
               );

            -- We've found it. We can quit now.
            return;
         end if;
      end loop;

      Log.Report
        (Event    => "Unable to find any thread data in scheduler-ios_communications.adb for " & Band_Name,
         Severity => Log.Warning
         );
   end Register_Band_Data;

   ----------------------------------------------------------------------------
   -- Register the IOS variables for the named module.
   ----------------------------------------------------------------------------
   procedure Register_Module_Data
     (Real_Time_Module_Name : in     String;
      Active_Flag_Ptr       : access Boolean;
      Execution_Time_Ptr    : access Float
     ) is

      Module_Id : Module_Types;
   begin

      -- Find the module type of the given module name
      Module_Id := Module_Types'Value
        (Ada.Strings.Fixed.Translate
         (Source  => Real_Time_Module_Name,
          Mapping => Ada.Strings.Maps.To_Mapping (From => ".", To => "_")
          )
         );

      -- Register its variables
      Register_Scheduler_Variables
        (Module_ID          => Module_ID,
         Active_Flag_Ptr    => Active_Flag_Ptr,
         Execution_Time_Ptr => Execution_Time_Ptr
         );

      Module_Data (Module_ID).Variable_Ptr := Bool_Ptr(Active_Flag_Ptr);

   exception
      when Constraint_Error =>
         Log.Report
           (Event    => "Module " & Real_Time_Module_Name & " has no IOS data variables defined.",
            Severity => Log.Warning
            );
   end Register_Module_Data;

end Scheduler.Ios_Communications;
