-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

-- Ada library packages
with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Real_Time;
with Ada.Text_IO;
with Interfaces.C;

-- JPATS packages
with Buffer_Stream.Simulation;
with Log;
with Mission_Time;
with Save_Restore.IOS_Interface;

-- Packages in the Scheduler Hierarchy
with Scheduler.Band;
with Scheduler.Configuration;
with Scheduler.IOS_Communications;
with Scheduler.Record_Playback;
with Scheduler.Snapshot;

use type Ada.Real_Time.Time_Span;
use type Interfaces.C.Int;
use type Interfaces.C.Long;

----------------------------------------------------------------------------
-- This package implements the simulation's real-time executive scheduler.
-- Once started, it will initialize all registered real-time schedulers and
-- thereafter ensure they update at the configured rates.
----------------------------------------------------------------------------
package body Scheduler.Executive is

   ----------------------------------------------------------------------------
   -- Adjustable constants

   -- The priority of the simulation clock task relative to the simulation bands.
   Sim_Clock_Relative_Priority : constant := 20;

   -- The amount of clock iterations to wait before performing a reboot
   Reboot_Delay : constant := 40;

   ----------------------------------------------------------------------------
   -- Internal data
   --

   Nano_Seconds_Per_Second : constant := 1_000_000_000.0;

   type Abort_Id_Type is (Task_Cant_Resume, Unsuccessful_Processing_Of_Config_Data);

   -- True when the simulation is aborted.
   Simulation_Is_Aborted : Boolean := false;

   Interval : Ada.Real_Time.Time_Span;

   Clock_Rate_Set_Failed       : exception;
   Clock_Resolution_Set_Failed : exception;

   -- Indicates whether saving of model data is enabled or disabled
   Model_Saving : Boolean := True;

   -- Indicates the number of iterations since a reboot request
   Reboot_Counter : Natural := 0;

   Cycle_Count : Natural := 0;

   -- Flag for signalling scheduler to terminate a freeze on a certian cycle.
   No_Thaw : constant := -1;
   Scheduled_Thaw : Integer := No_Thaw;

   ----------------------------------------------------------------------------
   -- Internal helper routines.
   --

   -- Interface to the vxWorks routine to set the clock rate.
   function sysClkRateSet (ticksPerSecond : in Integer) return Integer;
   pragma Import(C, sysClkRateSet, "sysClkRateSet" );

   type Time_Spec is record
      Seconds     : Interfaces.C.Unsigned_Long;
      Nanoseconds : Interfaces.C.Long;
   end record;
   for Time_Spec use record
      Seconds     at 0 range 0..31;
      Nanoseconds at 4 range 0..31;
   end record;

   function Clock_SetRes (Clock_Id   : Interfaces.C.Int := 0; -- Must always be 0
                          Resolution : Time_Spec
                         ) return Interfaces.C.Int;
   pragma Import (C, Clock_SetRes, "clock_setres");


   type Reboot_Start_Type is mod 2**Interfaces.C.Int'Size;
   for Reboot_Start_Type'Size use Interfaces.C.Int'Size;

   Normal         : constant Reboot_Start_Type := 0;
   No_Autoboot    : constant Reboot_Start_Type := 1;
   Clear          : constant Reboot_Start_Type := 2;
   Quick_Autoboot : constant Reboot_Start_Type := 4;

   procedure Reboot (Start_Type : in Reboot_Start_Type := Normal);
   pragma Import (C, Reboot, "reboot");


   -- This routine is one of ours, written by Mike Bates. It is located in
   -- dosfsdt.c in the kernel configuration directory.
   procedure Set_System_Time_To_CMOS;
   pragma Import (C, Set_System_Time_To_CMOS, "setSystemTimeToCmos");

   ----------------------------------------------------------------------------
   -- Process an event at the system clock interval.
   ----------------------------------------------------------------------------
   procedure Process_Timer_Event (Current_Timer_Cycle : in Natural) is
   begin

      --
      -- Keep an accumulator for the total number of cycles that the sim has executed.
      --
      IOS_Communications.Total_Number_Of_Cycles := IOS_Communications.Total_Number_Of_Cycles + 1;

      -- Reboot if requested to do so. The delay gives the IOS a chance to see
      -- their button press.
      if IOS_Communications.Reboot_Request then
         if Reboot_Counter < Reboot_Delay then
            Reboot_Counter := Reboot_Counter + 1;
         else
            Reboot (Quick_Autoboot or Clear);
         end if;
      end if;

      -- Check to see if it is time to terminate a freeze
      if Scheduled_Thaw = Current_Timer_Cycle then
         IOS_Communications.Host_Update_In_Freeze_Active := False;
         Scheduled_Thaw := No_Thaw;
      end if;

      -- Check to see if its time to snapshot all the models yet. If so, they will all be
      -- marked as in need of snapshot.
      if Model_Saving and then Record_Playback.Model_Snapshot_Time (Current_Timer_Cycle) then
         Band.Set_Module_Snapshot_Requests;
      end if;

      -- Update all the bands
      Band.Update_All
        (Total_Cycles => IOS_Communications.Total_Number_Of_Cycles,
         Subframe     => Current_Timer_Cycle
         );

   exception
      when others =>
         Log.Report (Event    => "Scheduler-Executive.adb: Process_Event_Timer",
                     Severity => Log.Error);
      raise;

   end Process_Timer_Event;

   ----------------------------------------------------------------------------
   -- Attempt to halt the simulation.
   ----------------------------------------------------------------------------
   procedure Abort_Simulation( Abort_Id : in Abort_Id_Type ) is

   begin

      case Abort_Id is

         when Task_Cant_Resume =>

           Ada.Text_IO.Put_Line( "Can't Resume Task - Aborting" );

         when Unsuccessful_Processing_Of_Config_Data =>

           Ada.Text_IO.Put_Line( "Config Data or File Error - Aborting" );

      end case;

      Simulation_Is_Aborted := true;

   exception
      when others =>
         Log.Report (Event    => "Scheduler-Executive.adb: Abort_Simulation",
                     Severity => Log.Fatal);
      raise;

   end Abort_Simulation;


   procedure Set_Real_Time_Clock
     (System_Clock_Rate : in Integer;
      Timer_Period      : in Float
     ) is

      NS_Period : constant Float := Timer_Period * Nano_Seconds_Per_Second;

      Resolution : Time_Spec :=
        (Seconds     => Interfaces.C.Unsigned_Long(Timer_Period),
         Nanoseconds => Interfaces.C.Long(Ns_Period) mod Interfaces.C.Long(Nano_Seconds_Per_Second)
         );
   begin

      -- Set the CPU clock frequency
      if sysClkRateSet( ticksPerSecond => System_Clock_Rate ) /= 0 then
         raise Clock_Rate_Set_Failed;
      end if;

      -- Reset the TOD clock resolution to match the new frequency
      if Clock_SetRes (Resolution => Resolution) /= 0 then
         raise Clock_Resolution_Set_Failed;
      end if;

      -- Reset the system time from the CMOS clock
      Set_System_Time_To_CMOS;

      -- Calculate the iteration interval. The smallest possible amount is subtracted off
      -- to prevent forward slippage due to rounding. The backward slippage will be
      -- compensated for periodicly.
      Interval := Ada.Real_Time.Nanoseconds (Integer(NS_Period)) - Ada.Real_Time.Time_Span_Unit;
      Log.Report ("Interval = " & Integer'Image(Integer(Ns_Period)) & "ns");
      Log.Report ("Interval = " &
                  Duration'Image(Ada.Real_Time.To_Duration(Interval)) & "s");

   end Set_Real_Time_Clock;

   ----------------------------------------------------------------------------
   -- Internal tasks
   --

   ----------------------------------------------------------------------------
   -- The simulation clock tick thread.
   ----------------------------------------------------------------------------
   task Simulation_Clock is
      entry Start;
   end Simulation_Clock;

   task body Simulation_Clock is
      Next_Time   : Ada.Real_Time.Time;
      Frame_Start : Ada.Real_Time.Time;
   begin

      accept Start;

      Band.Set_Task_Priority (Sim_Clock_Relative_Priority);

      Frame_Start := Ada.Real_Time.Clock;
      Next_Time   := Frame_Start + Interval;

      loop
         Process_Timer_Event (Cycle_Count);
         delay until Next_Time;

         Cycle_Count := Cycle_Count + 1;

         if Cycle_Count >= Configuration.Cycles_Per_Frame then
            Cycle_Count := 0;

            -- Adjust the Next_Time for accumulating roundoff errors.
            Frame_Start := Frame_Start + Ada.Real_Time.Nanoseconds (Integer(Nano_Seconds_Per_Second));
            Next_Time   := Frame_Start;
         else
            Next_Time := Next_Time + Interval;
         end if;
      end loop;


   exception
      when Error : others =>
         Log.Report (Event    => "Scheduler.Executive.Simulation_Clock task died." &
                     Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information(Error),
                     Severity => Log.Fatal);
      raise;

   end Simulation_Clock;

   -------------------------------------------------------------------------------
   -- Externally visible routines.
   --

   ----------------------------------------------------------------------------
   -- This routine starts scheduling tasks that have been registered with the
   -- scheduler collection.
   ----------------------------------------------------------------------------
   procedure Start is
   begin

      Configuration.Initialize;
      Record_Playback.Reset;

      Set_Real_Time_Clock
        (System_Clock_Rate => Configuration.Cycles_Per_Frame,
         Timer_Period      => Configuration.Cycle_Length
         );

      -- Reset all the real-time clock based functionality. This *must* be done
      -- after the real-time clock is changed, as changing the real-time clock
      -- renders old real-time clock values meaningless.
      Mission_Time.Initialize;

      Save_Restore.IOS_Interface.Register_Variables;

      -- Initialize all the bands (and thus all the modules).
      Band.Initialize_All;

      Snapshot.Initialize (Band.Snapshot_Size_Sum + Buffer_Stream.Simulation.Series_Overhead);

      Log.Report ("Simulation Initialized");

      Simulation_Clock.Start;

      IOS_Communications.Initialize;

   exception
      when Configuration.Bad =>
         Abort_Simulation(Unsuccessful_Processing_Of_Config_Data);

      when others =>
         Log.Report (Event    => "Scheduler-Executive.adb: RUN_SIMULATION",
                     Severity => Log.Error);
      raise;

   end Start;

   -------------------------------------------------------------------------------
   -- Enable or disable saving of model data. An input value of False indicates
   -- saving is to be disabled.
   -------------------------------------------------------------------------------
   procedure Enable_Saving ( Enable : in Boolean := True) is
   begin
      Model_Saving := Enable;
   end Enable_Saving;

   -------------------------------------------------------------------------------
   -- Put the scheduler into freeze. This will cause the registered schedulers to
   -- have their Update_In_Freeze routines called rather than their Update
   -- routines.
   -------------------------------------------------------------------------------
   procedure Freeze (On : in Boolean := True) is
   begin
      IOS_Communications.Host_Update_In_Freeze_Active := On;
   end Freeze;

   -------------------------------------------------------------------------------
   -- Empty out all the replay buffers for all the schedulers.
   -------------------------------------------------------------------------------
   procedure Flush_All_Replay_Buffers is
   begin
      Band.Flush_All_Replay_Buffers;
   end Flush_All_Replay_Buffers;

   -------------------------------------------------------------------------------
   -- Return the last cycle executed (in the range 0 ..
   -- Scheduler.Configuration.Cycles_Per_Frame - 1).
   -------------------------------------------------------------------------------
   function Last_Cycle return Natural is
   begin
      return Cycle_Count;
   end Last_Cycle;

   -------------------------------------------------------------------------------
   -- Terminate the current freeze condition (if one exists) on the next cycle of
   -- the given value.
   -------------------------------------------------------------------------------
   procedure Thaw_On_Cycle (Cycle : in Natural) is
   begin
      Scheduled_Thaw := Cycle;
   end Thaw_On_Cycle;

end Scheduler.Executive;
