-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Real_Time;
with Ada.Strings.Maps;
with Ada.Text_IO;

with Token.Analyzer;       -- The lecial analyzer
with Token.Line_Comment;   -- Recognizer for line-oriented comments
with Token.End_Of_File;    -- Recognizer for the end of a file
with Token.Keyword;        -- Recognizer for specified space-delimited words
with Token.Character_Set;  -- Recognizer for sequences of specified characters.
with Token.String;         -- Recognizer for string literals
with Token.Csv_Field;      -- Recognizer for separated DB-style fields
with Token.Integer;        -- Recognizer for integer literals

with Log;

with Scheduler.Band;
with Scheduler.IOS_Communications;
with Scheduler.Module;
with Scheduler.Registrar;

use type Ada.Strings.Maps.Character_Set;
use type Ada.Real_Time.Time_Span;

-------------------------------------------------------------------------------
-- This package provides the simulation scheduler's startup configuration.
-------------------------------------------------------------------------------
package body Scheduler.Configuration is

   Config_File_Name : constant STRING := "sim.cfg";

   Config_File : Ada.Text_IO.File_Type;

   -- Define the tokens in a config file

   -- Note that Name needs to be defined last in the ID list, because a
   -- space-delimited string will match with just about everything else too.
   type Sim_Cfg_Token_ID is (Run_The_Clock, Run_The, AT_ID, Hz, Schedule, Plus, Cycles,
                             In_Id, Enabled, Disabled,
                             Int, Quoted_String, EOL, Comment, Equals, Whitespace, Name, EOF);

   package Config_Analyzer is new Token.Analyzer(Sim_Cfg_Token_ID);

   Run_The_Clock_Token : aliased Token.Class := Token.Keyword.Get("Run the clock");
   Run_The_Token       : aliased Token.Class := Token.Keyword.Get("Run the");
   At_Token            : aliased Token.Class := Token.Keyword.Get("At");
   Hz_Token            : aliased Token.Class := Token.Keyword.Get("Hz");
   Schedule_Token      : aliased Token.Class := Token.Keyword.Get("Schedule");
   Plus_Token          : aliased Token.Class := Token.Keyword.Get("Plus");
   Cycles_Token        : aliased Token.Class := Token.Keyword.Get("Cycles");
   In_Token            : aliased Token.Class := Token.Keyword.Get("in");
   Enabled_Token       : aliased Token.Class := Token.Keyword.Get("Enabled");
   Disabled_Token      : aliased Token.Class := Token.Keyword.Get("Disabled");
   Integer_Literal     : aliased Token.Class := Token.Integer.Get;
   String_Literal      : aliased Token.Class := Token.String.Get;
   EOL_Token           : aliased Token.Class := Token.Character_Set.Get
     (Ada.Strings.Maps.To_Set (Token.EOL_Character));
   Comment_Token       : aliased Token.Class := Token.Line_Comment.Get("#");
   Equals_Token        : aliased Token.Class := Token.Keyword.Get("=");
   Whitespace_Token    : aliased Token.Class := Token.Character_Set.Get
     (Token.Character_Set.Standard_Whitespace -
      Ada.Strings.Maps.To_Set(Token.EOL_Character));
   Name_Token          : aliased Token.Class := Token.Csv_Field.Get
     (Separators => Ada.Strings.Maps.To_Set(" ="));
   EOF_Token           : aliased Token.Class := Token.End_Of_File.Get;

   Configuration_Syntax : Config_Analyzer.Syntax :=
     (Run_The_Clock => Run_The_Clock_Token'Access,
      Run_The       => Run_The_Token'Access,
      AT_ID         => At_Token'Access,
      Hz            => Hz_Token'Access,
      Schedule      => Schedule_Token'Access,
      Plus          => Plus_Token'Access,
      Cycles        => Cycles_Token'Access,
      In_Id         => In_Token'Access,
      Enabled       => Enabled_Token'Access,
      Disabled      => Disabled_Token'Access,
      Int           => Integer_Literal'Access,
      Quoted_String => String_Literal'Access,
      EOL           => EOL_Token'Access,
      Comment       => Comment_Token'Access,
      Equals        => Equals_Token'Access,
      Whitespace    => Whitespace_Token'Access,
      Name          => Name_Token'Access,
      EOF           => EOF_Token'Access
      );

   -- Define a text feeder function for this analyzer
   function Get_Text_Line return String;

   type Text_Feeder is access function return String;
   Feeder : constant Text_Feeder := Get_Text_Line'Access;


   -- Define the token analyzer
   Analyzer : Config_Analyzer.Instance := Config_Analyzer.Initialize
     (Language_Syntax => Configuration_Syntax,
      Feeder          => Get_Text_Line'access);

   Parse_Error : exception;

   ----------------------------------------------
   -- Band Frequency Calculation declarations
   --

   Slowest_Band_Frequency : Ada.Real_Time.Time_Span := Ada.Real_Time.Time_Span_Zero;

   type Band_Frequency_Node;
   type Band_Frequency_List is access Band_Frequency_Node;
   type Band_Frequency_Node is record
      Frequency : Natural;
      New_Band  : Band.Instance;
      Next      : Band_Frequency_List;
   end record;

   Frequency_Sorted_List : Band_Frequency_List;

   ------------------------------------------------------------------------------------
   -- Insert a new node into the given list in ascending order of frequency. If the
   -- keys are equal, the new node is placed in front of the older nodes.
   ------------------------------------------------------------------------------------
   procedure Ordered_Insert
     (Item : in     Band.Instance;
      Key  : in     Natural;
      List : in out Band_Frequency_List
     ) is
      Lower_Node : Band_Frequency_List := List;
   begin
      -- See if the new item goes at the front.
      if List = null or else Key <= List.Frequency then
         List := new Band_Frequency_Node'
           (Frequency => Key,
            New_Band  => Item,
            Next      => List
            );
      else

         -- Find the first node that has a higher or equal frequency (or the end of the list).
         while Lower_Node.Next /= null and then Key > Lower_Node.Next.Frequency loop
            Lower_Node := Lower_Node.Next;
         end loop;

         -- Insert the new node in front of that node.
         Lower_Node.Next := new Band_Frequency_Node'
           (Frequency => Key,
            New_Band  => Item,
            Next      => Lower_Node.Next
            );
      end if;

   end Ordered_Insert;



   ------------------------------------------------------------------------------------
   -- The routine used to feed data into the token analizer. This routine reads from
   -- the analizer file one line at a time until EOF is reached.
   ------------------------------------------------------------------------------------
   function Get_Text_Line return String is
      Buffer : String(1..512);
      Data_Length : Integer;
   begin

      Ada.Text_IO.Get_Line (File => Config_File, Item => Buffer, Last => Data_Length);

      if Ada.Text_IO.End_Of_File(Config_File) then
         return Buffer(1..Data_Length) & Token.EOF_Character;
      else
         return Buffer(1..Data_Length) & Token.EOL_Character;
      end if;
   exception
      when Ada.Text_IO.End_Error =>
         return (1 => Token.EOF_Character);

   end Get_Text_Line;

   -------------------------------------------------------------------------------
   -- Parse Routines
   --

   -------------------------------------------------------------------------------
   -- Parse a keyword. There's nothing to be done with these, other than
   -- error handling.
   -------------------------------------------------------------------------------
   procedure Parse_Keyword (Keyword : in Sim_Cfg_Token_ID) is
   begin
      if Config_Analyzer.Token(Analyzer) /= Keyword then
         Log.Report
           (Event => "Expected " & Sim_Cfg_Token_ID'Image(Keyword) &
            " token. Found """ & Config_Analyzer.Lexeme(Analyzer) & """.",
            Severity => Log.Fatal);
         raise Parse_Error;
      end if;

      Config_Analyzer.Find_Next (Analyzer);
   end Parse_Keyword;

   -------------------------------------------------------------------------------
   -- Parse an equals sign. There's nothing to be done with these, other than
   -- error handling.
   -------------------------------------------------------------------------------
   procedure Parse_Equals is
   begin
      if Config_Analyzer.Token(Analyzer) /= Equals then
         Log.Report
           (Event => "Expected ""="". Found """ & Config_Analyzer.Lexeme(Analyzer)
            & """.",
            Severity => Log.Fatal);
         raise Parse_Error;
      end if;

      Config_Analyzer.Find_Next (Analyzer);
   end Parse_Equals;

   -------------------------------------------------------------------------------
   -- Parse a name. If successful, the name's value and length are passed back.
   -------------------------------------------------------------------------------
   procedure Parse_Name
     (Name_Value  : out String;
      Name_Length : out Natural
     ) is
      Lexeme : constant String := Config_Analyzer.Lexeme (Analyzer);
   begin
      if Config_Analyzer.Token(Analyzer) /= Name then
         Log.Report
           (Event    => "Space-delimited name expected. Found """ &
            Config_Analyzer.Lexeme(Analyzer) & """.",
            Severity => Log.Fatal);
         raise Parse_Error;
      end if;

      Name_Value (1..Lexeme'length) := Lexeme;
      Name_Length                   := Lexeme'Length;

      Config_Analyzer.Find_Next (Analyzer);
   end Parse_Name;

   -------------------------------------------------------------------------------
   -- Parse a name. If successful, the name's value (lexeme) is passed back.
   -------------------------------------------------------------------------------
   function Parse_Name return String is
      Lexeme : constant String := Config_Analyzer.Lexeme (Analyzer);
   begin
      if Config_Analyzer.Token(Analyzer) /= Name then
         Log.Report
           (Event => "Space-delimited name expected. Found """ &
            Config_Analyzer.Lexeme(Analyzer) & """.",
            Severity => Log.Fatal);
         raise Parse_Error;
      end if;

      Config_Analyzer.Find_Next (Analyzer);

      return Lexeme;
   end Parse_Name;

   -------------------------------------------------------------------------------
   -- Parse an integer. If successful, the integer's value is passed back.
   -------------------------------------------------------------------------------
   procedure Parse_Integer (Value : out Natural) is
   begin
      if Config_Analyzer.Token(Analyzer) /= Int then
         Log.Report
           (Event => "Integer literal expected. Found """ &
            Config_Analyzer.Lexeme(Analyzer) & """.",
            Severity => Log.Fatal);
         raise Parse_Error;
      end if;

      Value := Integer'Value(Config_Analyzer.Lexeme (Analyzer));

      Config_Analyzer.Find_Next (Analyzer);
   end Parse_Integer;

   -------------------------------------------------------------------------------
   -- Parse a variable declaration line.
   -- Right now this routine just parses and ignores the declaration.
   -------------------------------------------------------------------------------
   procedure Parse_Variable is
      Variable_Name   : String (1..256);
      Variable_Length : Natural;
   begin
      Parse_Name
        (Name_Value  => Variable_Name,
         Name_Length => Variable_Length
         );
      Parse_Equals;
      -- Skip the string field
      Config_Analyzer.Find_Next (Analyzer);
   end Parse_Variable;

   -------------------------------------------------------------------------------
   -- Parse a rate, and return its value in Hz.
   -------------------------------------------------------------------------------
   procedure Parse_Rate (Rate : out Natural) is
   begin
      Parse_Keyword (At_Id);
      Parse_Integer (Rate);
      Parse_Keyword (Hz);
   end Parse_Rate;

   -------------------------------------------------------------------------------
   -- Parse an optional cycle offset, and return its value in cycles.
   -------------------------------------------------------------------------------
   procedure Parse_Offset (Offset : out Natural) is
   begin
      if Config_Analyzer.Token(Analyzer) = Plus then
         Config_Analyzer.Find_Next (Analyzer);

         Parse_Integer (Offset);
         Parse_Keyword (Cycles);
      else
         Offset := 0;
      end if;
   end Parse_Offset;

   -------------------------------------------------------------------------------
   -- Parse the simulation clock rate line.
   -------------------------------------------------------------------------------
   procedure Parse_Clock_Line is
   begin

      Parse_Keyword (Run_The_Clock);
      Parse_Rate (Cycles_Per_Frame);
      Cycle_Length := 1.0 / Float(Cycles_Per_Frame);

   end Parse_Clock_Line;

   -------------------------------------------------------------------------------
   -- Parse a run line.
   -------------------------------------------------------------------------------
   procedure Parse_Run_Line is
   begin

      -- Load up the next token.
      Config_Analyzer.Find_Next (Analyzer);

      declare
         -- Parse and save the module and band names
         Module_Name     : constant String := Parse_Name;
         Module_Active   : Boolean := True;
      begin
         -- Figure out if its disabled or enabled.
         if Config_Analyzer.Token(Analyzer) = Disabled then
            Module_Active := False;
            Config_Analyzer.Find_Next (Analyzer);
         elsif Config_Analyzer.Token(Analyzer) = Enabled then
            Config_Analyzer.Find_Next (Analyzer);
         end if;

         Parse_Keyword (In_Id);

         Band.Add_Module
           (Subject    => Band.Find (Parse_Name),
            New_Module => Module.Create
            (Name   => Module_Name,
             Active => Module_Active
             )
            );

      end;
   end Parse_Run_Line;

   -------------------------------------------------------------------------------
   -- Parse a band configuration line.
   -------------------------------------------------------------------------------
   procedure Parse_Schedule_Line is
   begin
      -- Load up the next token, and parse the configuration line.
      Config_Analyzer.Find_Next (Analyzer);

      declare
         Start_Cycle       : Natural;
         Scheduling_Period : Natural;

         Band_Name         : constant String := Parse_Name;
         Band_Active       : Boolean := True;
      begin
         -- Figure out if its disabled or enabled.
         if Config_Analyzer.Token(Analyzer) = Disabled then
            Band_Active := False;
            Config_Analyzer.Find_Next (Analyzer);
         elsif Config_Analyzer.Token(Analyzer) = Enabled then
            Config_Analyzer.Find_Next (Analyzer);
         end if;

         Parse_Rate (Scheduling_Period);

         -- Verify that the requested rate is attainable
         if Cycles_Per_Frame mod Scheduling_Period /= 0 then
            Log.Report
              (Event    => "Unable to schedule a" & Integer'Image(Scheduling_Period) &
               "hz band using a " & Integer'Image(Cycles_Per_Frame) & "hz clock.",
               Severity => Log.Fatal
               );
            raise Parse_Error;
         end if;

         Parse_Offset (Start_Cycle);

         -- Create the designated frequency band
         Ordered_Insert
           (Item => Band.Create
            (Name              => Band_Name,
             Start_Cycle       => Start_Cycle,
             Scheduling_Period => Cycles_Per_Frame / Scheduling_Period,
             Active            => Band_Active
             ),
            Key  => Scheduling_Period,
            List => Frequency_Sorted_List
            );

         -- Recalculate the slowest band frequency yet found.
         if
           Ada.Real_Time.To_Time_Span
           (Scheduling_Period * Duration(Cycle_Length)) > Slowest_Band_Frequency
         then
            Slowest_Band_Frequency := Ada.Real_Time.To_Time_Span
              (Scheduling_Period * Duration(Cycle_Length));
         end if;
      end;
   end Parse_Schedule_Line;

   -------------------------------------------------------------------------------
   -- Parse a configuration line. This one of several declarations.
   -------------------------------------------------------------------------------
   procedure Parse_Config_Line is
   begin
      case Config_Analyzer.Token(Analyzer) is
         when Run_The =>
            Parse_Run_Line;
         when Schedule =>
            Parse_Schedule_Line;
         when Run_The_Clock =>
            Log.Report (Event => "Extra simulation clock line",
                        Severity => Log.Fatal
                        );
            raise Parse_Error;
         when Name =>
            Parse_Variable;
         when others =>
            Log.Report (Event => "Expected ""Run the clock"", ""Run the"", " &
                        """Schedule"", or variable name. Found """ & Config_Analyzer.Lexeme(Analyzer) &
                        """.",
                        Severity => Log.Fatal);
            raise Parse_Error;
      end case;
   end Parse_Config_Line;

   -------------------------------------------------------------------------------
   -- Initialize the scheduler's configuration. This is accomplished by reading
   -- and parsing the file "sim.cfg" in the current working directory.
   -------------------------------------------------------------------------------
   procedure Initialize is
      Next_Band : Band_Frequency_List;
      Priority  : Natural := 1;
   begin
      -- Open the config file and get the first token.
      Ada.Text_IO.Open
        (File => Config_File,
         Mode => Ada.Text_IO.In_File,
         Name => Config_File_Name
         );

      Config_Analyzer.Find_Next (Analyzer);

      -- Parse the clock frequency line
      Parse_Clock_Line;

      -- Parse lists of command lines
      while Config_Analyzer.Token (Analyzer) /= EOF loop
         Parse_Config_Line;
      end loop;

      Log.Report ("Closing " & Config_File_Name);
      Ada.Text_IO.Close (Config_File);

      -- Assign band priorities based on their relative rates
      Next_Band := Frequency_Sorted_List;
      while Next_Band /= null loop
         Band.Initialize
           (Subject  => Next_Band.New_Band,
            Priority => Priority
            );

         Priority := Priority + 1;
         Next_Band := Next_Band.Next;
      end loop;

   exception
      when Ada.Text_IO.Name_Error =>
         Log.Report (Event    => "Unable to locate or open the configuration file """ &
                     Config_File_Name & "",
                     Severity => Log.Fatal);
         raise;
      when Error : others =>
         Log.Report (Event    => "Parsing configuration file """ & Config_File_Name & """ at line" &
                     Integer'Image(Config_Analyzer.Line(Analyzer)) & ", column" &
                     Integer'Image(Config_Analyzer.Column(Analyzer)) & ".",
                     Severity => Log.Fatal);
         Log.Report (Event    => Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Fatal);

         raise;
   end Initialize;

   -------------------------------------------------------------------------------
   -- Return the time required for every model to run once.
   -------------------------------------------------------------------------------
   function Simulation_Frequency return Ada.Real_Time.Time_Span is
   begin
      return Slowest_Band_Frequency;
   end Simulation_Frequency;

end Scheduler.Configuration;
