-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with System;

with Buffer_Stream.Simulation;
with Scheduler.Module;
with ScopeProfile;

-------------------------------------------------------------------------------
-- A scheduler band consists of a thread of control under which one or more
-- scheduled items may run. Each subsystem scheduler will be assigned to a band
-- based on the configuration information.
-------------------------------------------------------------------------------
private package Scheduler.Band is

   type Instance is private;

   Not_Found      : exception;

   ----------------------------------------------------------------------------
   -- Find the scheduler band with the given name. If no such module exists,
   -- Not_Found will be raised.
   ----------------------------------------------------------------------------
   function Find (Name : in String) return Instance;

   ----------------------------------------------------------------------------
   -- Add the given module to the given scheduler band.
   ----------------------------------------------------------------------------
   procedure Add_Module
     (Subject    : in Instance;
      New_Module : in Scheduler.Module.Instance
     );

   ----------------------------------------------------------------------------
   -- Create a new band with the given attributes.
   ----------------------------------------------------------------------------
   function Create
     (Name              : in String;
      Start_Cycle       : in Natural;
      Scheduling_Period : in Natural;
      Active            : in Boolean
     ) return Instance;

   ----------------------------------------------------------------------------
   -- Initialize the given band at the given (relative) priority.
   ----------------------------------------------------------------------------
   procedure Initialize
     (Subject    : in Instance;
      Priority   : in Natural
     );

   ----------------------------------------------------------------------------
   -- Update all scheduled items in all frequency bands.
   -- The current cycle count is used for bookkeeping. The subframe is used
   -- to decide which models update.
   ----------------------------------------------------------------------------
   procedure Update_All
     (Total_Cycles : in Natural;
      Subframe     : in Natural
     );

   -------------------------------------------------------------------------------
   -- Initialize all the real-time modules in all frequency bands.
   -------------------------------------------------------------------------------
   procedure Initialize_All;

   ----------------------------------------------------------------------------
   -- Set the Ada priority for the current task.
   ----------------------------------------------------------------------------
   procedure Set_Task_Priority( Relative_Priority : in Integer );

   -------------------------------------------------------------------------------
   -- Request a snapshot for all real-time modules. This should cause
   -- Record_Playback.Snapshot to be called for each module before its next
   -- update.
   -------------------------------------------------------------------------------
   procedure Set_Module_Snapshot_Requests;

   -------------------------------------------------------------------------------
   -- Empty out all the replay buffers for all the schedulers in all the bands.
   -------------------------------------------------------------------------------
   procedure Flush_All_Replay_Buffers;

   -------------------------------------------------------------------------------
   -- Snapshot all the models in all the bands to the given stream.
   -------------------------------------------------------------------------------
   procedure Snapshot_All (Destination  : in Buffer_Stream.Simulation.Handle);

   -------------------------------------------------------------------------------
   -- Return the sum of all the snapshot sizes for all schedulers in all bands.
   -- This size includes enough space to store the module names as well.
   -------------------------------------------------------------------------------
   function Snapshot_Size_Sum return Natural;

private

   type Band_List_Node;
   type Instance is access all Band_List_Node;
end Scheduler.Band;
