-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Dynamic_Priorities;
with Ada.Exceptions;
with Interfaces.C;
with System.Storage_Elements;

with High_Resolution_Timer;
with Log;
with Scheduler.Configuration;
with Scheduler.IOS_Communications;

-------------------------------------------------------------------------------
-- A scheduler band consists of a thread of control under which one or more
-- scheduled items may run. Each subsystem scheduler will be assigned to a band
-- based on the configuration information.
-------------------------------------------------------------------------------
package body Scheduler.Band is

   ----------------------------------------------------------------------------
   -- Tunable Constant Section
   --
   Scheduler_Base_Priority : constant System.Any_Priority := System.Default_Priority + 4;


   ----------------------------------------------------------------------------
   -- Band information declarations.
   --
   type Band_List is new Instance;

   ----------------------------------------------------------------------------
   -- Band task type spec.  The most common usage is for periodic activation
   -- of real time processing e.g. there usually are 60 HZ, 30 HZ, 15 HZ, etc.
   -- tasks.  There will be a frequency band type allocated for each task.
   ----------------------------------------------------------------------------
   task type Frequency_Band is

      -------------------------------------------------------------------------
      -- Rendezvous for initializing the band task.
      -------------------------------------------------------------------------
      entry Initialize (Data       : in Band_List;
                        Priority   : in Integer
                       );

      -------------------------------------------------------------------------
      -- Called to tell the band to call its modules.
      -------------------------------------------------------------------------
      entry Update (Total_Cycles : in Natural);

      -------------------------------------------------------------------------
      -- Called to cancel the band.
      -------------------------------------------------------------------------
      entry Cancel;

   end Frequency_Band;

   type Frequency_Band_Ptr is access Frequency_Band;

   type String_Ptr is access String;

   -- Link list node that comprises the list which contains the modules to process.
   type Band_List_Node is record

      -- Pointer to the band task.
      Band_Ptr : Frequency_Band_Ptr;

      -- Band configuration data
      Name              : String_Ptr;
      Start_Cycle       : Natural;
      Scheduling_Period : Positive;

      -- Modules associated with this band
      Module_List       : Module.List;

      -- IOS display data (accessed indirectly).
      Active            : aliased Boolean := True;
      Execution_Time    : aliased Float   := 0.0;

      -- Profiling (for debugging. Requires a profiler license to use.)
      Profiling         : Boolean := False;
      Profiler_Instance : ScopeProfile.Instance;

      -- Pointer to next node in the linked list.
      Next        : Band_List;
   end record;

   -- Points to the head of a linked list of real-time band tasks.
   Registered_Band_List_Head : Band_List;
   Registered_Band_List_Tail : Band_List;

   ----------------------------------------------------------------------------
   -- Imported subprograms
   --

   function taskIdSelf return Interfaces.C.int;
   pragma Import(C, taskIdSelf, "taskIdSelf" );

   ----------------------------------------------------------------------------
   -- Internal helper routines.
   --

   ----------------------------------------------------------------------------
   -- Return an absolute Ada task priority for the given relative priority.
   ----------------------------------------------------------------------------
   function Absolute_Priority (Relative_Priority : System.Any_Priority )
     return System.Any_Priority
   is
   begin
      return Relative_Priority + Scheduler_Base_Priority;
   end Absolute_Priority;

   ----------------------------------------------------------------------------
   -- Band task type.  The most common usage is for periodic activation
   -- of real time processing e.g. there usually are 60 HZ, 30 HZ, 15 HZ, etc.
   -- tasks.  There will be a frequency band type allocated for each task.
   ----------------------------------------------------------------------------
   task body Frequency_Band is

      -- Band data for this band.
      Band_Data : Band_List;

      -- The cycle count for this iteration
      Total_Number_Of_Cycles : Natural := 0;

      Delta_Time : Float;

      Base_Priority : constant := 10;

      Timer  : High_Resolution_Timer.Instance;
   begin

      accept Initialize (Data       : in Band_List;
                         Priority   : in Integer
                        ) do

         Band_Data := Data;
         Delta_Time := Configuration.Cycle_Length * Float(Data.Scheduling_Period);

         Set_Task_Priority (Priority + Base_Priority);

         Band_Data.Profiler_Instance := ScopeProfile.Initialize (Task_ID => TaskIdSelf);

         ScopeProfile.Freeze (Band_Data.Profiler_Instance);

         Log.Report (Band_Data.Name.all & " band's profile ID is " &
                     ScopeProfile.Image (Band_Data.Profiler_Instance)
                     );

      end Initialize;

      loop

         select

            accept Update (Total_Cycles : in Natural) do
               Total_Number_Of_Cycles := Total_Cycles;
            end Update;

            -- Call Update on all the modules in the module list
            High_Resolution_Timer.Start (Timer);
            Module.Update_All
              (Module_List  => Band_Data.Module_List,
               Delta_Time   => Delta_Time,
               Total_Cycles => Total_Number_Of_Cycles,
               Enabled      => Band_Data.Active);

            -- Update the band's execution time.
            High_Resolution_Timer.Stop (Timer);

            Band_Data.Execution_Time := High_Resolution_Timer.Milliseconds (Timer);

            or accept Cancel;

               exit;

            or terminate;

         end select;

      end loop;

   exception
      when Error : others =>
         Log.Report (Event    => Ada.Exceptions.Exception_Information(Error) &
                     Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
                     "Scheduler.Band: Frequency_Band" & Band_Data.Name.all,
                     Severity => Log.Error);
      raise;

   end Frequency_Band;


   ----------------------------------------------------------------------------
   -- Externally visible routines.
   --

   ----------------------------------------------------------------------------
   -- Find the scheduler band with the given name. If no such module exists,
   -- Not_Found will be raised.
   ----------------------------------------------------------------------------
   function Find (Name : in String) return Instance is
      Next_Band : Band_List := Registered_Band_List_Head;
   begin

      while Next_Band /= null loop
         if Next_Band.Name.all = Name then
            return Instance(Next_Band);
         end if;
         Next_Band := Next_Band.Next;
      end loop;

       -- Nothing matched. Report an error
       Log.Report
         (Event => Name & " is not a known scheduler band name.",
          Severity => Log.Fatal
          );
       raise Not_Found;
   end Find;

   ----------------------------------------------------------------------------
   -- Add the given module to the given scheduler band.
   ----------------------------------------------------------------------------
   procedure Add_Module
     (Subject    : in Instance;
      New_Module : in Scheduler.Module.Instance
     ) is
   begin
      Module.Enqueue
        (Module_List => Subject.Module_List,
         New_Module  => New_Module
         );
   end Add_Module;

   ----------------------------------------------------------------------------
   -- Create a new band with the given attributes.
   ----------------------------------------------------------------------------
   function Create
     (Name              : in String;
      Start_Cycle       : in Natural;
      Scheduling_Period : in Natural;
      Active            : in Boolean
     ) return Instance is
   begin
      -- Put a new band on the band list
      if Registered_Band_List_Tail = null then

         Registered_Band_List_Tail := new Band_List_Node;
         Registered_Band_List_Head := Registered_Band_List_Tail;

      else

         Registered_Band_List_Tail.Next := new Band_List_Node;
         Registered_Band_List_Tail      := Registered_Band_List_Tail.Next;

      end if;

      -- Fill in the band's info.
      Registered_Band_List_Tail.Name              := new String'(Name);
      Registered_Band_List_Tail.Start_Cycle       := Start_Cycle;
      Registered_Band_List_Tail.Scheduling_Period := Scheduling_Period;
      Registered_Band_List_Tail.Active            := Active;

      -- Create a new band task, and initialize it with the priority and band ID.
      Registered_Band_List_Tail.Band_Ptr := new Frequency_Band;

      return Instance(Registered_Band_List_Tail);
   end Create;

   ----------------------------------------------------------------------------
   -- Initialize the given band at the given (relative) priority.
   ----------------------------------------------------------------------------
   procedure Initialize
     (Subject    : in Instance;
      Priority   : in Natural
     ) is
   begin
      -- Register the band's IOS variables
      IOS_Communications.Register_Band_Data
        (Band_Name          => Subject.Name.all,
         Active_Flag_Ptr    => Subject.Active'access,
         Execution_Time_Ptr => Subject.Execution_Time'access
         );

      Subject.Band_Ptr.Initialize
        (Data       => Band_List(Subject),
         Priority   => Priority
         );
   end Initialize;

   ----------------------------------------------------------------------------
   -- Update all scheduled items in all frequency bands.
   -- The current cycle count is used for bookkeeping. The subframe is used
   -- to decide which models update.
   ----------------------------------------------------------------------------
   procedure Update_All
     (Total_Cycles : in Natural;
      Subframe     : in Natural
     ) is

      Next_Band : Band_List := Registered_Band_List_Head;

   begin

      -- While there are bands to process...
      while Next_Band /= null loop

         -- Update the profile if we are profiling.
         if Next_Band.Profiling then
            ScopeProfile.Sample (Next_Band.Profiler_Instance);
         end if;

         -- If time to execute band within the frame...
         if (Subframe - Next_Band.Start_Cycle) rem Next_Band.Scheduling_Period = 0 then

            -- Resume the band task
            select

               Next_Band.Band_Ptr.Update (Total_Cycles);

               -- Temporarily turn off profiling after update.
               if Next_Band.Profiling then
                  Next_Band.Profiling := False;
                  ScopeProfile.Freeze (Next_Band.Profiler_Instance);
               end if;

             else

                -- The band's task wasn't ready: Frame overrun.

                -- Profile things so we can find the slow parts. (Assming the overrun wasn't
                -- caused by a higher-priority band being too slow.)
                if not Next_Band.Profiling then
                  Next_Band.Profiling := True;

                  ScopeProfile.Freeze
                    (Profile   => Next_Band.Profiler_Instance,
                     Freeze_On => False
                     );

                end if;

                -- Report the problem.
                Log.Report (Event    => "Frame Overrun Band: " & Next_Band.Name.all,
                           Severity => Log.Error);

            end select;

         end if;

         Next_Band := Next_Band.Next;
      end loop;

   end Update_All;


   ----------------------------------------------------------------------------
   -- Set the Ada priority for the current task.
   ----------------------------------------------------------------------------
   procedure Set_Task_Priority (Relative_Priority : in Integer) is
   begin
      Ada.Dynamic_Priorities.Set_Priority (Absolute_Priority (Relative_Priority));
   end Set_Task_Priority;

   -------------------------------------------------------------------------------
   -- Initialize all the real-time modules in all frequency bands.
   -------------------------------------------------------------------------------
   procedure Initialize_All is
      Next_Band : Band_List := Registered_Band_List_Head;
   begin
      while Next_Band /= null loop

         Module.Initialize_All (Next_Band.Module_List);

         Next_Band := Next_Band.Next;
      end loop;
   end Initialize_All;

   -------------------------------------------------------------------------------
   -- Request a snapshot for all real-time modules. This should cause
   -- Record_Playback.Snapshot to be called for each module before its next
   -- update.
   -------------------------------------------------------------------------------
   procedure Set_Module_Snapshot_Requests is
      Next_Band : Band_List := Registered_Band_List_Head;
   begin
      while Next_Band /= null loop
         Module.Set_Snapshot_Requests (Next_Band.Module_List);

         Next_Band := Next_Band.Next;
      end loop;
   end Set_Module_Snapshot_Requests;


   -------------------------------------------------------------------------------
   -- Empty out all the replay buffers for all the schedulers in all the bands.
   -------------------------------------------------------------------------------
   procedure Flush_All_Replay_Buffers is
      Next_Band : Band_List := Registered_Band_List_Head;
   begin
      while Next_Band /= null loop

         Module.Flush_All_Replay_Buffers (Next_Band.Module_List);

         Next_Band := Next_Band.Next;
      end loop;

   end Flush_All_Replay_Buffers;

   -------------------------------------------------------------------------------
   -- Snapshot all the models in all the bands to the given stream.
   -------------------------------------------------------------------------------
   procedure Snapshot_All (Destination  : in Buffer_Stream.Simulation.Handle) is

      Next_Band : Band_List := Registered_Band_List_Head;
   begin
      while Next_Band /= null loop

         Module.Snapshot_All
           (Module_List => Next_Band.Module_List,
            Destination => Destination
            );

         Next_Band := Next_Band.Next;
      end loop;

   end Snapshot_All;


   -------------------------------------------------------------------------------
   -- Return the sum of all the snapshot sizes for all schedulers in all bands.
   -------------------------------------------------------------------------------
   function Snapshot_Size_Sum return Natural  is

      Next_Band : Band_List := Registered_Band_List_Head;
      Sum       : Natural := 0;
   begin

      while Next_Band /= null loop
         Sum := Sum + Module.Snapshot_Size_Sum (Next_Band.Module_List);

         Next_Band := Next_Band.Next;
      end loop;

      return Sum;
   end Snapshot_Size_Sum;

end Scheduler.Band;
