-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Real_Time;
with JPATS_IOS_Interface;
with Log;

-------------------------------------------------------------------------------
-- This package provides the mission time. Mission time is the amount of
-- seconds that have elapsed since mission time was last reset.
-- Values of type Duration are used to externally represent mission time. Ada
-- specifies that Duration must support representation of at least an entire
-- day in units at least as small as 20ms.
-------------------------------------------------------------------------------
package body Mission_Time is

   use type Ada.Real_Time.Time;

   -- Mission time is the difference between the current wall time and this
   -- time stamp.
   Mission_Time_Start : Ada.Real_Time.Time := Ada.Real_Time.Time_First;

   IOS_Mission_Time   : Float := 0.0;

   EOL : constant String := Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF;

   ----------------------------------------------------------------------------
   -- Set the Mission time to the given IOS Interface value.
   -- The expected type of this value is Float.
   ----------------------------------------------------------------------------
   procedure Set_Mission_Time (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      IOS_Mission_Time := JPATS_IOS_Interface.Convert_To_Float(Value);
      Reset (Duration(IOS_Mission_Time));
   end Set_Mission_Time;

   ----------------------------------------------------------------------------
   -- Update the IOS's value for the mission time.
   ----------------------------------------------------------------------------
   procedure Update is
   begin
      IOS_Mission_Time := Float (Get);
   end Update;

   ----------------------------------------------------------------------------
   -- Initialize this facility. This will reset the mission time to 0 seconds
   -- and register the appropriate IOS variables.
   ----------------------------------------------------------------------------
   procedure Initialize is
   begin
      Reset;
      JPATS_IOS_Interface.Register
        (Name        => "Mission_Time",
         Variable    => IOS_Mission_Time'address,
         Set_Routine => Set_Mission_Time'access
         );
   end Initialize;

   ----------------------------------------------------------------------------
   -- Reset the mission time to the given amount of seconds.
   ----------------------------------------------------------------------------
   procedure Reset (To : in Duration := 0.0) is
   begin
      Mission_Time_Start := Ada.Real_Time.Clock - Ada.Real_Time.To_Time_Span(To);
   end Reset;

   ----------------------------------------------------------------------------
   -- Return the current mission time (in seconds).
   ----------------------------------------------------------------------------
   function Get return Duration is
   begin
      return Ada.Real_Time.To_Duration(Ada.Real_Time.Clock - Mission_Time_Start);

   -- Handle unexpected results
   exception
      when Constraint_Error =>
         declare
            Now : Ada.Real_Time.Time := Ada.Real_Time.Clock;

            Now_Sc, Mt_Sc : Ada.Real_Time.Seconds_Count;
            Now_Ts, Mt_Ts : Ada.Real_Time.Time_Span;
         begin
            Ada.Real_Time.Split (Now, Now_Sc, Now_Ts);
            Ada.Real_Time.Split (Mission_Time_Start ,  Mt_Sc,  Mt_Ts);

            if Now > Mission_Time_Start then
               Log.Report
                 ("Bad Mission Time (" &
                  Ada.Real_Time.Seconds_Count'Image (Now_Sc) & " seconds +" &
                  Duration'Image(Ada.Real_Time.To_Duration(Now_Ts)) & "  - (" &
                  Ada.Real_Time.Seconds_Count'Image (Mt_Sc) & " seconds +" &
                  Duration'Image(Ada.Real_Time.To_Duration(Mt_Ts)) & ") )" & EOL &
                  "Mission time will be reset to 0."
                  );
               Mission_Time_Start := Ada.Real_Time.Clock;
               return 0.0;
            end if;


            Log.Report (Event => "Bad Mission Time (" &
                        Ada.Real_Time.Seconds_Count'Image (Now_Sc) & " seconds +" &
                        Duration'Image(Ada.Real_Time.To_Duration(Now_Ts)) & "  -  " &
                        Ada.Real_Time.Seconds_Count'Image (Mt_Sc) & " seconds +" &
                        Duration'Image(Ada.Real_Time.To_Duration(Mt_Ts)) & ")"
                        ,
                        Severity => Log.Error);
         end;
         raise;
   end Get;

end Mission_Time;
