-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

with Coordinate_Transformation;
with Integration_G;


package body Velocity is

   min_v : constant float := 0.000_001;
   package Velocity_Integration is new Integration_G (Variable_Type => Float);
   package Ang_Rate_Integration is new Integration_G (Variable_Type => Float);

     --| calc Velocity in Earth axis coordinate system
   procedure Set_Earth_Axis
     (Accel_Earth_Axis       :in  Coordinate_Types.Cartesian;
      lp_accel_Ea            :in  Coordinate_Types.Cartesian;
      Dt                     :in  Float;
      Stopped                :in  Boolean;
      An_Instance            :in  out Instance) is
      Temp : Float := 0.0;
   begin
      An_Instance.The_Last_Pass_Vel_Earth_Axis
                   := An_Instance.The_Vel_Earth_Axis;
      An_Instance.The_Vel_Earth_Axis.X  -- this is north velocity
             :=   Velocity_Integration.Predictor
                        (An_Instance.The_Vel_Earth_Axis.x ,
                         Accel_Earth_Axis.x ,
                         lp_accel_ea.x,
                         Dt);

      An_Instance.The_Vel_Earth_Axis.Y -- this is east velocity
             :=   Velocity_Integration.Predictor
                        (An_Instance.The_Vel_Earth_Axis.y ,
                         Accel_Earth_Axis.y ,
                         lp_accel_ea.y,
                         Dt);

      An_Instance.The_Vel_Earth_Axis.z
              :=   Velocity_Integration.Predictor
                        (An_Instance.The_Vel_Earth_Axis.z ,
                         Accel_Earth_Axis.z ,
                         lp_accel_ea.z,
                         Dt);
      if Stopped then
         An_Instance.The_Vel_Earth_Axis.Y := 0.0;
         An_Instance.The_Vel_Earth_Axis.X := 0.0;
      end if;
      Temp := an_Instance.The_vel_earth_Axis.x;
      if abs(temp) <= min_v then
         if Temp > 0.0 then
            temp := min_v;
         else
            Temp := - Min_V;
         end if;
      end if;
      an_instance.Direction := Arctan( an_instance.The_vel_earth_Axis.y/
                                              Temp);
      if Temp < 0.0 then -- 2nd or 3rd quadrant
         an_Instance.direction := an_Instance.direction + 3.1416;
      end if;
   end Set_Earth_Axis;

   procedure Set_Inertial(Roll_Angle            :in     Angle_Types.Radians;
                          Pitch_Angle           :in     Angle_Types.Radians;
                          hdg_Angle             :in     Angle_Types.Radians;
                          An_Instance           :in out Instance) is
   begin
      An_Instance.The_Inertial_Vel := Coordinate_Transformation.Earth_to_Body
        (An_Instance.The_Vel_Earth_Axis,
         Roll_Angle                  ,
         Pitch_Angle                 ,
         hdg_Angle                   );
   end Set_Inertial;

   procedure Assign_Inertial
        (Inertial_Vel   :in     Coordinate_Types.Cartesian;
         An_Instance    :in out Instance) is
   begin
      An_Instance.The_Inertial_Vel := Inertial_Vel;
   end Assign_Inertial;

   function Get_Inertial(An_Instance :in Instance)
      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Inertial_Vel;
   end Get_Inertial;

--| calc Velocity in Body axis coordinates
   procedure Set_Body_Axis(Roll_Angle     :in     Angle_Types.Radians;
                           Pitch_Angle    :in     Angle_Types.Radians;
                           Hdg_Angle      :in     Angle_Types.Radians;
                           North_Vel      :in     Length_Types.Feet_Per_Sec;
                           east_Vel       :in     Length_Types.Feet_Per_Sec;
                           vert_Vel       :in     Length_Types.Feet_Per_Sec;
                           An_Instance    :in out Instance) is
      Temp : Coordinate_Types.Cartesian := (0.0,0.0,0.0);
   begin
      Temp := (North_Vel,East_Vel,Vert_Vel);

      An_Instance.The_Vel_Body_Axis := Coordinate_Transformation.Earth_to_Body
              (Temp         ,
               Roll_Angle   ,
               Pitch_Angle  ,
               hdg_Angle     );
      if abs(An_Instance.The_Vel_Body_Axis.X) < Min_V then
         if An_Instance.The_Vel_Body_Axis.X >= 0.0 then
             An_Instance.The_Vel_Body_Axis.X := min_v;
         else
             An_Instance.The_Vel_Body_Axis.X := -Min_V;
         end if;
      end if;
   end Set_Body_Axis;

   procedure Set_Body_Axis_based_on_True_Airspeed
     (TAS              :in      Length_Types.Feet_per_Sec;
      AOA              :in      Angle_Types.Radians;
      Side_Slip_Angle  :in      Angle_Types.Radians;
      An_Instance      :in  out Instance) is
      Temp    : Float := 0.0;
   begin
      An_Instance.The_Vel_Body_Axis.z := TAS * Sin(AOA);
      Temp := Tas * Cos(Aoa);
      An_Instance.The_Vel_Body_Axis.x := Temp * Cos(Side_Slip_Angle);
      An_Instance.The_Vel_Body_Axis.y := Temp  * sin(Side_Slip_Angle);
   end Set_Body_Axis_based_on_True_Airspeed;

   procedure Assign_Body_Axis(Init_V_B     :in     Coordinate_Types.Cartesian;
                              An_Instance  :in out Instance) is
   begin
      An_Instance.The_Vel_Body_Axis   := Init_V_B;
      An_Instance.The_Vel_Body_Axis.x := Float'Max(An_Instance.The_Vel_Body_Axis.x,min_v);
   end Assign_Body_Axis;

   function Get_Body_Axis(An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Vel_Body_Axis;
   end Get_Body_Axis;


   --| To be used during formation flying for lead aircraft.
   procedure Set_Body_to_Earth_Axis
     (Roll_Angle         :in     Angle_Types.Radians;
      Pitch_Angle        :in     Angle_Types.Radians;
      Yaw_Angle          :in     Angle_Types.Radians;
      North_Wind         :in  Length_Types.Feet_per_Sec;
      east_Wind          :in  Length_Types.Feet_per_Sec;
      vertical_Wind      :in  Length_Types.Feet_per_Sec;
      An_Instance        :in out Instance) is
   begin
      An_Instance.The_Vel_Earth_Axis
                := Coordinate_Transformation.Body_to_Earth
                   (An_Instance.The_Vel_Body_Axis,
                    Roll_Angle                 ,
                    Pitch_Angle                ,
                    Yaw_Angle                  );

      An_Instance.The_Vel_Earth_Axis.X
           := An_Instance.The_Vel_Earth_Axis.x + North_wind;
      An_Instance.The_Vel_Earth_Axis.Y
           := An_Instance.The_Vel_Earth_Axis.y + East_wind;
      An_Instance.The_Vel_Earth_Axis.z
           := An_Instance.The_Vel_Earth_Axis.z + Vertical_wind;

   end Set_Body_to_Earth_Axis;

   procedure Assign_Earth_Axis
         (Vel_Earth_Axis           :in  Coordinate_Types.Cartesian;
          Last_Pass_Vel_Earth_Axis :in  Coordinate_Types.Cartesian;
          An_Instance                   :in  out Instance) is
   begin
      An_Instance.The_Last_Pass_Vel_Earth_Axis
                          := Last_Pass_Vel_Earth_Axis;
      An_Instance.The_Vel_Earth_Axis  := Vel_Earth_Axis;
   end Assign_Earth_Axis;

   function Get_Earth_Axis(An_Instance :in Instance)
      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Vel_Earth_Axis;
   end Get_Earth_Axis;

   function Get_Last_Pass_Earth_Axis(An_Instance :in Instance)
      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Last_Pass_Vel_Earth_Axis;
   end Get_Last_Pass_Earth_Axis;

   procedure Set_Roll_Rate
     (Roll_Accel              :in     Angle_Types.Radians_per_Sq_Sec;
      Last_Pass_Roll_Accel    :in     Angle_Types.Radians_per_Sq_Sec;
      Dt                      :in     Float;
      An_Instance             :in out Instance) is
   begin
      An_Instance.The_Last_Pass_Roll_Rate := An_Instance.The_Roll_Rate;
      An_Instance.The_Roll_Rate
        := Ang_Rate_Integration.Predictor
              (An_Instance.The_Roll_Rate,  Roll_Accel,
               Last_Pass_Roll_Accel, Dt);
   end Set_Roll_Rate;

   procedure Assign_Roll_Rate
        (Roll_Rate        :in     Angle_Types.Radians_per_Sec;
         An_Instance      :in out Instance) is
   begin
      An_Instance.The_Roll_Rate           := Roll_Rate;
   end Assign_Roll_Rate;

   function Get_Roll_Rate (An_Instance :in Instance)
      return Angle_Types.Radians_per_Sec is
   begin
      return  An_Instance.The_Roll_Rate;
   end Get_Roll_Rate;

   procedure Set_Pitch_Rate
     (Pitch_Accel            :in     Angle_Types.Radians_per_Sq_Sec;
      Last_Pass_Pitch_Accel  :in     Angle_Types.Radians_per_Sq_Sec;
      Dt                     :in     Float;
      An_Instance            :in out Instance) is
   begin
      An_Instance.The_Last_Pass_Pitch_Rate := An_Instance.The_Pitch_Rate;
      An_Instance.The_Pitch_Rate
          := Ang_Rate_Integration.Predictor(An_Instance.The_Pitch_Rate,
                   Pitch_Accel, Last_Pass_Pitch_Accel, Dt);
   end Set_Pitch_Rate;

   procedure Assign_Pitch_Rate
       (Pitch_Rate      :in     Angle_Types.Radians_per_Sec;
        An_Instance     :in out Instance) is
   begin
      An_Instance.The_Pitch_Rate := Pitch_Rate;
   end Assign_Pitch_Rate;

   function Get_Pitch_Rate (An_Instance :in Instance)
      return Angle_Types.Radians_per_Sec is
   begin
      return  An_Instance.The_Pitch_Rate;
   end Get_Pitch_Rate;

   --| This procedure computes the Yaw Angular Rate.
   procedure Set_Yaw_Rate
      (Yaw_Accel            :in     Angle_Types.Radians_per_Sq_Sec;
       Last_Pass_Yaw_Accel  :in     Angle_Types.Radians_per_Sq_Sec;
       Dt                   :in     Float;
      An_Instance           :in out Instance) is
   begin
      An_Instance.The_Last_Pass_Yaw_Rate := An_Instance.The_Yaw_Rate;
      An_Instance.The_Yaw_Rate
           := Ang_Rate_Integration.Predictor(An_Instance.The_Yaw_Rate,
                               Yaw_Accel, Last_Pass_Yaw_Accel,
                               Dt);
   end Set_Yaw_Rate;

   procedure Assign_Yaw_Rate
        (Yaw_Rate        :in     Angle_Types.Radians_per_Sec;
         An_Instance     :in out Instance) is
   begin
      An_Instance.The_Yaw_Rate := Yaw_Rate;
   end Assign_Yaw_Rate;

   function Get_Yaw_Rate (An_Instance :in Instance)
      return Angle_Types.Radians_per_Sec is
   begin
      return  An_Instance.The_Yaw_Rate;
   end Get_Yaw_Rate;

   --| This procedure computes the Nondimensional Angular Rate.
   procedure Set_Nondimensional_Angular
     (Span_Wing        :in     Length_Types.Feet;
      Chord_Wing       :in     Length_Types.Feet;
      TAS              :in     Length_Types.Feet_per_Sec;
      AOA              :in     Angle_Types.Radians;
      An_Instance      :in out Instance) is
      Sin_AOA     : Float;
      Cos_Aoa     : Float;
   begin
      Sin_AOA    := Sin(AOA);
      Cos_Aoa    := Cos(AOA);

      An_Instance.The_Nondimensional_Angular.Roll :=
          Span_Wing * (An_Instance.The_Roll_Rate * Cos_Aoa
                       + An_Instance.The_Yaw_Rate * Sin_Aoa) /
                            Float'Max(2.0 * TAS,0.01);

      An_Instance.The_Nondimensional_Angular.Pitch :=
             An_Instance.The_Pitch_Rate * Chord_Wing /
                           Float'Max(2.0 * TAS,0.01);

      An_Instance.The_Nondimensional_Angular.Yaw   :=
          Span_Wing * (An_Instance.The_Yaw_Rate * Cos_AOA
                       - An_Instance.The_Roll_Rate * Sin_AOA) /
                             Float'Max(2.0 * TAS,0.01);

   end Set_Nondimensional_Angular;

   procedure Assign_Nondimensional_Angular
     (ND_Angular_Rate   :in     Coordinate_Types.Attitude;
      An_Instance       :in out Instance) is
   begin
      An_Instance.The_Nondimensional_Angular := ND_Angular_Rate;
    end Assign_Nondimensional_Angular;

   function Get_Nondimensional_Angular(An_Instance :in Instance)
      return Coordinate_Types.Attitude is
   begin
      return An_Instance.The_Nondimensional_Angular;
   end Get_Nondimensional_Angular;

   function direction (An_Instance :in Instance)
      return Angle_Types.Radians is
   begin
      return an_Instance.Direction;
   end Direction;
end Velocity;



