-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Integration_G;
with Limit_G;

package body Location is

   package  Lat_Long_Integration is new Integration_G (Variable_Type => Long_Float);
   package  Altitude_Integration is new Integration_G (Variable_Type => Float);

   procedure Set_North(North_vel     :in     Length_Types.Feet_per_Sec;
                       Old_North_vel :in     Length_Types.Feet_per_Sec;
                       Dt            :in     Float;
                       An_Instance   :in out Instance) is
      Temp : Long_Float := 0.0;
   begin -- 364566 ft = 1 deg latitude - convert ft/sec to deg/sec
      An_Instance.The_Last_Pass_North := An_Instance.The_North;
      Temp := Lat_Long_Integration.Corrector
        (An_Instance.The_North, Long_Float(North_vel/364566.0),
         Long_Float(Old_North_vel/364566.0), Long_Float(Dt));

      if Temp > 90.0 then
         An_Instance.The_North := 180.0 - Temp;
      elsif Temp < -90.0 then
         An_Instance.The_North := -(180.0 + Temp);
      else
         An_Instance.The_North := Temp;
      end if;

   end Set_North;

   procedure Set_Slew_North(Slew         :in     Length_Types.Feet_per_Sec;
                            An_Instance  :in out Instance) is
      Temp : Long_Float := 0.0;
   begin
      An_Instance.The_Last_Pass_North := An_Instance.The_North;
      Temp := An_Instance.The_North + Long_Float(Slew);
      if Temp > 90.0 then
         An_Instance.The_North := 180.0 - Temp;
      elsif Temp < -90.0 then
         An_Instance.The_North := -(180.0 + Temp);
      else
         An_Instance.The_North := temp;
      end if;
   end Set_Slew_North;

   procedure Assign_North (North       :in Lat_Long_Types.Latitude_Degrees;
                           old_North   :in     Lat_Long_Types.Latitude_Degrees;
                           An_Instance :in out Instance) is
   begin
      An_Instance.The_Last_Pass_North := old_North;
      An_Instance.The_North           := North;
   end Assign_North;

   function Get_North(An_Instance :in Instance) return Lat_Long_Types.Latitude_Degrees is
   begin
      return An_Instance.The_North;
   end Get_North;

   procedure Set_East(East_vel     :in     Length_Types.Feet_per_Sec;
                      Old_East_vel :in     Length_Types.Feet_per_Sec;
                      Dt           :in     Float;
                      An_Instance  :in out Instance) is
      Temp : Long_Float := 0.0;
   begin
      An_Instance.The_Last_Pass_East := An_Instance.The_East;
      Temp  := Lat_Long_Integration.Corrector (An_Instance.The_East,
         Long_Float(East_vel/(364566.0*Cos(float(An_Instance.The_North),360.0))),
         Long_Float(Old_East_vel/(364566.0*
         Cos(float(An_Instance.The_Last_Pass_North),360.0))),Long_Float(Dt));
      if Temp > 180.0 then
         An_Instance.The_East := -360.0 + Temp;
      elsif Temp < -180.0 then
         An_Instance.The_East := 360.0 + Temp;
      else
         An_Instance.The_East := Temp;
      end if;
   end Set_East;

   procedure Set_Slew_East(Slew         :in     Length_Types.Feet_per_Sec;
                           An_Instance  :in out Instance) is
      Temp : Long_Float := 0.0;
   begin
      Temp := An_Instance.The_East + Long_Float(Slew);
      if An_Instance.The_East > 180.0 then
         An_Instance.The_East := -360.0 + Temp;
      elsif An_Instance.The_East < -180.0 then
         An_Instance.The_East := 360.0 + Temp;
      else
         An_Instance.The_East := Temp;
      end if;
   end Set_Slew_East;

   procedure Assign_East (East        :in  Lat_Long_Types.Longitude_Degrees;
                          old_East    :in     Lat_Long_Types.Longitude_Degrees;
                          An_Instance :in out Instance) is
   begin
      An_Instance.The_Last_Pass_East := old_East;
      An_Instance.The_East           := East;
   end Assign_East;

   function Get_East(An_Instance :in Instance) return Lat_Long_Types.Longitude_Degrees is
   begin

      return An_Instance.The_East;

   end Get_East;

   procedure Set_Geometric_Altitude(z_Vel       :in     Length_Types.Feet_per_Sec;
                                    old_z_Vel   :in     Length_Types.Feet_per_Sec;
                                    Dt          :in     Float;
                                    An_Instance :in out Instance) is
   begin
      An_Instance.The_Geometric_Altitude           := Altitude_Integration.Corrector
                (float(An_Instance.The_Geometric_Altitude),
                 float(z_Vel)      ,
                 float(old_z_Vel)  ,
                 Dt);

   end Set_Geometric_Altitude;

   procedure Set_Slew_Geometric_Altitude(Slew        :in     Length_Types.Feet;
                                         An_Instance :in out Instance) is
   begin
      An_Instance.The_Geometric_Altitude := An_Instance.The_Geometric_Altitude + Slew;
   end Set_Slew_Geometric_Altitude;

   procedure Assign_Geometric_Altitude (Altitude     :in     Length_Types.Feet;
                                        An_Instance  :in out Instance) is
   begin
         An_Instance.The_Geometric_Altitude           := Altitude;
    end Assign_Geometric_Altitude;

   function Get_Geometric_Altitude(An_Instance :in Instance) return Length_Types.Feet is
      Temp_Alt : Length_Types.Feet;
   begin

      if An_Instance.The_Geometric_Altitude < -35000.0 then
         Temp_Alt := -35000.0;
      else
         Temp_Alt := An_Instance.The_Geometric_Altitude;
      end if;
      return Temp_Alt;

   end Get_Geometric_Altitude;

   function Get_Height_Above_Local_Terrain(An_Instance :in Instance) return Length_Types.Feet is
      Temp_Hgt : Length_Types.Feet;
   begin

      if An_Instance.The_Height_Above_Local_Terrain < -35000.0 then
         Temp_Hgt := -35000.0;
      else
         Temp_Hgt := An_Instance.The_Height_Above_Local_Terrain;
      end if;
      return Temp_Hgt;

   end Get_Height_Above_Local_Terrain;

   procedure Set_Height_Above_Local_Terrain (Terrain_Height  :in     Length_Types.Feet;
                                             An_Instance     :in out Instance) is
   begin
      An_Instance.The_Height_Above_Local_Terrain  := An_Instance.The_Geometric_Altitude
                                                                    - Terrain_Height;
   end Set_Height_Above_Local_Terrain;

end Location;
