------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcas)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontracors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Kimball, D., "Flight Simulation Mathematical Model of
--|            the Beech MkII Joint Primary Aircraft Training System
--|            (JPATS)", Document 133E794 Rev. B, 11 November, 1998.
--|            Flight Manual Dated April 30,1998.
--|                          AIR FORCE TO 1T-6A-1,
--|                          NAVY (NAVAIR) A1-T6AAA-NFM-100
--|            Standards from Flight Dynamics Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with JPATS_Aircraft_Body;
with JPATS_Atmosphere;
with JPATS_Auto_Test;
with JPATS_Flight_Instruments;
with JPATS_Powerplant;
with JPATS_Secondary_Flight_Controls;
with JPATS_Simulated_Aircraft_Types;
with JPATS_Simulated_Aircraft.Container;
with JPATS_Simulated_Aircraft.IOS_Interface;
with JPATS_Simulated_Aircraft.Io_Interface;
with JPATS_Visual_Buffer;
with Angle_Types;
with Coordinate_Types;
with Force_Types;
with Length_Types;
with Mass_Types;
with Vector_Types;
with Normalized_Types;
with Jpats_Io_Types;
with Temperature_Types;
with Io_Types;
with Jpats_Propeller_Types;
with Jpats_Reposition_Types;

with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Save_Restore;
with Simulation_Dictionary;

with Acceleration;
with Aerodynamic_Angle;
with Aircraft_Weight;
with Airspeed;
with Attitude_Angle;
with Location;
with Moment_Of_Inertia;
with Reference_Coordinates;
with Quaternion;
with Velocity;

with JPATS_IOS_Interface;
with JPATS_IO;
with JPATS_Radio_DB_IF;
with JPATS_Visual_Airport;
with JPATS_Propeller;
with JPATS_DCLS;
with JPATS_Landing_Gear;
with JPATS_Reposition;
with JPATS_Formation_Manager;
with JPATS_Environmental;
with JPATS_Electrical;
with JPATS_Hydraulics;
with JPATS_IOS_Pilot;

package body JPATS_Simulated_Aircraft.Controller is
   -- -------------------------------------------------------------------------------------------------
   --| Package renaming
   -- -------------------------------------------------------------------------------------------------
   package IT    renames Interpolation_Table;
   package JAB   renames JPATS_Aircraft_Body;
   package JSAT  renames Jpats_Simulated_Aircraft_Types;
   package JAT   renames JPATS_Auto_test;
   package CNT   renames JPATS_Simulated_Aircraft.Container;
   package IOS   renames JPATS_Simulated_Aircraft.IOS_Interface;
   package IO    renames JPATS_Simulated_Aircraft.Io_Interface;
   package JIP   renames JPATS_IOS_Pilot;
   Pi : constant := Ada.Numerics.Pi;
   Spwing              : constant Float := 33.42;
   Chord_Wing          : constant Float := 5.413;

   -- -------------------------------------------------------------------------------------------------
   --| Declare Lookup table (Stall Buffet)
   -- -------------------------------------------------------------------------------------------------
   SABDS_Table   : aliased IT.Doubly_Indexed.Instance;


   procedure Initialize is

      -- Point to data table directory
      File_Path : String := Simulation_Dictionary.Lookup("Simulated_Aircraft_Dir");

   begin

      Quaternion.Set_Quat_From_Angles
        (0.0    ,
         0.035  ,
         2.24   ,
         An_Instance  => CNT.This_Subsystem.The_Quaternion);

      -- Load data table
      It.Read(File_Path & "SABDS.ito",SABDS_Table );

      IO.Register_Io_Variables;
      IOS.Register_Ios_Variables;  -- software variables
   end Initialize;


   procedure Update(Dt: in Float) is
      Force                      : Coordinate_Types.Cartesian;
      Moment                     : Coordinate_Types.Cartesian;
      Inertia                    : Coordinate_Types.Inertia_Axis;
      Angular_Rate               : Coordinate_Types.Attitude;
      TAS                        : Length_Types.Feet_Per_Sec;
      AOA                        : Angle_Types.Radians;
      Roll_Rate                  : Angle_Types.Radians_Per_Sec:= 0.0;
      Pitch_Rate                 : Angle_Types.Radians_Per_Sec:= 0.0;
      Yaw_Rate                   : Angle_Types.Radians_Per_Sec:= 0.0;
      CAS                        : Length_Types.Knots := 0.0;
      Stall_Speed                : Length_Types.Knots;
      Int_Time                   : Float:= 0.0;
      Roll_Angle                 : Angle_Types.Radians:= 0.0;
      Pitch_Angle                : Angle_Types.Radians:= 0.0;
      Hdg_Angle                  : Angle_Types.Radians:= 0.0;
      Mass                       : Mass_Types.Slugs_Per_Cubic_Feet;
      V_B                        : Coordinate_Types.Cartesian;
      Air_Dens                   : Mass_Types.Slugs_Per_Cubic_Feet;
      Press_Ratio                : Normalized_Types.Normalize;
      Temperature                : Temperature_Types.Rankine;
      Freeze_Time                : Float:=0.0;
      GW                         : Mass_Types.Lbm;
      Side_Slip_Angle            : Angle_Types.Radians;
      Vel_Ea                     : Coordinate_Types.Cartesian;
      North_Wind                 : Length_Types.Feet_Per_Sec;
      East_Wind                  : Length_Types.Feet_Per_Sec;
      Vert_Wind                  : Length_Types.Feet_Per_Sec;
      Accel_Body_Axis            : Coordinate_Types.Cartesian;
      Nz                         : Float  := 0.0;
      Roll_Accel                 : Angle_Types.Radians_Per_Sq_Sec:= 0.0;
      Old_Roll_Accel             : Angle_Types.Radians_Per_Sq_Sec;
      Pitch_Accel                : Angle_Types.Radians_Per_Sq_Sec := 0.0;
      Old_Pitch_Accel            : Angle_Types.Radians_Per_Sq_Sec;
      Yaw_Accel                  : Angle_Types.Radians_Per_Sq_Sec := 0.0;
      Old_Yaw_Accel              : Angle_Types.Radians_Per_Sq_Sec;
      L_3                        : Float;
      N_3                        : Float;
      M_3                        : Float;
      Flap_Position               : Angle_Types.Degrees:=0.0;
      L_1                         : Float;
      L_2                         : Float;

      Slew                        : Float := 0.0;
      Accel_Earth_Axis            : Coordinate_Types.Cartesian;
      Old_Accel_Ea                : Coordinate_Types.Cartesian;
      Variable_Slew_Rate          : Float := 0.0;
      Slew_Rate                   : Float := 0.0;
      Stopped                     : Boolean;
      Alt                         : Float := 0.0;
      Sign                        : Float := 1.0;
      Mag_Var                     : Float := 0.0;
      Mag_Hdg                     : Float := 0.0;
      North_Vel                   : Length_Types.Feet_Per_Sec := 0.0;
      East_Vel                    : Length_Types.Feet_Per_Sec := 0.0;
      Vert_Vel                    : Length_Types.Feet_Per_Sec := 0.0;
      P_Alt                       : Length_Types.Feet := 0.0;
      G_Alt                       : Length_Types.Feet := 0.0;
      P_Rat                       : Float := 1.0;
      Spd_Check                   : Boolean := False;
      Old_Vel_Ea                  : Coordinate_Types.Cartesian;
      On_Jacks                    : Boolean;
      V_Slew                      : Float;
      Pos_Slew                    : Boolean;
      Slew_N                      : Float := 0.0;
      Slew_E                      : Float := 0.0;
      Prop_Force                  : JPATS_Propeller_Types.Force_Vector_Type.Instance;
      Prop_Moment                 : JPATS_Propeller_Types.Torque_Vector_Type.Instance;
      HAT                         : Length_Types.Feet := 0.0;
      Repo_Lat                    : Long_Float;
      Repo_Long                   : Long_Float;
      Repo_Alt                    : Float;
      Repo_Hdg                    : Float;
      Repo_Airspd                 : Float;
      Phase                       : Integer := 0;
   begin

      Press_Ratio    := JPATS_Atmosphere.Pressure_Ratio;
      Temperature    := JPATS_Atmosphere.Temperature;
      Air_Dens       := JPATS_Atmosphere.Density;
      CNT.This_Subsystem.Terrain_Height := - JPATS_Visual_Buffer.Average_Terrain_Height;
      North_Wind     := JPATS_Atmosphere.North_Wind;
      East_Wind      := JPATS_Atmosphere.East_Wind;
      Vert_Wind      := JPATS_Atmosphere.Vertical_Wind;
      Stopped        := Jab.Get_Stand_Still;
      Phase          := Jat.At_Phase;
      Hat            := Location.Get_Height_Above_Local_Terrain(An_Instance => CNT.This_Subsystem.The_Location);

      -- ----------------------------------------------------------------------------------------------
      --|  Skip entire module if reposition is initialilizing in order to not
      --|  compute ridiculous numbers while replay is loading data
      -- ----------------------------------------------------------------------------------------------

      if not Save_Restore.Replay_Trimming then

         -- -------------------------------------------------------------------------------------------
         --| if autotest is on initialize with Auto Test values
         -- -------------------------------------------------------------------------------------------
         if Phase /= 9 then
            --| not if reposition trim
            if Phase > 0 then
               CNT.This_Subsystem.Terrain_Height := - Jat.Field_Alt;
            end if;

            if Phase = 1 or Phase = 2 then
               Repo_Lat := Long_Float(Jat.Latitude);
               Repo_Long := Long_Float(Jat.Longitude);
               Location.Assign_North (Repo_Lat,Repo_Lat,
                                      An_Instance => CNT.This_Subsystem.The_Location);
               Location.Assign_East (Repo_Long, Repo_Long,
                                     An_Instance => CNT.This_Subsystem.The_Location);
            end if;

            if Phase = 1 or Phase = 2 or Phase > 5 then

               --|  use auto_test pressure alt to compute pressure ratio
               P_Alt := Jat.Altitude;
               P_Rat := (1.0 - (P_Alt/145448.0))** 5.2581;

               --| compute geometric alt from pressure ratio & sl press
               G_Alt := (1.0 - P_Rat ** 0.19018) * 145448.0 * 29.92/JPATS_Atmosphere.Sl_Pressure;

               Location.Assign_Geometric_Altitude
                 (-G_Alt,
                  An_Instance => CNT.This_Subsystem.The_Location);

               if Jat.Ground_Reset then
                  G_Alt := Jat.Field_Alt - JSAT.The_Normal_Height_On_Ground; -- + 4.5;
                  Location.Assign_Geometric_Altitude
                    (-G_Alt,
                     An_Instance => CNT.This_Subsystem.The_Location);
               end if;

               Hat := Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location)
                 - CNT.This_Subsystem.Terrain_Height;

               AOA  := Jat.AoA;
               Aerodynamic_Angle.Assign_Angle_Of_Attack
                 (AOA,
                  An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);

               Side_Slip_Angle   := Jat.Beta;
               Aerodynamic_Angle.Assign_Side_Slip_Angle
                 (Side_Slip_Angle,
                  An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);

               Roll_Angle    := Jat.Roll_Angle;
               Pitch_Angle   := Jat.Pitch_Angle;
               Hdg_Angle     := Jat.Heading;

               Attitude_Angle.Assign_Roll
                 (Roll_Angle,
                  An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
               Attitude_Angle.Assign_Pitch
                 (Pitch_Angle,
                  An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
               Attitude_Angle.Assign_Hdg
                 (Hdg_Angle,
                  An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

               Quaternion.Set_Quat_From_Angles
                 (Roll_Angle,
                  Pitch_Angle,
                  Hdg_Angle,
                  An_Instance  => CNT.This_Subsystem.The_Quaternion);

               Roll_Rate  := Jat.Roll_Rate * 0.01745;
               Pitch_Rate := Jat.Pitch_Rate * 0.01745;
               Yaw_Rate   := Jat.Yaw_Rate * 0.01745;

               Velocity.Assign_Yaw_Rate
                 (Yaw_Rate,
                  An_Instance  => CNT.This_Subsystem.The_Velocity);
               Velocity.Assign_Pitch_Rate
                 (Pitch_Rate,
                  An_Instance  => CNT.This_Subsystem.The_Velocity);
               Velocity.Assign_Roll_Rate
                 (Roll_Rate,
                  An_Instance  => CNT.This_Subsystem.The_Velocity);

               CAS := Jat.CAS;
               Airspeed.Assign_Calibrated_Airspeed
                 (CAS,
                  An_Instance => CNT.This_Subsystem.The_Airspeed);
               Airspeed.Set_Mach_Number_Based_On_Calibrated_Airspeed
                 (Air_Dens,
                  An_Instance => CNT.This_Subsystem.The_Airspeed);
               Airspeed.Set_True_Airspeed_Based_On_Mach_Number
                 (Temperature,
                  An_Instance => CNT.This_Subsystem.The_Airspeed);

               TAS := Airspeed.Get_True_Airspeed (CNT.This_Subsystem.The_Airspeed);
               Side_Slip_Angle    := Aerodynamic_Angle.Get_Side_Slip_Angle
                 (CNT.This_Subsystem.The_Aerodynamic_Angle);
               AOA    := Aerodynamic_Angle.Get_Angle_Of_Attack
                 (CNT.This_Subsystem.The_Aerodynamic_Angle);
               Velocity.Set_Body_Axis_Based_On_True_Airspeed
                 (TAS,
                  Aoa,
                  Side_Slip_Angle,
                  An_Instance => CNT.This_Subsystem.The_Velocity);

               --| remove wind during reset if weight on wheels (alt < 10.0)
               --| zero wind if reposition to ground
               if Hat > -10.0 and Cas < 60.0 then
                  North_Wind := 0.0;
                  East_Wind := 0.0;
                  Vert_Wind := 0.0;
               end if;

               --| remove wind to get body to earth during reposition
               Velocity.Set_Body_To_Earth_Axis
                 (Roll_Angle     ,
                  Pitch_Angle    ,
                  Hdg_Angle      ,
                  - North_Wind   ,
                  - East_Wind    ,
                  - Vert_Wind    ,
                  An_Instance => CNT.This_Subsystem.The_Velocity);

               Roll_Rate      := Velocity.Get_Roll_Rate(CNT.This_Subsystem.The_Velocity);
               Pitch_Rate     := Velocity.Get_Pitch_Rate(CNT.This_Subsystem.The_Velocity);
               Yaw_Rate       := Velocity.Get_Yaw_Rate(CNT.This_Subsystem.The_Velocity);
               Airspeed.Calc_Dynamic_Pressure
                 (Air_Dens,
                  TAS ,
                  An_Instance => CNT.This_Subsystem.The_Airspeed);

               Velocity.Set_Nondimensional_Angular
                 (SpWing       ,
                  Chord_Wing   ,
                  TAS          ,
                  Aoa          ,
                  An_Instance => CNT.This_Subsystem.The_Velocity);
            elsif Phase = 5 then
               IOS.This_Ios_Interface.System_Reset := True;
            end if;  -- end autotest ic
         end if;  -- end not phase 9
         -- *******************************************************************************************

         Mag_Var         := JPATS_Radio_DB_IF.Magnetic_Variation;
         Moment          := JAB.Get_Moment;
         Angular_Rate    := (Velocity.Get_Roll_Rate(CNT.This_Subsystem.The_Velocity) ,
                             Velocity.Get_Pitch_Rate(CNT.This_Subsystem.The_Velocity),
                             Velocity.Get_Yaw_Rate(CNT.This_Subsystem.The_Velocity));
         Inertia         := Moment_Of_Inertia.Get_Inertia(CNT.This_Subsystem.The_Moment_Of_Inertia);

         Flap_Position   := JPATS_Secondary_Flight_Controls.Mean_Flap_Position;
         Gw              := JPATS_Simulated_Aircraft.Gw;
         Mass            := Aircraft_Weight.Get_Mass(CNT.This_Subsystem.The_Aircraft_Weight);
         Force           := JAB.Get_Force;
         Nz              := - Acceleration.Load_Factor(CNT.This_Subsystem.The_Acceleration).Z;


         -- -------------------------------------------------------------------------------------------
         --| process reposition if commanded: trim will be done
         --| by jpats auto test which will turn on with phase set to 9
         -- -------------------------------------------------------------------------------------------

         if JPATS_Reposition.Reposition_Command.Active or CNT.This_Subsystem.Repo_Active then
            --Reposition_active then
            Repo_Lat :=JPATS_Reposition.Reposition_Command.Position.Latitude;
            Repo_Long := JPATS_Reposition.Reposition_Command.Position.Longitude;
            Repo_Alt := -JPATS_Reposition.Reposition_Command.Position.Altitude; -- alt negative
            Repo_Hdg := JPATS_Reposition.Reposition_Command.Heading/57.296;
            Repo_Airspd := JPATS_Reposition.Reposition_Command.Airspeed;
            Location.Assign_Geometric_Altitude
              (Repo_Alt,
               An_Instance => CNT.This_Subsystem.The_Location);
            Location.Assign_North
              (Repo_Lat,Repo_Lat,
               An_Instance => CNT.This_Subsystem.The_Location);
            Location.Assign_East
              (Repo_Long, Repo_Long,
               An_Instance => CNT.This_Subsystem.The_Location);

            if Repo_Airspd > 60.0 then
               Pitch_Angle := Jat.Pitch_Angle;
            else
               Pitch_Angle := 0.0;
            end if;

            Roll_Angle := JPATS_Reposition.Reposition_Command.Roll_Angle;

            if JPATS_Reposition.Reposition_Command.Pitch_Angle /= 0.0 then
               Pitch_Angle := JPATS_Reposition.Reposition_Command.Pitch_Angle;
            end if;

            Hdg_Angle     := Repo_Hdg;
            Attitude_Angle.Assign_Roll
              (Roll_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
            Attitude_Angle.Assign_Pitch
              (Pitch_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
            Attitude_Angle.Assign_Hdg
              (Hdg_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

            Quaternion.Set_Quat_From_Angles
              (Roll_Angle ,
               Pitch_Angle,
               Hdg_Angle  ,
               An_Instance => CNT.This_Subsystem.The_Quaternion);

            Airspeed.Assign_Calibrated_Airspeed
              (Repo_Airspd,
               An_Instance => CNT.This_Subsystem.The_Airspeed);
            Airspeed.Set_Mach_Number_Based_On_Calibrated_Airspeed
              (Air_Dens,
               An_Instance => CNT.This_Subsystem.The_Airspeed);
            Airspeed.Set_True_Airspeed_Based_On_Mach_Number
              (Temperature,
               An_Instance => CNT.This_Subsystem.The_Airspeed);
            TAS := Airspeed.Get_True_Airspeed (CNT.This_Subsystem.The_Airspeed);
            Aerodynamic_Angle.Assign_Angle_Of_Attack
              (Jat.Aoa,
               An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);
            Aerodynamic_Angle.Assign_Side_Slip_Angle
              (Jat.Beta,
               An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);
            Aerodynamic_Angle.Assign_Angle_Of_Attack_Rate
              (0.0,
               An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);
            Velocity.Set_Body_Axis_Based_On_True_Airspeed
              (TAS,
               Jat.Aoa,
               Jat.Beta ,-- sideslip
               An_Instance => CNT.This_Subsystem.The_Velocity);

            Velocity.Assign_Yaw_Rate
              (0.0,
               An_Instance  => CNT.This_Subsystem.The_Velocity);
            Velocity.Assign_Pitch_Rate
              (0.0,
               An_Instance  => CNT.This_Subsystem.The_Velocity);
            Velocity.Assign_Roll_Rate
              (0.0,
               An_Instance  => CNT.This_Subsystem.The_Velocity);
            --| zero wind if reposition to ground
            if Hat > -10.0 then
               North_Wind := 0.0;
               East_Wind := 0.0;
               Vert_Wind := 0.0;
            end if;
            --| remove wind to get body to earth during reposition
            Velocity.Set_Body_To_Earth_Axis
              (0.0          , -- roll
               Jat.Aoa         , -- pitch
               Repo_Hdg     ,
               - North_Wind  ,
               - East_Wind   ,
               - Vert_Wind   ,
               An_Instance  => CNT.This_Subsystem.The_Velocity);

            IOS.This_Ios_Interface.The_Flight_Freeze := True;

            CNT.This_Subsystem.Repo_Active := True;
            IOS.This_Ios_Interface.Trim_Request := True; -- flag for autotest to retrim

            if not JPATS_Reposition.Reposition_Command.Active then
               if Phase /= 9 then
                  CNT.This_Subsystem.Repo_Active := False;
                  IOS.This_Ios_Interface.Trim_Request := False;
                  -- Reset roll angle
                  JPATS_Reposition.Set_Roll_Angle(0.0);
                  -- Reset pitch angle
                  JPATS_Reposition.Set_Pitch_Angle(0.0);
               end if;
            end if;


            -- ----------------------------------------------------------------------------------------
            --|  if altitude slew from on ground set airspeed (request flag set in altitude slew)
            --|  set airspeed to 140 knots if reposition from stopped on ground and retrim
            --|  compute and assign earth and body axis velocities corresponding to 140 knots
            -- ----------------------------------------------------------------------------------------
         elsif CNT.This_Subsystem.Set_Cas then
            Airspeed.Assign_Calibrated_Airspeed
              (140.0,
               An_Instance => CNT.This_Subsystem.The_Airspeed);

            Airspeed.Set_Mach_Number_Based_On_Calibrated_Airspeed
              (Air_Dens,
               An_Instance => CNT.This_Subsystem.The_Airspeed);
            Airspeed.Set_True_Airspeed_Based_On_Mach_Number
              (Temperature,
               An_Instance => CNT.This_Subsystem.The_Airspeed);
            TAS := Airspeed.Get_True_Airspeed (CNT.This_Subsystem.The_Airspeed);
            Aerodynamic_Angle.Assign_Angle_Of_Attack
              (Jat.Aoa,
               An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);
            Aerodynamic_Angle.Assign_Side_Slip_Angle
              (Jat.Beta,
               An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);
            Aerodynamic_Angle.Assign_Angle_Of_Attack_Rate
              (0.0,
               An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);
            Velocity.Set_Body_Axis_Based_On_True_Airspeed
              (TAS,
               Jat.Aoa,
               Jat.Beta,
               An_Instance => CNT.This_Subsystem.The_Velocity);

            Roll_Angle    := 0.0;
            Pitch_Angle   := Jat.Pitch_Angle;
            Attitude_Angle.Assign_Roll
              (Roll_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
            Attitude_Angle.Assign_Pitch
              (Pitch_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

            --| no change to heading during airspeed/altitude slew
            Hdg_Angle   := Attitude_Angle.Get_Hdg(CNT.This_Subsystem.The_Attitude_Angle);

            Quaternion.Set_Quat_From_Angles
              (Roll_Angle ,
               Pitch_Angle,
               Hdg_Angle  ,
               An_Instance => CNT.This_Subsystem.The_Quaternion);

            Velocity.Set_Body_To_Earth_Axis
              (Roll_Angle  ,
               Pitch_Angle ,
               Hdg_Angle   ,
               North_Wind  ,
               East_Wind   ,
               Vert_Wind   ,
               An_Instance  => CNT.This_Subsystem.The_Velocity);
            Vel_Ea  := Velocity.Get_Earth_Axis(CNT.This_Subsystem.The_Velocity);
            IOS.This_Ios_Interface.The_Flight_Freeze := True;

            -- ----------------------------------------------------------------------------------------
            --| trim request (set below) will cause autotest to set phase to 9
            --| when autotest trim is complete phase will be set back to zero
            --| this will allow set_cas flag to clear.  use as_count to latch
            --| set_cas until until autotest phase changes
            -- ----------------------------------------------------------------------------------------
            CNT.This_Subsystem.As_Count := CNT.This_Subsystem.As_Count + 1;
            if CNT.This_Subsystem.As_Count >= 5  then
               if Phase /= 9 then
                  CNT.This_Subsystem.Set_Cas := False;
                  CNT.This_Subsystem.As_Count := 0;
                  IOS.This_Ios_Interface.Trim_Request := False;
               end if;
            end if;

         end if;  -- end set airspeed


         -- -------------------------------------------------------------------------------------------
         --| Update Crash; Read crash override button at 1 hz and latch
         -- -------------------------------------------------------------------------------------------

         CNT.This_Subsystem.Latch_Count := CNT.This_Subsystem.Latch_Count + 1;
         if CNT.This_Subsystem.Latch_Count > 60 then
            CNT.This_Subsystem.Latch_Count := 60;
            if IO.This_Io_Interface.The_Crash_Override_On then
               CNT.This_Subsystem.Override_Latch := not CNT.This_Subsystem.Override_Latch;
               CNT.This_Subsystem.Latch_Count := 0;
            end if;
         end if;

         if CNT.This_Subsystem.Override_Latch then
            IO.This_Io_Interface.The_Crash_Override_Amber := True;
            IO.This_Io_Interface.The_Crash_Override_Green := False;
         else
            IO.This_Io_Interface.The_Crash_Override_Amber := False;
            IO.This_Io_Interface.The_Crash_Override_Green := True;
         end if;



         --| Sink Rate
         -- -------------------------------------------------------------------------------------------
         if (Velocity.Get_Earth_Axis(CNT.This_Subsystem.The_Velocity).Z > 12.5) and
           Jab.Get_WOW then
            IOS.This_IOs_Interface.The_Sink_Rate_Crash := True;
         end if;

         --|  Over G
         -- -------------------------------------------------------------------------------------------
         Flap_Position := JPATS_Secondary_Flight_Controls.Mean_Flap_Position;
         if not Jab.Get_Wow then
            if (((Nz > 2.5) or (Nz <  0.0)) and (Flap_Position > 1.0)) or
              (((Nz > 7.0) or (Nz < -3.5)) and (Flap_Position = 0.0)) then
                 IOS.This_IOs_Interface.The_Vertical_Load_Crash := True;
            end if;
         end if;

         --| Overspeed
         -- -------------------------------------------------------------------------------------------
         if (JPATS_Flight_Instruments.Primary_Ias >
               (JSAT.Maximum_Operating_Airspeed + 15.0)) then    --| in Knots
              IOS.This_IOs_Interface.The_Excess_Airspeed_Crash := True;
         end if;

         --| Terrain Impact
         -- -------------------------------------------------------------------------------------------
         if (JPATS_Visual_Buffer.Fuselage_Collision and
             not CNT.This_Subsystem.The_Fuselage_Collision_Last_Pass and
             not IOS.This_IOs_Interface.The_Terrain_Impact_Crash) then
            CNT.This_Subsystem.The_Fuselage_Collision_Counter :=
               CNT.This_Subsystem.The_Fuselage_Collision_Counter + 1;
         end if;

         if (JPATS_Visual_Buffer.Left_Wing_Collision and
             not CNT.This_Subsystem.The_Left_Wing_Collision_Last_Pass and
             not IOS.This_IOs_Interface.The_Terrain_Impact_Crash) then
            CNT.This_Subsystem.The_Left_Wing_Collision_Counter :=
               CNT.This_Subsystem.The_Left_Wing_Collision_Counter + 1;
         end if;

         if (JPATS_Visual_Buffer.Right_Wing_Collision and
             not CNT.This_Subsystem.The_Right_Wing_Collision_Last_Pass and
             not IOS.This_IOs_Interface.The_Terrain_Impact_Crash) then
            CNT.This_Subsystem.The_Right_Wing_Collision_Counter :=
               CNT.This_Subsystem.The_Right_Wing_Collision_Counter + 1;
         end if;

         if (Jab.Get_Tail_Wow and Jab.Get_Wow) or
           Jab.Crash_Wingtip_Impact or
           CNT.This_Subsystem.The_Fuselage_Collision_Counter > 2 or
           CNT.This_Subsystem.The_Right_Wing_Collision_Counter > 2 or
           CNT.This_Subsystem.The_Left_Wing_Collision_Counter > 2  or
           -- Raise terrain impact in case of WOW with gears up
           (Jab.Get_Wow and
            (JPATS_Landing_Gear.Left_Landing_Gear_Position < 0.5) and
            (JPATS_Landing_Gear.Right_Landing_Gear_Position < 0.5))
         then
            IOS.This_Ios_Interface.The_Terrain_Impact_Crash := True;
         end if;

         --|  Turn off all crashes if  in autotest, reposition or snapshot restore
         -- -------------------------------------------------------------------------------------------
         if Phase > 0 or
           JPATS_Reposition.Reposition_Command.Active  or
           Misc_Freeze_Request then
            IOS.This_IOs_Interface.The_Terrain_Impact_Crash := False;
            IOS.This_IOs_Interface.The_Excess_Airspeed_Crash := False;
            IOS.This_IOs_Interface.The_Vertical_Load_Crash := False;
            IOS.This_IOs_Interface.The_Sink_Rate_Crash := False;
            CNT.This_Subsystem.The_Fuselage_Collision_Counter := 0;
            CNT.This_Subsystem.The_Left_Wing_Collision_Counter := 0;
            CNT.This_Subsystem.The_Right_Wing_Collision_Counter := 0;
         end if;


         --|  Do not turn off terrain impact crashes if  crash override
         -- -------------------------------------------------------------------------------------------
         if CNT.This_Subsystem.Override_Latch then
            IOS.This_IOs_Interface.The_Excess_Airspeed_Crash := False;
            IOS.This_IOs_Interface.The_Vertical_Load_Crash := False;
            IOS.This_IOs_Interface.The_Sink_Rate_Crash := False;
         end if;

         IOS.This_Ios_Interface.Crashed :=
           IOS.This_Ios_Interface.The_Terrain_Impact_Crash or
           IOS.This_Ios_Interface.The_Excess_Airspeed_Crash or
           IOS.This_Ios_Interface.The_Vertical_Load_Crash or
           IOS.This_Ios_Interface.The_Sink_Rate_Crash;

         if  IOS.This_Ios_Interface.Crashed then
            IOS.This_Ios_Interface.The_Flight_Freeze := True;
         end if;
         --|  *****************************************************************************************


         On_Jacks := IOS.This_Ios_Interface.Jacks and CNT.This_Subsystem.Ground_Speed < 0.1 ;
         if On_Jacks then -- wheels just off ground and in freeze
              Location.Assign_Geometric_Altitude
                (CNT.This_Subsystem.Terrain_Height + JSAT.The_Normal_Height_On_Ground - 1.0, -- - 5.25 ,
                 An_Instance => CNT.This_Subsystem.The_Location);
         end if;
         Stopped := Stopped or
           (JPATS_Landing_Gear.Wheel_Chocks and
            JPATS_Aircraft_Body.Get_Wow and CNT.This_Subsystem.Ground_Speed < 0.25) or
           (Jat.Chocked and Phase > 0);

         if  JPATS_Formation_Manager.Flight_Freeze_Request or
           JPATS_Environmental.Freeze_Req or
           Misc_Freeze_Request  then
            IOS.This_Ios_Interface.The_Flight_Freeze := True;
         end if;


         --| Misc_freeze_request is a request from restoring snapshot process to freeze the system
         --| after the snapshot has been restored.
         --| It is important to override the crash logic since restoring snapshot introduce sudden
         --| jumps.
         --| This variable is reset to false after 359 frames (6 s).
         if Misc_Freeze_Request then
            Misc_Freeze_Request_Timer := Misc_Freeze_Request_Timer + 1;
            if Misc_Freeze_Request_Timer > 359 then
               Misc_Freeze_Request := False;
               Misc_Freeze_Request_Timer := 0;
            end if;
         end if;
         --********************************************************************************************


         -- -------------------------------------------------------------------------------------------
         --|  Update Freeze information
         -- -------------------------------------------------------------------------------------------
         if IO.This_Io_Interface.Total_Freeze_On
           and not CNT.This_Subsystem.Total_Freeze_On_Last_Pass  then
            IOS.This_Ios_Interface.The_Flight_Freeze := not IOS.This_Ios_Interface.The_Flight_Freeze;
         end if;

         CNT.This_Subsystem.Total_Freeze_On_Last_Pass := IO.This_Io_Interface.Total_Freeze_On;

         IO.This_Io_Interface.Total_Amber := IOS.This_Ios_Interface.The_Flight_Freeze;
         IO.This_Io_Interface.Total_Green := not IOS.This_Ios_Interface.The_Flight_Freeze;

         if (Phase > 0 and Phase < 4) or
           (Jat.Test_Number >= 441 and Jat.Test_Number <= 453) or
           On_Jacks  or Phase > 4 then
            IOS.This_Ios_Interface.The_Flight_Freeze := True;
         elsif Phase = 4 then
            IOS.This_Ios_Interface.The_Flight_Freeze := False;
         end if;

         --|  Use zero for integration time to freeze heading  (only on heading axis)
         if IOS.This_Ios_Interface.The_Flight_Freeze then
            Int_Time    := 0.0;
            Freeze_Time := 0.0;
         elsif IOS.This_Ios_Interface.The_Heading_Freeze then
            Freeze_Time := 0.0;
            Int_Time    := Dt;
         else
            Int_Time    := Dt;
            Freeze_Time := Dt;
         end if;



         -- -------------------------------------------------------------------------------------------
         --|    Compute angular acceleration and Update Angular Rates
         -- -------------------------------------------------------------------------------------------
         Acceleration.Calc_Ang_Acc
           (Moment , Inertia , Angular_Rate,
            An_Instance => CNT.This_Subsystem.The_Acceleration);

         Roll_Accel := Acceleration.Get_Angular
           (CNT.This_Subsystem.The_Acceleration).Roll;
         Old_Roll_Accel := Acceleration.Get_Last_Pass_Angular
           (CNT.This_Subsystem.The_Acceleration).Roll;
         Pitch_Accel :=  Acceleration.Get_Angular
           (CNT.This_Subsystem.The_Acceleration).Pitch;
         Old_Pitch_Accel := Acceleration.Get_Last_Pass_Angular
           (CNT.This_Subsystem.The_Acceleration).Pitch;
         Yaw_Accel := Acceleration.Get_Angular
           (CNT.This_Subsystem.The_Acceleration).Yaw;
         Old_Yaw_Accel :=  Acceleration.Get_Last_Pass_Angular
           (CNT.This_Subsystem.The_Acceleration).Yaw;

         Velocity.Set_Roll_Rate
           (Roll_Accel,
            Old_Roll_Accel,
            Int_Time,
            An_Instance => CNT.This_Subsystem.The_Velocity);
         Velocity.Set_Pitch_Rate
           (Pitch_Accel,
            Old_Pitch_Accel,
            Int_Time,
            An_Instance => CNT.This_Subsystem.The_Velocity);
         Velocity.Set_Yaw_Rate
           (Yaw_Accel,
            Old_Yaw_Accel,
            Int_Time,
            An_Instance => CNT.This_Subsystem.The_Velocity);

         if Stopped then
            --| zero heading rate to stop drift when standing still on ground
            Velocity.Assign_Yaw_Rate
              (0.0,
               An_Instance  => CNT.This_Subsystem.The_Velocity);
         end if;


         -- -------------------------------------------------------------------------------------------
         --|   Calculate quaternion rates based on euler rates
         -- -------------------------------------------------------------------------------------------
         Roll_Rate  := Velocity.Get_Roll_Rate(CNT.This_Subsystem.The_Velocity);
         Pitch_Rate := Velocity.Get_Pitch_Rate(CNT.This_Subsystem.The_Velocity);
         Yaw_Rate   := Velocity.Get_Yaw_Rate(CNT.This_Subsystem.The_Velocity);

         Quaternion.Set_Quaternion_Rate
           (Roll_Rate ,
            Pitch_Rate,
            Yaw_Rate  ,
            An_Instance => CNT.This_Subsystem.The_Quaternion);

         --|  Integrate quaternion rate to get new quaternions and compute direction cosines
         Quaternion.Set_Quaternion
           (Int_Time,
            Freeze_Time,
            An_Instance  => CNT.This_Subsystem.The_Quaternion);

         L_1  := Quaternion.Get_L_1(CNT.This_Subsystem.The_Quaternion);
         L_2  := Quaternion.Get_L_2(CNT.This_Subsystem.The_Quaternion);
         L_3  := Quaternion.Get_L_3(CNT.This_Subsystem.The_Quaternion);
         N_3  := Quaternion.Get_N_3(CNT.This_Subsystem.The_Quaternion);
         M_3  := Quaternion.Get_M_3(CNT.This_Subsystem.The_Quaternion);

         --| compute new euler angles from direction cosines
         Attitude_Angle.Set_Pitch
           (L_3,
            An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
         Attitude_Angle.Set_Roll
           (N_3,
            M_3,
            An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
         Attitude_Angle.Set_Hdg
           (L_1,
            L_2,
            An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

         --| retrieve angles which were just computed
         Roll_Angle  := Attitude_Angle.Get_Roll(An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
         Pitch_Angle := Attitude_Angle.Get_Pitch(An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
         Hdg_Angle   := Attitude_Angle.Get_Hdg(CNT.This_Subsystem.The_Attitude_Angle);


         -- -------------------------------------------------------------------------------------------
         --|    Process Slew in progress
         -- -------------------------------------------------------------------------------------------
         --| Add flashing Boolean for IOS to report slewing in progress
         CNT.This_Subsystem.Slew_In_Progress :=
           IOS.This_Ios_Interface.Slew_North
           or IOS.This_Ios_Interface.Slew_South
           or IOS.This_Ios_Interface.Slew_East
           or IOS.This_Ios_Interface.Slew_West
           or IOS.This_Ios_Interface.Slew_Fwd
           or IOS.This_Ios_Interface.Slew_Aft
           or IOS.This_Ios_Interface.Slew_Left
           or IOS.This_Ios_Interface.Slew_Right
           or CNT.This_Subsystem.Alt_Slew
           or abs(CNT.This_Subsystem.Delta_Hdg) > 0.005
           or IOS.This_Ios_Interface.Turn_Left
           or IOS.This_Ios_Interface.Turn_Right;

         if CNT.This_Subsystem.Slew_In_Progress or
           (JIP.Is_On and (not JIP.Is_Established)) or
           Save_Restore.Replay_In_Progress or
           Save_Restore.Replay_Trimming then
            CNT.This_Subsystem.Slewing_Timer := CNT.This_Subsystem.Slewing_Timer + Dt;
            if CNT.This_Subsystem.Slewing_Timer > 1.0 then
               CNT.This_Subsystem.Slewing_Timer := 0.0;
            end if;
         else
            CNT.This_Subsystem.Slewing_Timer := 0.0;
            if (JIP.Is_On and JIP.Is_Established) then
               IOS.This_Ios_Interface.Slewing := 4;
            end if;
         end if;

         --| Note that Dt = 0.0 implies update in freeze
         if CNT.This_Subsystem.Slewing_Timer > 0.5 and Dt > 0.0 then
            IOS.This_Ios_Interface.Slewing := 0;
            if (JIP.Is_On and JIP.Is_Established) then
               IOS.This_Ios_Interface.Slewing := 4;
            end if;
         elsif CNT.This_Subsystem.Slew_In_Progress and Dt > 0.0 then
            IOS.This_Ios_Interface.Slewing := 1;
         elsif Save_Restore.Replay_In_Progress then
            IOS.This_Ios_Interface.Slewing := 2;
         elsif Save_Restore.Replay_Trimming then
            IOS.This_Ios_Interface.Slewing := 3;
         elsif (JIP.Is_On ) and Dt > 0.0 then
            IOS.This_Ios_Interface.Slewing := 4;
         else
            IOS.This_Ios_Interface.Slewing := 0;
         end if;

         -- -------------------------------------------------------------------------------------------
         --| Process Heading Slew
         -- -------------------------------------------------------------------------------------------
         --| Disable heading slew for record-playback and snapshot restore
         if Save_Restore.Replay_Trimming or Save_Restore.Replay_In_Progress or
            Misc_Freeze_Request then
            CNT.This_Subsystem.Old_Cmnd_Hdg := IOS.This_Ios_Interface.Command_Heading;
         end if;

         --| check if slew just turned on or in progress
         if (abs(IOS.This_Ios_Interface.Command_Heading - CNT.This_Subsystem.Old_Cmnd_Hdg) > 0.3) then

            IOS.This_Ios_Interface.Command_Heading := IOS.This_Ios_Interface.Command_Heading + 0.4;
            CNT.This_Subsystem.Old_Cmnd_Hdg :=  IOS.This_Ios_Interface.Command_Heading;
            CNT.This_Subsystem.Delta_Hdg := (IOS.This_Ios_Interface.Command_Heading + Mag_Var) * Pi/180.0
              - Hdg_Angle;
            if CNT.This_Subsystem.Delta_Hdg > Pi then
               CNT.This_Subsystem.Delta_Hdg := CNT.This_Subsystem.Delta_Hdg - 2.0 * Pi;
            elsif CNT.This_Subsystem.Delta_Hdg < -Pi then
               CNT.This_Subsystem.Delta_Hdg := CNT.This_Subsystem.Delta_Hdg + 2.0 * Pi;
            else
               null;
            end if;

         end if;

         --| immediate change in flight freeze otherwise slew at selected rate
         if abs(CNT.This_Subsystem.Delta_Hdg) > 0.000873 or
            IOS.This_Ios_Interface.Turn_Left or
            IOS.This_Ios_Interface.Turn_Right then

            if IOS.This_Ios_Interface.Use_Variable_Slew_Rate then
               CNT.This_Subsystem.Hdg_Slew := Float(IOS.This_Ios_Interface.Variable_Slew_Rate) * 0.0000161111;
            else
               if  IOS.This_Ios_Interface.Slew_Rate = 3 then
                  CNT.This_Subsystem.Hdg_Slew := 0.0058;-- 20 deg sec
               elsif  IOS.This_Ios_Interface.Slew_Rate = 2 then
                  CNT.This_Subsystem.Hdg_Slew := 0.0029; -- 10 deg sec
               else
                  CNT.This_Subsystem.Hdg_Slew := 0.000873; -- 3 deg sec
               end if;
            end if;
            if not  (IOS.This_Ios_Interface.Turn_Left or
                     IOS.This_Ios_Interface.Turn_Right) then

               if IOS.This_Ios_Interface.The_Flight_Freeze or
                 (abs(CNT.This_Subsystem.Delta_Hdg) <= CNT.This_Subsystem.Hdg_Slew) then

                  CNT.This_Subsystem.Delta_Hdg := 0.0;
                  Hdg_Angle := (IOS.This_Ios_Interface.Command_Heading + Mag_Var) * Pi/180.0;
               else
                  if CNT.This_Subsystem.Delta_Hdg >= 0.0 then
                     Sign := 1.0;
                  else
                     Sign := -1.0;
                  end if;
                  Hdg_Angle := Hdg_Angle + CNT.This_Subsystem.Hdg_Slew * Sign;
                  CNT.This_Subsystem.Delta_Hdg :=
                    (IOS.This_Ios_Interface.Command_Heading + Mag_Var) * Pi/180.0
                    - Hdg_Angle;
                  if CNT.This_Subsystem.Delta_Hdg > Pi then
                     CNT.This_Subsystem.Delta_Hdg := CNT.This_Subsystem.Delta_Hdg - 2.0 * Pi;
                  elsif CNT.This_Subsystem.Delta_Hdg < -Pi then
                     CNT.This_Subsystem.Delta_Hdg := CNT.This_Subsystem.Delta_Hdg + 2.0 * Pi;
                  end if;
               end if;

            elsif IOS.This_Ios_Interface.Turn_Left or
              IOS.This_Ios_Interface.Turn_Right then
               --| get IOS slew rate
               if IOS.This_Ios_Interface.Turn_Left then
                  Hdg_Angle := Hdg_Angle - CNT.This_Subsystem.Hdg_Slew;
               else
                  Hdg_Angle := Hdg_Angle + CNT.This_Subsystem.Hdg_Slew;
               end if;
               IOS.This_Ios_Interface.Command_Heading := Hdg_Angle*57.3 - Mag_Var;
               CNT.This_Subsystem.Old_Cmnd_Hdg := Hdg_Angle*57.3 - Mag_Var;
               CNT.This_Subsystem.Delta_Hdg := 0.0;
            end if;
            --|  recompute quaternions based on new heading
            Quaternion.Set_Quat_From_Angles
              (Roll_Angle   ,
               Pitch_Angle  ,
               Hdg_Angle    ,
               An_Instance  => CNT.This_Subsystem.The_Quaternion);

            --|  Recompute Earth Axis Velocities for new heading
            Attitude_Angle.Assign_Hdg
              (Hdg_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

            Velocity.Set_Body_To_Earth_Axis
              (Roll_Angle     ,
               Pitch_Angle    ,
               Hdg_Angle      ,
               - North_Wind     ,
               - East_Wind      ,
               - Vert_Wind      ,
               An_Instance  => CNT.This_Subsystem.The_Velocity);
         end if;  -- end heading slew
         --  ******************************************************************************************



         -- -------------------------------------------------------------------------------------------
         --| Compute Acceleration in body axis
         --| Note that in the earth axis calculations the X & Y are really north & east
         --| in body axis +X is forward, etc
         -- -------------------------------------------------------------------------------------------
         Acceleration.Calc_Linear
           (Force      ,
            Roll_Angle ,
            Pitch_Angle,
            Hdg_Angle  ,
            Mass       ,
            Stopped    ,
            An_Instance => CNT.This_Subsystem.The_Acceleration);

         --|  transform body axis acceleration into earth axis
         Accel_Earth_Axis  := Acceleration.Get_Earth_Axis(CNT.This_Subsystem.The_Acceleration);
         Old_Accel_Ea := Acceleration.Get_Last_Pass_Earth_Axis(CNT.This_Subsystem.The_Acceleration);

         --|  integrate to get new earth axis velocities
         Velocity.Set_Earth_Axis
           (Accel_Earth_Axis,
            Old_Accel_Ea    ,
            Int_Time        ,
            Stopped         ,
            An_Instance => CNT.This_Subsystem.The_Velocity);

         --|   retrieve velocities which were just computed
         Vel_Ea      := Velocity.Get_Earth_Axis(CNT.This_Subsystem.The_Velocity);
         Old_Vel_Ea  := Velocity.Get_Last_Pass_Earth_Axis(CNT.This_Subsystem.The_Velocity);

         --| limit vert velocity near ground coming off jacks or during reset to prevent crash
         if Hat > -10.0 and abs(Vel_Ea.X) < 2.5 then
            if Vel_Ea.Z > 4.0 then
               Vel_Ea.Z := 4.0;
            elsif Vel_Ea.Z < -4.0 then
               Vel_Ea.Z := -4.0;
            end if;

            --| store z velocity if it was just changed
            Velocity.Assign_Earth_Axis
              ((Vel_Ea.X,Vel_Ea.Y,Vel_Ea.Z),
               (Vel_Ea.X,Vel_Ea.Y,Vel_Ea.Z),
               An_Instance => CNT.This_Subsystem.The_Velocity);
         end if;


         -- -------------------------------------------------------------------------------------------
         --|    Process Altitude Slewing
         -- -------------------------------------------------------------------------------------------
         --| Disable altitude slew for record-playback and snapshot restore
         if Save_Restore.Replay_Trimming or Save_Restore.Replay_In_Progress or
            Misc_Freeze_Request then
            CNT.This_Subsystem.Old_Cmnd_Alt := IOS.This_Ios_Interface.Command_Altitude;
         end if;

         --| check if slew turned on or in process
         if Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location) <  -35000.0 and
           IOS.This_Ios_Interface.Slew_Up then
            IOS.This_Ios_Interface.Slew_Up := False;
         end if;

         if (abs(IOS.This_Ios_Interface.Command_Altitude - CNT.This_Subsystem.Old_Cmnd_Alt) > 0.3)  or
           IOS.This_Ios_Interface.Slew_Up or
           IOS.This_Ios_Interface.Slew_Down or
           abs(CNT.This_Subsystem.Delta_Alt) > 0.1 then

            CNT.This_Subsystem.Alt_Slew := True;
         else
            CNT.This_Subsystem.Alt_Slew := False;
         end if;

         --| no slew into ground
         if Hat > JSAT.The_Normal_Height_On_Ground - 1.0 then ---5.0 then
            IOS.This_Ios_Interface.Slew_Down := False;
            if IOS.This_Ios_Interface.Slew_Up or
              (IOS.This_Ios_Interface.Command_Altitude  > CNT.This_Subsystem.Old_Cmnd_Alt) then

               CAS := Airspeed.Get_Calibrated_Airspeed(CNT.This_Subsystem.The_Airspeed);
               if Cas < 50.0 then  -- set airspeed if slew into air
                    CNT.This_Subsystem.Set_Cas := True;  -- flag processed above
                    IOS.This_Ios_Interface.Trim_Request := True;
               end if;
            end if;
         end if;

         if CNT.This_Subsystem.Alt_Slew then
            if IOS.This_Ios_Interface.Command_Altitude > 35000.0 then
               IOS.This_Ios_Interface.Command_Altitude := 35000.0;
            end if;

            --| Do not allow command below ground
            if IOS.This_Ios_Interface.Command_Altitude < -(CNT.This_Subsystem.Terrain_Height +
                                                           JSAT.The_Normal_Height_On_Ground) then -- -4.0) then
               IOS.This_Ios_Interface.Command_Altitude := -(CNT.This_Subsystem.Terrain_Height+
                                                           JSAT.The_Normal_Height_On_Ground); ---4.0);
            end if;

            --| get IOS slew rate
            if IOS.This_Ios_Interface.Use_Variable_Slew_Rate then
               V_Slew := Float(IOS.This_Ios_Interface.Variable_Slew_Rate) * 0.0694444444;
            else
               if  IOS.This_Ios_Interface.Slew_Rate = 3 then
                  V_Slew := 25.0; -- 90000 ft/min
               elsif  IOS.This_Ios_Interface.Slew_Rate = 2 then
                  V_Slew := 5.0; -- 18000 ft/min
               else
                  V_Slew := 1.0; -- 3600 ft/min
               end if;
            end if;

            --| not in freeze ... do not allow airplane to outclimb or outdive slew rate
            if not IOS.This_Ios_Interface.The_Flight_Freeze and V_Slew < 5.0 then
               V_Slew := 5.0;
            end if;

            --| slow down near ground to prevent crash
            if Hat > 2.0*JSAT.The_Normal_Height_On_Ground then -- -6.0 then
               V_Slew := 0.5;
            end if;

            CNT.This_Subsystem.Delta_Alt := IOS.This_Ios_Interface.Command_Altitude
              + Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location);

            if IOS.This_Ios_Interface.Slew_Up then
               Location.Set_Slew_Geometric_Altitude
                 (-V_Slew,
                  An_Instance => CNT.This_Subsystem.The_Location);

               --| Reset commanded altitude while slew up in progress
               IOS.This_Ios_Interface.Command_Altitude :=
                 -Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location);
               CNT.This_Subsystem.Old_Cmnd_Alt := IOS.This_Ios_Interface.Command_Altitude;
               CNT.This_Subsystem.Delta_Alt := 0.0;

            elsif IOS.This_Ios_Interface.Slew_Down then
               Location.Set_Slew_Geometric_Altitude
                 (V_Slew,
                  An_Instance => CNT.This_Subsystem.The_Location);

               --| Reset commanded altitude while slew down in progress
               IOS.This_Ios_Interface.Command_Altitude :=
                 -Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location);
               CNT.This_Subsystem.Old_Cmnd_Alt := IOS.This_Ios_Interface.Command_Altitude;
               CNT.This_Subsystem.Delta_Alt := 0.0;

            else
               --| must be ios altitude command
               if (abs(IOS.This_Ios_Interface.Command_Altitude - CNT.This_Subsystem.Old_cmnd_Alt) > 0.3) then
                  IOS.This_Ios_Interface.Command_Altitude :=
                    IOS.This_Ios_Interface.Command_Altitude + 0.4;
               end if;
               CNT.This_Subsystem.Old_Cmnd_Alt := IOS.This_Ios_Interface.Command_Altitude;

               if CNT.This_Subsystem.Delta_Alt >= 0.0 then
                  Sign := -1.0;
               else
                  Sign := 1.0;
               end if;

               if IOS.This_Ios_Interface.The_Flight_Freeze then
                  --| one step in freeze
                  V_Slew := - CNT.This_Subsystem.Delta_Alt;
                  CNT.This_Subsystem.Delta_Alt := 0.0; -- turn off slew command
               elsif abs(CNT.This_Subsystem.Delta_Alt) <= V_Slew then
                  --| close to command altitude adjust last step
                  V_Slew := CNT.This_Subsystem.Delta_Alt * Sign;
                  CNT.This_Subsystem.Delta_Alt := 0.0; -- turn off slew command
               else
                  V_Slew := V_Slew * Sign;
               end if;

               Location.Set_Slew_Geometric_Altitude
                 (V_Slew,
                  An_Instance => CNT.This_Subsystem.The_Location);
            end if;
         elsif not IOS.This_Ios_Interface.The_Altitude_Freeze then
            Location.Set_Geometric_Altitude
              (Vel_Ea.Z        ,
               Old_Vel_Ea.Z    ,
               Int_Time  ,
               An_Instance => CNT.This_Subsystem.The_Location);
         end if;
         Location.Set_Height_Above_Local_Terrain
           (CNT.This_Subsystem.Terrain_Height,
            An_Instance => CNT.This_Subsystem.The_Location);

	 Hat := Location.Get_Height_Above_Local_Terrain
                (CNT.This_Subsystem.The_Location);

         --|  reset altitude if less than 1 foot above ground
         if (Hat > -1.0 and not Misc_Freeze_Request and
             not Jpats_Reposition.Lat_Long_Reposition_Active) then
            Location.Assign_Geometric_Altitude
              (-1.0 + CNT.This_Subsystem.Terrain_Height ,
               An_Instance => CNT.This_Subsystem.The_Location);
         end if;
         -- **********************************************************************


         -- -------------------------------------------------------------------------------------------
         --|    Process Position Slewing
         -- -------------------------------------------------------------------------------------------

         Pos_Slew := IOS.This_Ios_Interface.Slew_North or IOS.This_Ios_Interface.Slew_South or
           IOS.This_Ios_Interface.Slew_East or IOS.This_Ios_Interface.Slew_West or
           IOS.This_Ios_Interface.Slew_Fwd or IOS.This_Ios_Interface.Slew_Aft or
           IOS.This_Ios_Interface.Slew_Left or IOS.This_Ios_Interface.Slew_Right;

         if Pos_Slew then
            if IOS.This_Ios_Interface.Use_Variable_Slew_Rate then
               Slew := Float(IOS.This_Ios_Interface.Variable_Slew_Rate) * 0.00000006705; -- degrees @ 60 HZ = 1 to 360 mph
            else
               if  IOS.This_Ios_Interface.Slew_Rate = 3 then
                  Slew := 0.0016667; -- 0.1 deg/sec = 360 mph
               elsif  IOS.This_Ios_Interface.Slew_Rate = 2 then
                  Slew := 0.00016667; -- 0.01 deg/sec = 36 mph
               else
                  Slew := 0.00005;
               end if;
            end if;
            if IOS.This_Ios_Interface.Slew_North then
               Location.Set_Slew_North
                 (Slew,
                  An_Instance => CNT.This_Subsystem.The_Location);

            elsif IOS.This_Ios_Interface.Slew_South then
               Location.Set_Slew_North
                 (-Slew,
                  An_Instance => CNT.This_Subsystem.The_Location);

            elsif IOS.This_Ios_Interface.Slew_East then
               Location.Set_Slew_East
                 (Slew,
                  An_Instance => CNT.This_Subsystem.The_Location);

            elsif IOS.This_Ios_Interface.Slew_West then
               Location.Set_Slew_East
                 (-Slew,
                  An_Instance => CNT.This_Subsystem.The_Location);

            else
               if IOS.This_Ios_Interface.Slew_Fwd then
                  Slew_N := Slew * Cos(Hdg_Angle);
                  Slew_E := Slew * Sin(Hdg_Angle);
               elsif IOS.This_Ios_Interface.Slew_Aft then
                  Slew_N := -Slew * Cos(Hdg_Angle);
                  Slew_E := -Slew * Sin(Hdg_Angle);
               elsif IOS.This_Ios_Interface.Slew_Right then
                  Slew_N := - Slew * Sin(Hdg_Angle);
                  Slew_E := Slew * Cos(Hdg_Angle);
               elsif IOS.This_Ios_Interface.Slew_Left then
                  Slew_N := Slew * Sin(Hdg_Angle);
                  Slew_E := -Slew * Cos(Hdg_Angle);
               end if;
               declare
                  Temp : Float;
               begin

                  Temp := Float(Location.Get_North(CNT.This_Subsystem.The_Location));
                  if Temp > 0.0 and Temp < 0.0001 then
                     Temp := 0.0001;
                  elsif Temp < 0.0 and Temp > -0.0001 then
                     Temp := -0.0001;
                  end if;

                  Slew_E := Slew_E/ Cos(Temp,360.0);
               end;

               Location.Set_Slew_East
                 (Slew_E,
                  An_Instance => CNT.This_Subsystem.The_Location);
               Location.Set_Slew_North
                 (Slew_N,
                  An_Instance => CNT.This_Subsystem.The_Location);
            end if;

         else  -- normal position update
            if IOS.This_Ios_Interface.The_Position_Freeze  then
               null;
            else
               Location.Set_North
                 (Vel_Ea.X,
                  Old_Vel_Ea.X,
                  Int_Time,
                  An_Instance => CNT.This_Subsystem.The_Location);
               Location.Set_East
                 (Vel_Ea.Y,
                  Old_Vel_Ea.Y,
                  Int_Time,
                  An_Instance => CNT.This_Subsystem.The_Location);
            end if;
         end if;  -- end update or slew position

         if IOS.This_Ios_Interface.The_Altitude_Freeze or
           (Phase > 0 and Phase /= 4) then
            -- ----------------------------------------------------------------------------------------
            --|    if altitude freeze is on adjust rate of climb to account for aircraft
            --|    acceleration/deceleration which would occur at constant airspeed with
            --|    altitude changing (also autotest freeze)
            -- ----------------------------------------------------------------------------------------
            Airspeed.Set_Rate_Of_Climb_For_Altitude_Freeze
              (An_Instance=> CNT.This_Subsystem.The_Airspeed);

         else   -- normal R/C calculation
            Airspeed.Set_Rate_Of_Climb
              (Vel_Ea.Z            ,
               An_Instance => CNT.This_Subsystem.The_Airspeed);
         end if;

         L_1 := Quaternion.Get_L_1(CNT.This_Subsystem.The_Quaternion);
         L_2 := Quaternion.Get_L_2(CNT.This_Subsystem.The_Quaternion);

         Airspeed.Set_Ground_Speed
           (Vel_Ea.X,
            Vel_Ea.Y,
            L_1     ,
            L_2     ,
            An_Instance => CNT.This_Subsystem.The_Airspeed);
         CNT.This_Subsystem.Ground_Speed := Sqrt( Vel_Ea.X * Vel_Ea.X + Vel_Ea.Y * Vel_Ea.Y);

         --| velocity relative to the earth complete
         --| add in wind velocity to compute velocity relative to the air
         --| mach, ias, cas, aoa, and sideslip
         North_Vel := Vel_Ea.X + North_Wind;
         East_Vel  := Vel_Ea.Y + East_Wind;
         Vert_Vel  := Vel_Ea.Z + Vert_Wind;

         --|   transform these inertial (north east down) velocities into body axis (x y z)
         Velocity.Set_Body_Axis
           (Roll_Angle   ,
            Pitch_Angle  ,
            Hdg_Angle    ,
            North_Vel    ,
            East_Vel     ,
            Vert_Vel     ,
            An_Instance => CNT.This_Subsystem.The_Velocity);

         V_B   := Velocity.Get_Body_Axis(CNT.This_Subsystem.The_Velocity);
         Accel_Body_Axis := Acceleration.Get_Body_Axis(CNT.This_Subsystem.The_Acceleration);


         -- -------------------------------------------------------------------------------------------
         --|    Update Aero Angles
         -- -------------------------------------------------------------------------------------------
         Aerodynamic_Angle.Set_Angle_Of_Attack
           (V_B,
            An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);

         Aerodynamic_Angle.Set_Side_Slip_Angle
           (V_B,
            An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);

         Aerodynamic_Angle.Set_Angle_Of_Attack_Rate
           (V_B,
            Accel_Body_Axis,
            Roll_Rate,
            Pitch_Rate,
            Yaw_Rate,
            Pitch_Angle,
            Roll_Angle,
            An_Instance => CNT.This_Subsystem.The_Aerodynamic_Angle);

         Airspeed.Set_True_Airspeed (V_B, An_Instance => CNT.This_Subsystem.The_Airspeed);

         --|  retrieve true airspeed just computed
         TAS := Airspeed.Get_True_Airspeed(CNT.This_Subsystem.The_Airspeed);
         Airspeed.Calc_Dynamic_Pressure
           (Air_Dens ,
            TAS,
            An_Instance => CNT.This_Subsystem.The_Airspeed);

         Airspeed.Set_Mach_Number
           (Temperature,       --| Degree Rankine
            An_Instance => CNT.This_Subsystem.The_Airspeed);

         Airspeed.Set_Calibrated_Airspeed
           (Air_Dens,
            An_Instance => CNT.This_Subsystem.The_Airspeed);

         AOA  := Aerodynamic_Angle.Get_Angle_Of_Attack(CNT.This_Subsystem.The_Aerodynamic_Angle);

         Velocity.Set_Nondimensional_Angular --| (Computing ND angular velocity
           (SpWing        ,
            Chord_Wing    ,
            TAS           ,
            Aoa           ,
            An_Instance => CNT.This_Subsystem.The_Velocity);

         CAS := Airspeed.Get_Calibrated_Airspeed(CNT.This_Subsystem.The_Airspeed);
         Stall_Speed := Airspeed.Get_Stall_Speed (CNT.This_Subsystem.The_Airspeed);

         if Jab.Get_Wow then
            CNT.This_Subsystem.Buffet_Intens := 0.0;
         else
            CNT.This_Subsystem.Buffet_Intens  := It.Doubly_Indexed.Interpolate
              ((Aoa * 57.3),Flap_Position,Sabds_Table'Access);
         end if;



         --|  compute body axis velocity relative to earth without winds
         --|  these are only used in aircraft body to compute ground reactions
         Velocity.Set_Inertial
           (Roll_Angle   ,
            Pitch_Angle  ,
            Hdg_Angle    ,
            An_Instance => CNT.This_Subsystem.The_Velocity);

         --| insert default runway condtion when visual runway changes, then allow
         --| ios to change it
         if CNT.This_Subsystem.Runway /= CNT.This_Subsystem.Old_Runway then
            if (JPATS_Visual_Airport.Dry_Runway)  then
               IOS.This_Ios_Interface.Runway_Condition := 23;
               CNT.This_Subsystem.Runway := 1;
            elsif (JPATS_Visual_Airport.Snowy_Runway) then
               IOS.This_Ios_Interface.Runway_Condition := 5;
               CNT.This_Subsystem.Runway := 2;
            elsif (JPATS_Visual_Airport.Wet_Runway) then
               IOS.This_Ios_Interface.Runway_Condition := 15;
               CNT.This_Subsystem.Runway := 3;
            end if;
         end if;
         CNT.This_Subsystem.Old_Runway := CNT.This_Subsystem.Runway;


         -- -------------------------------------------------------------------------------------------
         --| System Reset
         -- -------------------------------------------------------------------------------------------
         if IOS.This_Ios_Interface.System_Reset  or
           IO.This_Io_Interface.System_Reset_On or
           CNT.This_Subsystem.System_Reset_Timer > 0.0 then

            CNT.This_Subsystem.System_Reset_Timer := CNT.This_Subsystem.System_Reset_Timer + Dt;
            if CNT.This_Subsystem.System_Reset_Timer >= 1.0 then
               CNT.This_Subsystem.System_Reset_Timer := 0.0;
               IOS.This_Ios_Interface.System_Reset := False;
            end if;
            IO.This_Io_Interface.System_Reset_Amber := True;
            IO.This_Io_Interface.System_Reset_Green := False;
            JPATS_Electrical.Reset_Battery;
            JPATS_Hydraulics.Reset_Emerg_Hyd_Px;
            JPATS_Environmental.Reset_Ejection_Seat;
            JPATS_Environmental.Reset_Canopy;
            CNT.This_Subsystem.System_Reset_Flag := True;
         else
            CNT.This_Subsystem.System_Reset_Timer := 0.0;
            CNT.This_Subsystem.System_Reset_Flag := False;
            IO.This_Io_Interface.System_Reset_Amber := False;
            IO.This_Io_Interface.System_Reset_Green := True;
         end if;

      else

         -- ------------------------------------------------------------------------------------------
         --| During initialization of record playback, clear crash status and blink the trimming notice
         -- ------------------------------------------------------------------------------------------
         IOS.This_Ios_Interface.Crashed := False;
         IOS.This_IOs_Interface.The_Terrain_Impact_Crash := False;
         IOS.This_IOs_Interface.The_Excess_Airspeed_Crash := False;
         IOS.This_IOs_Interface.The_Vertical_Load_Crash := False;
         IOS.This_IOs_Interface.The_Sink_Rate_Crash := False;

         if CNT.This_Subsystem.Slewing_Timer > 0.5 and Dt > 0.0 then
            IOS.This_Ios_Interface.Slewing := 0;
         else
            IOS.This_Ios_Interface.Slewing := 3;
         end if;
         CNT.This_Subsystem.Slewing_Timer := CNT.This_Subsystem.Slewing_Timer + Dt;
         if CNT.This_Subsystem.Slewing_Timer > 1.0 then
            CNT.This_Subsystem.Slewing_Timer := 0.0;
         end if;
      end if; -- end case of initializing replay

      -- ----------------------------------------------------------------------------------------------
      --| Latency Test drivers
      -- ----------------------------------------------------------------------------------------------
      if JPATS_Auto_test.Test_Number > 900 and JPATS_Auto_test.Test_Number < 904 then
         if Jab.Latency_Trigger then
            --| if trigger set, then pitch to 60 degrees;
            --|      Pitch_Angle := -0.50;  -- Nose up
            Pitch_Angle := -0.50;
            Hdg_Angle := 1.0;
            Attitude_Angle.Assign_Pitch
              (Pitch_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

            Roll_Angle := 1.0;
            Attitude_Angle.Assign_Pitch
              (Roll_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

            Hdg_Angle := 1.0;
            Attitude_Angle.Assign_Hdg
              (Hdg_Angle,
               An_Instance => CNT.This_Subsystem.The_Attitude_Angle);

            Acceleration.Assign_Load_Factor
              (An_Instance => CNT.This_Subsystem.The_Acceleration,
               Load_Factor => (0.0,0.0,-8.0));

            --| increase (reduce) terrain height to allow A/C clearance to pitch up
            CNT.This_Subsystem.Terrain_Height := CNT.This_Subsystem.Terrain_Height + 20.0;
         end if;

         Quaternion.Set_Quat_From_Angles
           (Roll_Angle   ,
            Pitch_Angle  ,
            Hdg_Angle,
            An_Instance  => CNT.This_Subsystem.The_Quaternion);
      end if;



      -- ----------------------------------------------------------------------------------------------
      --| variables for IOS displays
      -- ----------------------------------------------------------------------------------------------
      IOS.This_Ios_Interface.Latitude            := Location.Get_North(CNT.This_Subsystem.The_Location);
      IOS.This_Ios_Interface.Longitude           := Location.Get_East(CNT.This_Subsystem.The_Location);
      IOS.This_Ios_Interface.Geometric_Altitude  := -Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location);
      IOS.This_Ios_Interface.Mag_Hdg             := Hdg_Angle * 180.0/Pi - Mag_Var;
      IOS.This_Ios_Interface.Height_Above_Ground := - Location.Get_Height_Above_Local_Terrain(CNT.This_Subsystem.The_Location);
      IOS.This_Ios_Interface.Cas                 := Cas;
      IOS.This_Ios_Interface.Elev                := - JPATS_DCLS.Get_Elev_Pos;
      IOS.This_Ios_Interface.Rud                 := JPATS_DCLS.Get_Rudder_Position;
      IOS.This_Ios_Interface.L_Ail               := JPATS_DCLS.Get_Left_Aileron_Position;
      IOS.This_Ios_Interface.R_Ail               := JPATS_DCLS.Get_Right_Aileron_Position;
      IOS.This_Ios_Interface.Nwa                 := JPATS_Aircraft_Body.Get_Nwa;
      IOS.This_Ios_Interface.L_Gear              := JPATS_Landing_Gear.Left_Landing_Gear_Position;
      IOS.This_Ios_Interface.R_Gear              := JPATS_Landing_Gear.Right_Landing_Gear_Position;
      IOS.This_Ios_Interface.N_Gear              := JPATS_Landing_Gear.Nose_Landing_Gear_Position;
      IOS.This_Ios_Interface.Aoa                 := Aerodynamic_Angle.Get_Angle_Of_Attack(CNT.This_Subsystem.The_Aerodynamic_Angle) * 57.3;
      IOS.This_Ios_Interface.Beta                := Aerodynamic_Angle.Get_Side_Slip_Angle(CNT.This_Subsystem.The_Aerodynamic_Angle) * 57.3;
      IOS.This_Ios_Interface.Flap                := JPATS_Secondary_Flight_Controls.Mean_Flap_Position;
      IOS.This_Ios_Interface.SB                  := JPATS_Secondary_Flight_Controls.Speedbrake_Position;
      IOS.This_Ios_Interface.E_Tab               := JPATS_Secondary_Flight_Controls.Elevator_Trim_Position;
      IOS.This_Ios_Interface.A_Tab               := JPATS_Secondary_Flight_Controls.Aileron_Trim_Position;
      IOS.This_Ios_Interface.R_Tab               := JPATS_Secondary_Flight_Controls.Rudder_Trim_Position;
      IOS.This_Ios_Interface.Ax                  := Accel_Body_Axis.X;
      IOS.This_Ios_Interface.Ay                  := Accel_Body_Axis.Y;
      IOS.This_Ios_Interface.Az                  := Accel_Body_Axis.Z;
      IOS.This_Ios_Interface.Vx                  := V_B.X;
      IOS.This_Ios_Interface.Vy                  := V_B.Y;
      IOS.This_Ios_Interface.Vz                  := V_B.Z;
      IOS.This_Ios_Interface.P_Dot               := Roll_Accel*57.3;
      IOS.This_Ios_Interface.P                   := Roll_Rate*57.3;
      IOS.This_Ios_Interface.Q_Dot               := Pitch_Accel*57.3;
      IOS.This_Ios_Interface.Q                   := Pitch_Rate*57.3;
      IOS.This_Ios_Interface.R_Dot               := Yaw_Accel*57.3;
      IOS.This_Ios_Interface.R                   := Yaw_Rate*57.3;
      IOS.This_Ios_Interface.Q_Dyn               := JPATS_Simulated_Aircraft.Get_Dynamic_Pressure;

      Prop_Force                                 := JPATS_Propeller.Force_Tensor;
      Prop_Moment                                := JPATS_Propeller.Moment_Tensor;
      IOS.This_Ios_Interface.Fx_Eng              := Prop_Force(Vector_Types.X);
      IOS.This_Ios_Interface.Fy_Eng              := Prop_Force(Vector_Types.Y);
      IOS.This_Ios_Interface.Fz_Eng              := Prop_Force(Vector_Types.Z);
      IOS.This_Ios_Interface.Mx_Eng              := Prop_Moment(Vector_Types.X);
      IOS.This_Ios_Interface.My_Eng              := Prop_Moment(Vector_Types.Y);
      IOS.This_Ios_Interface.Mz_Eng              := Prop_Moment(Vector_Types.Z);
      IOS.This_Ios_Interface.Fx_Tot              := Force.X;
      IOS.This_Ios_Interface.Fy_Tot              := Force.Y;
      IOS.This_Ios_Interface.Fz_Tot              := Force.Z;
      IOS.This_Ios_Interface.Mx_Tot              := Moment.X;
      IOS.This_Ios_Interface.My_Tot              := Moment.Y;
      IOS.This_Ios_Interface.Mz_Tot              := Moment.Z;

      IOS.This_Ios_Interface.Fx_Aero             := JPATS_Aircraft_Body.Aero_Force.X;
      IOS.This_Ios_Interface.Fy_Aero             := JPATS_Aircraft_Body.Aero_Force.Y;
      IOS.This_Ios_Interface.Fz_Aero             := JPATS_Aircraft_Body.Aero_Force.Z;
      IOS.This_Ios_Interface.Mx_Aero             := JPATS_Aircraft_Body.Aero_Moment.X;
      IOS.This_Ios_Interface.My_Aero             := JPATS_Aircraft_Body.Aero_Moment.Y;
      IOS.This_Ios_Interface.Mz_Aero             := JPATS_Aircraft_Body.Aero_Moment.Z;

      IOS.This_Ios_Interface.Fx_Gnd              := JPATS_Aircraft_Body.Gear_Force.X;
      IOS.This_Ios_Interface.Fy_Gnd              := JPATS_Aircraft_Body.Gear_Force.Y;
      IOS.This_Ios_Interface.Fz_Gnd              := JPATS_Aircraft_Body.Gear_Force.Z;
      IOS.This_Ios_Interface.Mx_Gnd              := JPATS_Aircraft_Body.Gear_Moment.X;
      IOS.This_Ios_Interface.My_Gnd              := JPATS_Aircraft_Body.Gear_Moment.Y;
      IOS.This_Ios_Interface.Mz_Gnd              := JPATS_Aircraft_Body.Gear_Moment.Z;

      IOS.This_Ios_Interface.L_Brake_Pos         := JPATS_DCLS.Left_Brake_Pos;
      IOS.This_Ios_Interface.L_Brake_F           := JPATS_DCLS.Left_Brake_Force;
      IOS.This_Ios_Interface.R_Brake_Pos         := JPATS_DCLS.Right_Brake_Pos;
      IOS.This_Ios_Interface.R_Brake_F           := JPATS_DCLS.Right_Brake_Force;
      IOS.This_Ios_Interface.Long_Stk_Pos        := JPATS_DCLS.Long_Stick_Pos;
      IOS.This_Ios_Interface.Long_Stk_F          := JPATS_DCLS.Long_Stick_Force;
      IOS.This_Ios_Interface.Lat_Stk_Pos         := JPATS_DCLS.Lat_Stick_Pos;
      IOS.This_Ios_Interface.Lat_Stk_F           := JPATS_DCLS.Lat_Stick_Force;
      IOS.This_Ios_Interface.Pedal_Pos           := JPATS_DCLS.Get_Pedal_Position;
      IOS.This_Ios_Interface.Pedal_F             := JPATS_DCLS.Get_Pedal_Force;
      IOS.This_Ios_Interface.Avg_Ail             := JPATS_DCLS.Avg_Ail;
      IOS.This_Ios_Interface.Ias                 := JPATS_Flight_Instruments.Primary_Ias;
      if IOS.This_Ios_Interface.Ias < 40.0 then
         IOS.This_Ios_Interface.Ias              := 40.0;   -- instrument minimum 40
      end if;
      IOS.This_Ios_Interface.Mach                := Airspeed.Get_Mach_Number(An_Instance => CNT.This_Subsystem.The_Airspeed);
      IOS.This_Ios_Interface.Oat                 := JPATS_Atmosphere.Temperature - 459.7;
      IOS.This_Ios_Interface.Density             := JPATS_Atmosphere.Density * 10_000.0;
      IOS.This_Ios_Interface.Pla                 := JPATS_Powerplant.Power_Control_Lever_Angle;
      IOS.This_Ios_Interface.Ice_Weight          := JPATS_Aircraft_Body.Get_Wing_Body_Plus_Weight_Ice;




      -- ----------------------------------------------------------------------------------------------
      --| Flight debug variables for IOS displays
      -- ----------------------------------------------------------------------------------------------

      if IOS.This_Ios_Interface.Float1 > 0.5 and IOS.This_Ios_Interface.Float1 < 1.5 then

         IOS.This_Ios_Interface.Float2  := Acceleration.Get_Angular(CNT.This_Subsystem.The_Acceleration).Roll;
         IOS.This_Ios_Interface.Float3  := Acceleration.Get_Angular(CNT.This_Subsystem.The_Acceleration).Pitch;
         IOS.This_Ios_Interface.Float4  := Acceleration.Get_Angular(CNT.This_Subsystem.The_Acceleration).Yaw;
         IOS.This_Ios_Interface.Float5  := Velocity.Get_Roll_Rate(CNT.This_Subsystem.The_Velocity);
         IOS.This_Ios_Interface.Float6  := Velocity.Get_Pitch_Rate(CNT.This_Subsystem.The_Velocity);
         IOS.This_Ios_Interface.Float7  := Velocity.Get_Yaw_Rate(CNT.This_Subsystem.The_Velocity);
         IOS.This_Ios_Interface.Float8  := Attitude_Angle.Get_Roll(An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
         IOS.This_Ios_Interface.Float9  := Attitude_Angle.Get_Pitch(An_Instance => CNT.This_Subsystem.The_Attitude_Angle);
         IOS.This_Ios_Interface.Float10 := Attitude_Angle.Get_Hdg(CNT.This_Subsystem.The_Attitude_Angle);


      elsif IOS.This_Ios_Interface.Float1 > 1.5 and IOS.This_Ios_Interface.Float1 < 2.5 then

         IOS.This_Ios_Interface.Float2  := Acceleration.Get_Earth_Axis(CNT.This_Subsystem.The_Acceleration).x;
         IOS.This_Ios_Interface.Float3  := Acceleration.Get_Earth_Axis(CNT.This_Subsystem.The_Acceleration).y;
         IOS.This_Ios_Interface.Float4  := Acceleration.Get_Earth_Axis(CNT.This_Subsystem.The_Acceleration).z;
         IOS.This_Ios_Interface.Float5  := Velocity.Get_Earth_Axis(CNT.This_Subsystem.The_Velocity).x;
         IOS.This_Ios_Interface.Float6  := Velocity.Get_Earth_Axis(CNT.This_Subsystem.The_Velocity).y;
         IOS.This_Ios_Interface.Float7  := Velocity.Get_Earth_Axis(CNT.This_Subsystem.The_Velocity).z;
         IOS.This_Ios_Interface.Float8  := Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location);
         IOS.This_Ios_Interface.Float9  := Nz;
         IOS.This_Ios_Interface.Float10 := JPATS_DCLS.Debug10;


      elsif IOS.This_Ios_Interface.Float1 > 2.5 and IOS.This_Ios_Interface.Float1 < 3.5 then

         IOS.This_Ios_Interface.Float2  := Accel_Body_Axis.X;
         IOS.This_Ios_Interface.Float3  := Accel_Body_Axis.y;
         IOS.This_Ios_Interface.Float4  := Accel_Body_Axis.z;
         IOS.This_Ios_Interface.Float5  := Velocity.Get_Body_Axis(CNT.This_Subsystem.The_Velocity).x;
         IOS.This_Ios_Interface.Float6  := Velocity.Get_Body_Axis(CNT.This_Subsystem.The_Velocity).y;
         IOS.This_Ios_Interface.Float7  := Velocity.Get_Body_Axis(CNT.This_Subsystem.The_Velocity).z;
         IOS.This_Ios_Interface.Float8  := Airspeed.Get_Calibrated_Airspeed(CNT.This_Subsystem.The_Airspeed);
         IOS.This_Ios_Interface.Float9  := Airspeed.Get_Mach_Number(An_Instance => CNT.This_Subsystem.The_Airspeed);
         IOS.This_Ios_Interface.Float10 := Aerodynamic_Angle.Get_Angle_Of_Attack(CNT.This_Subsystem.The_Aerodynamic_Angle) * 57.3;


      elsif IOS.This_Ios_Interface.Float1 > 3.5 and IOS.This_Ios_Interface.Float1 < 6.5 then

         IOS.This_Ios_Interface.Float2  := JPATS_DCLS.Debug2;
         IOS.This_Ios_Interface.Float3  := JPATS_DCLS.Debug3;
         IOS.This_Ios_Interface.Float4  := JPATS_DCLS.Debug4;
         IOS.This_Ios_Interface.Float5  := JPATS_DCLS.Debug5;
         IOS.This_Ios_Interface.Float6  := JPATS_DCLS.Debug6;
         IOS.This_Ios_Interface.Float7  := JPATS_DCLS.Debug7;
         IOS.This_Ios_Interface.Float8  := JPATS_DCLS.Debug8;
         IOS.This_Ios_Interface.Float9  := JPATS_DCLS.Debug9;
         IOS.This_Ios_Interface.Float10 := JPATS_DCLS.Debug10;

         if IOS.This_Ios_Interface.Bool1 then
            JPATS_DCLS.Set_Filter;
         end if;

         if IOS.This_Ios_Interface.Bool2 then
            JPATS_Simulated_Aircraft.Set_System_Freeze;
         end if;


      elsif IOS.This_Ios_Interface.Float1 > 6.5 and IOS.This_Ios_Interface.Float1 < 9.5 then

         IOS.This_Ios_Interface.Float2  := JPATS_aircraft_body.Debug2;
         IOS.This_Ios_Interface.Float3  := JPATS_aircraft_body.Debug3;
         IOS.This_Ios_Interface.Float4  := JPATS_aircraft_body.Debug4;
         IOS.This_Ios_Interface.Float5  := JPATS_aircraft_body.Debug5;
         IOS.This_Ios_Interface.Float6  := JPATS_aircraft_body.Debug6;
         IOS.This_Ios_Interface.Float7  := JPATS_aircraft_body.Debug7;
         IOS.This_Ios_Interface.Float8  := JPATS_aircraft_body.Debug8;
         IOS.This_Ios_Interface.Float9  := JPATS_aircraft_body.Debug9;
         IOS.This_Ios_Interface.Float10 := JPATS_aircraft_body.Debug0;


      elsif IOS.This_Ios_Interface.Float1 > 11.5 and IOS.This_Ios_Interface.Float1 < 12.5 then

         IOS.This_Ios_Interface.Float2  := Float(Boolean'Pos(abs(JPATS_Aircraft_Body.Get_Aircraft_Runway_Rumble) > 0.0));
         IOS.This_Ios_Interface.Float3  := Float(Boolean'Pos(abs(JPATS_Secondary_Flight_Controls.Speedbrake_Buffet)>0.0));
         IOS.This_Ios_Interface.Float4  := Float(Boolean'Pos(abs(JPATS_Aircraft_Body.Nose_Wheel_Scuffing) > 0.0));
         IOS.This_Ios_Interface.Float5  := Float(Boolean'Pos(abs(JPATS_Powerplant.Engine_Vibration) > 0.0));
         IOS.This_Ios_Interface.Float6  := Float(Boolean'Pos(abs(JPATS_Secondary_Flight_Controls.Flap_Buffet) > 0.0));
         IOS.This_Ios_Interface.Float7  := Float(Boolean'Pos(abs(JPATS_Simulated_Aircraft.Get_Stall_Buffet) > 0.0));
         IOS.This_Ios_Interface.Float8  := Float(Boolean'Pos(abs(JPATS_Atmosphere.Get_Total_Turbulence_Intensity) > 0.0));
         IOS.This_Ios_Interface.Float9  := Float(Boolean'Pos(JPATS_Atmosphere.Midair));
         IOS.This_Ios_Interface.Float10 := Float(Boolean'Pos(abs(JPATS_Aircraft_Body.Get_Aircraft_Landing_Gear_Buffet) > 0.0));

      elsif IOS.This_Ios_Interface.Float1 > 12.5 and IOS.This_Ios_Interface.Float1 < 13.5 then
         if not IOS.This_Ios_Interface.Bool1 then

         IOS.This_Ios_Interface.Float2  := 0.0;
         IOS.This_Ios_Interface.Float3  := 1.0;
         IOS.This_Ios_Interface.Float4  := 0.0;
         IOS.This_Ios_Interface.Float5  := 0.1;
         IOS.This_Ios_Interface.Float6  := 0.0;
         end if;
         IOS.This_Ios_Interface.Float7  := JPATS_DCLS.Debug7;
         IOS.This_Ios_Interface.Float8  := JPATS_DCLS.Debug8;
         IOS.This_Ios_Interface.Float9  := JPATS_DCLS.Debug9;


      elsif IOS.This_Ios_Interface.Float1 > 13.5 and IOS.This_Ios_Interface.Float1 < 14.5 then

         IOS.This_Ios_Interface.Float2  := Velocity.Get_Earth_Axis(CNT.This_Subsystem.The_Velocity).Z;
         IOS.This_Ios_Interface.Float3  := Float(Boolean'Pos(Jab.Get_WOW));
         IOS.This_Ios_Interface.Float4  := Flap_Position;
         IOS.This_Ios_Interface.Float5  := Nz;
         IOS.This_Ios_Interface.Float6  := JPATS_Flight_Instruments.Primary_Ias;
         IOS.This_Ios_Interface.Float7  := (JSAT.Maximum_Operating_Airspeed + 15.0);

      elsif IOS.This_Ios_Interface.Float1 > 14.5 and IOS.This_Ios_Interface.Float1 < 15.5 then

         IOS.This_Ios_Interface.Float2  := Hat;
         IOS.This_Ios_Interface.Float3  := Nz;
         IOS.This_Ios_Interface.Float4  := Float(Boolean'Pos(JPATS_Visual_Buffer.Aircraft_On_Runway(1)));
         IOS.This_Ios_Interface.Float5  := Float(Boolean'Pos(Jab.Get_Tail_Wow));
         IOS.This_Ios_Interface.Float6  := Float(Boolean'Pos(Jab.Get_Wow));
         IOS.This_Ios_Interface.Float7  := Float(Boolean'Pos(Jab.Crash_Wingtip_Impact));
         IOS.This_Ios_Interface.Float8  := Float(Boolean'Pos(JPATS_Visual_Buffer.Fuselage_Collision));
         IOS.This_Ios_Interface.Float9  := Float(Boolean'Pos(JPATS_Visual_Buffer.Right_Wing_Collision));
         IOS.This_Ios_Interface.Float10 := Float(Boolean'Pos(JPATS_Visual_Buffer.Left_Wing_Collision));

      elsif IOS.This_Ios_Interface.Float1 > 19.5 and IOS.This_Ios_Interface.Float1 < 20.5 then

         IOS.This_Ios_Interface.Float2  := Hat;
         IOS.This_Ios_Interface.Float3  := JPATS_Reposition.Reposition_Command.Position.Altitude;
         IOS.This_Ios_Interface.Float4  := - Location.Get_Geometric_Altitude(CNT.This_Subsystem.The_Location);
         IOS.This_Ios_Interface.Float5  :=  JPATS_Visual_Buffer.Height_Of_Terrain(1);
         IOS.This_Ios_Interface.Float6  := - Location.Get_Height_Above_Local_Terrain(CNT.This_Subsystem.The_Location);
         IOS.This_Ios_Interface.Float7  := Reference_Coordinates.Get_Z_CG(cnt.This_Subsystem.The_Reference_Coordinates);

      elsif Integer(IOS.This_Ios_Interface.Float1)=114 then
         IOS.This_Ios_Interface.Float7  := JPATS_DCLS.Debug7;
         IOS.This_Ios_Interface.Float8  := JPATS_DCLS.Debug8;
         IOS.This_Ios_Interface.Float9  := JPATS_DCLS.Debug9;
         IOS.This_Ios_Interface.Float10 := JPATS_DCLS.debug10;

      elsif Integer(IOS.This_Ios_Interface.Float1)=151 then
         IOS.This_Ios_Interface.Float8  := JPATS_DCLS.Debug8;
         IOS.This_Ios_Interface.Float9  := JPATS_DCLS.Debug9;
         IOS.This_Ios_Interface.Float10 := JPATS_DCLS.debug10;
      elsif Integer(IOS.This_Ios_Interface.Float1)=160 then
         if Integer(IOS.This_Ios_Interface.Float2)=1 then
            CNT.This_Subsystem.Slewing_Timer := 0.0;
            IOS.This_Ios_Interface.Slewing := 4;
         end if;
         IOS.This_Ios_Interface.Float3  := CNT.This_Subsystem.Slewing_Timer;
         IOS.This_Ios_Interface.Float4  := Float(IOS.This_Ios_Interface.Slewing);
         IOS.This_Ios_Interface.Float5  := Float(Boolean'Pos(JIP.Is_Established));
         IOS.This_Ios_Interface.Float6  := Float(CNT.This_Subsystem.The_Fuselage_Collision_Counter);
         IOS.This_Ios_Interface.Float7  := Float(CNT.This_Subsystem.The_Left_Wing_Collision_Counter);
         IOS.This_Ios_Interface.Float8  := Float(CNT.This_Subsystem.The_Right_Wing_Collision_Counter);

      end if;

   end Update;

   procedure Un_Freeze is
   begin
      IOS.This_Ios_Interface.The_Flight_Freeze := False;
   end Un_Freeze;

end JPATS_Simulated_Aircraft.Controller;
